#!/usr/bin/perl -s
##
## Razor::Client - Vipul's Razor Client API
##
## Copyright (c) 2001, Vipul Ved Prakash.  All rights reserved.
## This code is free software; you can redistribute it and/or modify
## it under the same terms as Perl itself.
##
## $Id: Client.pm,v 1.19 2001/12/26 02:29:21 vipul Exp $

package Razor::Client;


BEGIN { 
    local ($^W) = 0; 
    use strict;
    use Razor::Errorhandler;
    use Time::HiRes qw(gettimeofday);
    use Net::Ping;
    use Razor::Config;
    use Net::DNS::Resolver;
    use Data::Dumper;
    use Razor::String qw(hash2str hash);
    use Razor::Version; 
    use IO::Socket;
    use IO::Select;
    use vars qw( $VERSION $PROTOCOL );
    use base qw(Razor::Errorhandler);
}


($VERSION) = do { my @r = (q$Revision: 1.20 $ =~ /\d+/g); sprintf "%d."."%02d" x $#r, @r };
$PROTOCOL = $Razor::Version::PROTOCOL; 


sub new { 
    my ($class, $conf, %params) = @_;
    my $self = new Razor::Config ("client", $conf, %params);
    bless $self, $class;
    $self->debug ("Razor Agents $VERSION, protocol version $PROTOCOL.");
    $self->readserverlist(); 
    $self->discover();
    $self->{server} = @{$$self{serverlist}}[0];
    unless ($self->{server}) { 
        my $zone = $self->{razorzone};
        my $retval = $self->error("Razor Error 2: No Catalogue Servers in $zone zone.", "Construction Error") ;
        return $retval;
    }
    $self->debug("Closest server is $self->{server}");
    return $self;
} 


sub discover {

    my ($self) = @_; 
    return if $self->{turn_off_discovery} && (!($self->{force_discovery}));
    my $randomize = int(rand($self->{rediscovery_wait}/7));
    my $timeleft = -e $self->{listfile} ? 
        ((time) - ((stat ($self->{listfile}))[9] + $self->{rediscovery_wait} - $randomize)) : 0;
    my $size = -e $self->{listfile} ? (stat ($self->{listfile}))[7] : 0;
    if ($size && ($timeleft < 0) && !($self->{force_discovery})) { 
        $timeleft *= -1; 
        $self->debug ("$timeleft seconds before closest server discovery");
        return
    } 
    my $zone = $$self{razorzone};
    $self->debug ("Discovering closest server in the $zone zone");
    my $resolver = new Net::DNS::Resolver; 
    my @list;

    BUILDLIST: 
    for ( "a".."z", "1..9999" ) { 
        my $query = $resolver->query ($_.'.'.$zone);
        my $rr; next BUILDLIST unless $query;
        foreach $rr ($query->answer) { 
            my $pushed = 0;
            if ($rr->type eq "A") { 
                push @list, $rr->address; 
                $pushed = 1;
            } elsif ($rr->type eq "CNAME") { 
                if ($rr->cname eq 'list.terminator') { 
                    pop @list if $pushed;
                    last BUILDLIST;
                } elsif ($rr->cname eq "skip") { 
                    pop @list if $pushed;
                    next BUILDLIST;
                }
            }
        }
    }

    my $pinger = Net::Ping->new ("tcp", 4);
    my %times; my $timings;
   
    for (@list) {
        my $t1 = gettimeofday(); 
        $pinger->ping($_);
        my $t2 = gettimeofday();
        $times{$_} = $t2 - $t1;
    }

    @list = sort { $times{$a} <=> $times{$b} } keys %times;
    $self->debug ("Sorted (closest first) list of available servers & RTTs: ");
    for (@list) { $timings .= sprintf ("%.15s (%1.4f) ", $_, $times{$_}) }
    $self->debug ($timings);
    $self->{serverlist} = [@list];
    $self->writeserverlist() or return; 
    $self->{done_discovery} = 1;
    return $self;

}


sub check { 
    my ($self, %options) = @_;
    my $response = $self->_send (\%options, 'lookup') or return;
    my ($ident, @reply) = split /\n/, $response;
    my ($version, $protocol) = $self->_extract_protocol ($ident);
    $self->debug ("Server version: $version, protocol version $protocol");
    unless ($version) { 
        $self->debug ("Couldn't parse server greeting\n"); 
        $self->nextserver or return; 
        return $self->check(\%options);
    }
    if (($protocol - $PROTOCOL) >= 1) { 
        $self->debug ("A newer, unsupported protocol $protocol is in use.  Please upgrade Razor Client Tools\n");
        $self->nextserver or return;
        return $self->check(\%options);
    }
    $self->debug ("Server response: @reply");
    my (@results, $counter);
    for (@reply) { 
        my $reply = $_;
        $counter++;
        if ( $_ =~ /^Positive / ) {
            $self->debug ("- Message $counter is KNOWN SPAM -");
            push @results, 1;
        } elsif ( $_ =~ /^Negative / ) {
            $self->debug ("Message $counter NOT found in the catalogue.");
            push @results, 0;
        } elsif ( $_ =~ /Vipul's Razor/ ) { 
            next;
        } else { 
            $self->debug ("Unable to parse server response.");
            $self->nextserver or return;
            return $self->check(\%options);
        }
    }
    return \@results;
} 


sub report { 
    my ($self, %options) = @_;
    my $response = $self->_send (\%options, 'report') or return;
    my ($ident, @reply) = split /\n/, $response;
    my ($version, $protocol) = $self->_extract_protocol ($ident);
    $self->debug ("Server version: $version, protocol version $protocol");
    if (($protocol - $PROTOCOL) >= 1) { 
        $self->debug ("A newer, unsupported protocol $protocol is in use.  Please upgrade Razor Client Tools\n");
        $self->nextserver or return;
        return $self->report(\%options);
    }
    $self->debug ("Server response: @reply");
    my (@results, $counter);
    for (@reply) { 
        my $reply = $_;
        $counter++;
        if ($reply =~ /Accepted/) { 
            $self->debug ("$$self{server} accepted the report.");
            push @results, 1;
        } elsif ($reply =~ /Rejected/) { 
            $self->debug ("Server rejected the report.");
            push @results, 0;
        } elsif ($reply =~ /Vipul's Razor/) { 
            next;
        } elsif ($reply =~ /^\s+$/) { 
            next;
        } else {
            $self->debug ("Server did not understand report command."); 
            $self->nextserver or return;
            return $self->report (\%options);
        }
    }
    return \@results;;
}


sub writeserverlist { 
    my $self = shift;
    $self->{listfile} =~ m:([\d\w\-\.\/]+)$:;
    my $listfile = $1; 
    return $self->error("Razor Error 12: $self->{listfile} is not a valid filename") unless $listfile;
    unless (open LIST, ">$listfile") { 
        return $self->error("Razor Error 11: Couldn't open $listfile for writing.");
    }
    for (@{$$self{serverlist}}) { 
        print LIST "$_\n";
    }
    close LIST;
    $self->debug ("Wrote server list to $listfile");
    return $self;
}


sub readserverlist { 
    my $self = shift;
    open (LIST, "$$self{listfile}") || return undef;
    my @list;
    for (<LIST>) { 
        chomp;
        push @list, $1 if /^(\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3})$/;
    } 
    $self->{serverlist} = [@list] if @list;
    $self->debug ("Read server list from $$self{listfile}");
    return $self;
}


sub debug { 
    my ($self, $message) = @_;
    my $term = "\n" unless $message =~ m/\n\s*$/si;
    print "debug: $message\n" if $self->{debug};
}


sub _send { 
    my ($self, $options, $action) = @_;
    my (@sigs);
    if ($options->{spam}) { 
        push @sigs, hash ($options->{spam}); 
    } elsif ($options->{sigs}) { 
        push @sigs, @{$options->{sigs}};
    }
    my $response;
    while (scalar @sigs) { 
        my @psigs = splice @sigs, 0, $self->{rlimit};
        $self->connect() or return;
        my $sock = $self->{sock};
        for (@psigs) {
            $self->debug ("Signature: $_");
            my %message = ( key => $_, action => $action );
            my $str = hash2str ({%message});
            print $sock "$str\n";
        } 
        print $sock ".\n"; 
        { 
            local $/;
            undef $/; 
            $response .= <$sock> 
        }  
        close $sock; 
    }
    return $response; 
}


sub _extract_protocol { 
    my ($self, $ident) = @_;
    my ($version) = $ident =~ m/^Vipul's Razor Version (\S+)\./;
    unless ($version) { 
        ($version) = $ident =~ m/^Vipul's Razor (\S+)\,/ 
    }
    my ($protocol) = $ident =~ m/protocol version ([\d\.]+)\.$/;
    $protocol = 1 unless $protocol;
    return ($version, $protocol);
}


sub nextserver { 
    my ($self) = @_; 
    shift @{$self->{serverlist}};
    my $server = @{$self->{serverlist}}[0]; 
    if (!($server) || $server !~ /\./) { 
        if ($$self{done_discovery}) { 
            return $self->error ("Razor Error 3: No Catalogue Servers available at this time.", "Construction Error");
        }
        $self->{force_discovery} = 1;
        $self->discover;
        return $self;
    }
    $self->{server} = $server;
    $self->debug ("Next closest server is -$server-");
    $self->writeserverlist() or return;
    return $self;
}


sub connect {

    my ($self, %params) = @_;
    my $sock;

    if ($self->{simulate}) { 
        return $self->error ("Razor Error 4: This is a simulation. Won't connect to $$self{server}.");
    }
    my $server = $params{server} || $$self{server};
    $self->debug ("Connecting to $server..."); 
    unless ( $sock = IO::Socket::INET->new( 
                            PeerAddr => $server,
                            PeerPort => 2702,
                            Proto    => 'tcp',
                            Timeout  => 20,
                     )
    ) { 
        $self->debug ("Unable to connect to $$self{server}; Reason: $!.");
        $self->nextserver or return;
        return $self->connect;
    }    

    my $select = new IO::Select ($sock);
    if ($select->can_read (15)) {
        $select->remove($sock);
        $sock->autoflush;
        $self->{sock} = $sock;
        $self->debug ("Connection established");
        return $self;
    } else {
        $self->debug ("Timed out while reading from $$self{server}.");
        $select->remove($sock);
        $sock->close();
        $self->nextserver or return;
        return $self->connect;
    }

}    


sub DESTROY { 
    my $self = shift;
    $self->debug ("Agent terminated");
}


sub zonename { 
    my ($zone, $type) = @_;
    my ($sub, $dom) = split /\./, $zone, 2; 
    return "$sub-$type.$dom";
}


1;


=head1 NAME

Razor::Client -- Client API for Vipul's Razor.

=head1 DESCRIPTION 

check() & report() methods will be eventually moved to Razor::Agents. For
more information, read the sources for Razor::Client, Razor::Agents,
razor-check and razor-report.

=head1 ERROR HANDLING

In case of a fatal error, Razor::Client will return undef and set
$self->errstr() to one of the following error messages.

=head2 2 

Razor Error 2: No Catalogue Servers in the $zone zone. 

This error will occur when Razor::Client doesn't find any listed servers
in the specified zone. The most likely reason for this error is DNS
lookup failure.

=head2 3 

Razor Error 3: No Catalogue Servers available at this time.

None of the listed Catalogue servers are reachable. The most likely reason
for this is a local network outage or a firewall blocking outgoing
connections to port 2702.

=head2 4

Razor Error 4: This is a simulation.  Won't connect to $server.

When razor-agents are called with C<-s> option, they exit with value of 4.

=head2 11

Razor Error 11: Couldn't open $serverlist for writing. 

Razor::Client couldn't write out the list of available servers,
~/.razor.lst by default.

=head1 AUTHOR

Vipul Ved Prakash, E<lt>mail@vipul.netE<gt>

=cut


