#!/usr/bin/perl -s
##
## Razor::String -- String functions required by Razor.
##
## Copyright (c) 1999, Vipul Ved Prakash.  All rights reserved.
## This code is free software; you can redistribute it and/or modify
## it under the same terms as Perl itself.
##
## $Id: String.pm,v 1.6 2001/12/25 09:44:33 vipul Exp $

package Razor::String; 
use strict;
use Mail::Internet;
use Digest::SHA1;
use Exporter; 
use vars qw ( @ISA $VERSION @EXPORT_OK ); 
@ISA = "Exporter"; 
( $VERSION )  = '$Revision: 1.6 $' =~ /\s(\d+\.\d+)\s/; 

@EXPORT_OK = qw( hash2str str2hash hash headers ); 
my @ESCAPE = ( '&', ',', ':' );


sub hash2str { 

    my $href = shift; 
    my %hash = %$href; 
    my ($str, $key);

    for $key ( keys %hash ) { 
        my $tstr;
        if ( ref $hash{$key} eq 'ARRAY' ) { 
            for ( @{ $hash{ $key }} )  { $tstr .= escape( $_ ) . "," } $str =~ s/,$//;
        } elsif ( !(ref $hash{$key}) ) { 
            $tstr .= escape ( $hash{$key} );
        }
        if ( $tstr ) { $str .= "$key:$tstr&" }
    }

    $str =~ s/&$//; return $str;

}


sub str2hash { 

    my $str = shift; 
    my %hash;
    my @pairs = split /(?<!\\)&/, $str; 

    for ( @pairs ) { 
        my ( $key, $data ) = split /(?<!\\):/, $_, 2;
        if ( $data =~ /(?<!\\),/ ) { 
            my @list = split /(?<!\\),/, $data; 
            for ( @list ) { $_ = unescape ( $_ ) };
            $hash{$key} = [@list];
        } else { $hash{$key} = unescape ( $data ) } 
    }

    return \%hash; 

}


sub hash { 

    my ( $spam ) = @_; 
    my $mail = new Mail::Internet ( $spam ); 
    $mail->tidy_body(); 
    $mail->remove_sig (); $mail->tidy_body (); 
    my $body = $mail->body();

    my $ctx = Digest::SHA1->new;
    $ctx->add( @$body );
    return $ctx->hexdigest;

}


sub headers { 

    my ( $spam ) = @_;
    my @headers = ();
    for (@$spam) { 
        if ($_ !~ /^\s*\n$/) { 
            push @headers, $_ 
        } else { last }
    }
    return \@headers;

}


sub escape { 
    my $str = shift;
    for ( @ESCAPE ) { 
        my $e = $_; $str =~ s/$e/\\$e/g;
    }
    return $str;
}


sub unescape { 
    my $str = shift; 
    for ( @ESCAPE ) {
        my $e = $_; $str =~ s/\\$e/$e/g; 
    }
    return $str; 
}


1;

=head1 NAME

Razor::String - String manipulation functions for Vipul's Razor 

=head1 VERSION

    $Revision: 1.6 $

=head1 DESCRIPTION

The module provides functions to manipulate various types of strings,
including spam texts and protocol transactions. 

=head1 FUNCTIONS

=over 4

=item hash2str() 

Constructs a service request from a perl hash. Razor service requests are
modelled on HTTP GET requests. These requests are sent to razord who
reconstructs the perl hash with the str2hash() function, that is also
provided by this module.

=item hash()

Takes a message body and returns a signature. The signature is a SHA1
digest of the body text. All headers and the .sig line (if found) 
are stripped before computing the digest.

=back

=head1 AUTHOR

Vipul Ved Prakash, E<lt>mail@vipul.netE<gt>

=cut


