/*

settings.c
Deals with configuration file (CONF_FILE)

Copyright (C) 2001, Jem E. Berkes

*/

#include "defs.h"
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


int mem_error()
/* Error when buffer space can not be allocated */
{
fprintf(stderr, "Fatal error: unable to allocate required memory!\n");
fprintf(stderr, "This most likely indicates that the input stream (e-mail)\n");
fprintf(stderr, "has exceedingly long lines\n");
return -1;
}


int tokenize_list(char* thelist, int verbose)
/*
Goes through the list, (optionally displaying each entry) and returns
count of actual entries in the list. Destroys the list.
*/
{
char *token;
int count=0;

token = strtok(thelist, CONF_TOKS);
while (token != NULL)
	{
	count++;
	if (verbose == 1)
		printf("[%s] ", token);
	token = strtok(NULL, CONF_TOKS);
	}

return count;
}


/*
Reads good list and bad list from configuration file
Returns 0 if OK or -1 if failed
*/
int read_lists(	char** good_list, char** bad_list,
				char** gl_scratch, char** bl_scratch)
{
int line_len, line_num=0, good_len=DEF_BUFSIZE, bad_len=DEF_BUFSIZE;
FILE* cf;
char buffer[DEF_BUFSIZE];
char *temp_bigger;

*good_list = (char*)malloc(DEF_BUFSIZE);
if (*good_list == NULL)
	return mem_error();

*bad_list = (char*)malloc(DEF_BUFSIZE);
if (*bad_list == NULL)
	{
	free(*good_list);
	return mem_error();
	}

if ( (access(CONF_FILE, R_OK) == -1) ||
		((cf=fopen(CONF_FILE, "r")) == NULL) )
	{
	fprintf(stderr, "Error: unable to open configuration file %s\n", CONF_FILE);
	free(*good_list);
	free(*bad_list);
	return -1;
	}

**good_list = 0;
**bad_list = 0;

/*
All lines in cf must begin with "#" (comment), "good:", or "bad:"
*/

while (fgets(buffer, DEF_BUFSIZE, cf) != NULL)
	{
	++line_num;
	line_len = strlen(buffer);
	if (line_len == DEF_BUFSIZE-1)
		{
		fclose(cf);
		free(*good_list);
		free(*bad_list);
		fprintf(stderr, "Error: oversized line in %s line %d\n", CONF_FILE, line_num);
		return -1;
		}

	if (*buffer=='\n' || *buffer=='#')
		continue;
	else if (strncmp(buffer, "good:", 5) == 0)
		{
		while ( (strlen(*good_list)+line_len) >= (unsigned int)good_len)
			{
			if (good_len >= BUF_LIMIT)
				temp_bigger = NULL;
			else
				{
				good_len *= 2;
				temp_bigger = (char*)malloc(good_len);
				}
			if (temp_bigger == NULL)
				{
				free(*good_list);
				free(*bad_list);
				fclose(cf);
				return mem_error();
				}
			strcpy(temp_bigger, *good_list);
			free(*good_list);
			*good_list = temp_bigger;
			}
		strcat(*good_list, buffer+5);
		}
	
	else if (strncmp(buffer, "bad:", 4) == 0)
		{
		while ( (strlen(*bad_list)+line_len) >= (unsigned int)bad_len)
			{
			if (bad_len >= BUF_LIMIT)
				temp_bigger = NULL;
			else
				{
				bad_len *= 2;
				temp_bigger = (char*)malloc(bad_len);
				}
			if (temp_bigger == NULL)
				{
				free(*good_list);
				free(*bad_list);
				fclose(cf);
				return mem_error();
				}
			strcpy(temp_bigger, *bad_list);
			free(*bad_list);
			*bad_list = temp_bigger;
			}
		strcat(*bad_list, buffer+4);
		}

	else
		{
		fclose(cf);
		free(*good_list);
		free(*bad_list);
		fprintf(stderr, "Error: syntax error in %s line %d\n", CONF_FILE, line_num);
		return -1;
		}
	}

fclose(cf);

*gl_scratch = (char*)malloc(good_len);
if (*gl_scratch == NULL)
	{
	free(*good_list);
	free(*bad_list);
	return mem_error();
	}

*bl_scratch = (char*)malloc(bad_len);
if (bl_scratch == NULL)
	{
	free(*good_list);
	free(*bad_list);
	free(*gl_scratch);
	return mem_error();
	}

strcpy(*gl_scratch, *good_list);
strcpy(*bl_scratch, *bad_list);

if (	(tokenize_list(*gl_scratch, 0)==0) ||
		(tokenize_list(*bl_scratch, 0)==0) )
	{
	free(*good_list);
	free(*bad_list);
	free(*gl_scratch);
	free(*bl_scratch);
	fprintf(stderr, "Error: either goodlist or badlist is empty\n");
	return -1;
	}
return 0;
}
