/*
 *  Rexx/Tk Combobox Extension
 *  Copyright (C) 2000  Mark Hessling  <M.Hessling@qut.edu.au>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "rexxtk.h"

char *RxPackageName = "rexxtkcombobox";
char *ExtensionSource =
   "package require Tk 8.0\n"
   "package provide combobox 2.0\n"
   "namespace eval ::combobox {\n"
   "namespace export combobox\n"
   "variable widgetOptions\n"
   "variable widgetCommands\n"
   "variable scanCommands\n"
   "variable listCommands\n"
   "}\n"
   "proc ::combobox::combobox {w args} {\n"
   "variable widgetOptions\n"
   "variable widgetCommands\n"
   "variable scanCommands\n"
   "variable listCommands\n"
   "if {![info exists widgetOptions]} {\n"
   "Init\n"
   "}\n"
   "eval Build $w $args\n"
   "SetBindings $w\n"
   "return $w\n"
   "}\n"
   "proc ::combobox::Init {} {\n"
   "variable widgetOptions\n"
   "variable widgetCommands\n"
   "variable scanCommands\n"
   "variable listCommands\n"
   "variable defaultEntryCursor\n"
   "array set widgetOptions [list  "
   "-background          {background          Background}  "
   "-bd                  -borderwidth  "
   "-bg                  -background  "
   "-borderwidth         {borderWidth         BorderWidth}  "
   "-command             {command             Command}  "
   "-commandstate        {commandState        State}  "
   "-cursor              {cursor              Cursor}  "
   "-editable            {editable            Editable}  "
   "-fg                  -foreground  "
   "-font                {font                Font}  "
   "-foreground          {foreground          Foreground}  "
   "-height              {height              Height}  "
   "-highlightbackground {highlightBackground HighlightBackground}  "
   "-highlightcolor      {highlightColor      HighlightColor}  "
   "-highlightthickness  {highlightThickness  HighlightThickness}  "
   "-image               {image               Image}  "
   "-maxheight           {maxHeight           Height}  "
   "-relief              {relief              Relief}  "
   "-selectbackground    {selectBackground    Foreground}  "
   "-selectborderwidth   {selectBorderWidth   BorderWidth}  "
   "-selectforeground    {selectForeground    Background}  "
   "-state               {state               State}  "
   "-takefocus           {takeFocus           TakeFocus}  "
   "-textvariable        {textVariable        Variable}  "
   "-value               {value               Value}  "
   "-width               {width               Width}  "
   "-xscrollcommand      {xScrollCommand      ScrollCommand}  "
   "]\n"
   "set widgetCommands [list  "
   "bbox      cget     configure    curselection  "
   "delete    get      icursor      index         "
   "insert    list     scan         selection     "
   "xview     select   toggle       open          "
   "close  "
   "]\n"
   "set listCommands [list  "
   "delete       get       "
   "index        insert       size  "
   "]\n"
   "set scanCommands [list mark dragto]\n"
   "if {[lsearch -exact [package names] \"Tk\"] != -1} {\n"
   "if {$::tcl_platform(platform) == \"windows\"} {\n"
   "image create bitmap ::combobox::bimage -data {\n"
   "#define down_arrow_width 12\n"
   "#define down_arrow_height 12\n"
   "static char down_arrow_bits[] = {\n"
   "0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,\n"
   "0xfc,0xf1,0xf8,0xf0,0x70,0xf0,0x20,0xf0,\n"
   "0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00;\n"
   "}\n"
   "}\n"
   "} else {\n"
   "image create bitmap ::combobox::bimage -data  {\n"
   "#define down_arrow_width 15\n"
   "#define down_arrow_height 15\n"
   "static char down_arrow_bits[] = {\n"
   "0x00,0x80,0x00,0x80,0x00,0x80,0x00,0x80,\n"
   "0x00,0x80,0xf8,0x8f,0xf0,0x87,0xe0,0x83,\n"
   "0xc0,0x81,0x80,0x80,0x00,0x80,0x00,0x80,\n"
   "0x00,0x80,0x00,0x80,0x00,0x80\n"
   "}\n"
   "}\n"
   "}\n"
   "set tmpWidget \".tmp\"\n"
   "set count 0\n"
   "while {[winfo exists $tmpWidget] == 1} {\n"
   "set tmpWidget \".tmp$count\"\n"
   "incr count\n"
   "}\n"
   "scrollbar $tmpWidget\n"
   "set sb_width [winfo reqwidth $tmpWidget]\n"
   "destroy $tmpWidget\n"
   "entry $tmpWidget\n"
   "foreach foo [$tmpWidget configure] {\n"
   "if {[lindex $foo 0] == \"-cursor\"} {\n"
   "set defaultEntryCursor [lindex $foo 4]\n"
   "}\n"
   "if {[llength $foo] == 5} {\n"
   "set option [lindex $foo 1]\n"
   "set value [lindex $foo 4]\n"
   "option add *Combobox.$option $value widgetDefault\n"
   "if {[string compare $option \"foreground\"] == 0  "
   "|| [string compare $option \"background\"] == 0  "
   "|| [string compare $option \"font\"] == 0} {\n"
   "option add *Combobox*ComboboxListbox.$option $value  "
   "widgetDefault\n"
   "}\n"
   "}\n"
   "}\n"
   "destroy $tmpWidget\n"
   "option add *Combobox.cursor              {}\n"
   "option add *Combobox.commandState        normal widgetDefault\n"
   "option add *Combobox.editable            1      widgetDefault\n"
   "option add *Combobox.maxHeight           10     widgetDefault\n"
   "option add *Combobox.height              0\n"
   "}\n"
   "SetClassBindings\n"
   "}\n"
   "proc ::combobox::SetClassBindings {} {\n"
   "bind Combobox <Destroy> [list ::combobox::DestroyHandler %W]\n"
   "set this {[::combobox::convert %W -W]}\n"
   "bind Combobox <Any-ButtonPress>   \"$this close\"\n"
   "bind Combobox <Any-ButtonRelease> \"$this close\"\n"
   "bind Combobox <FocusIn> {tkTabToWindow [::combobox::convert %W -W].entry}\n"
   "bind Combobox <Unmap> {[::combobox::convert %W -W] close}\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::SetBindings {w} {\n"
   "upvar ::combobox::${w}::widgets  widgets\n"
   "upvar ::combobox::${w}::options  options\n"
   "bindtags $widgets(entry)  "
   "[concat $widgets(this) [bindtags $widgets(entry)]]\n"
   "bindtags $widgets(button)  "
   "[concat $widgets(this) [bindtags $widgets(button)]]\n"
   "bind $widgets(entry) <Tab>  "
   "\"tkTabToWindow \\[tk_focusNext $widgets(entry)\\]; break\"\n"
   "bind $widgets(entry) <Shift-Tab>  "
   "\"tkTabToWindow \\[tk_focusPrev $widgets(this)\\]; break\"\n"
   "bind $widgets(button) <ButtonPress-1> [list $widgets(this) toggle]\n"
   "bind $widgets(entry) <B1-Enter> \"break\"\n"
   "bind $widgets(listbox) <ButtonRelease-1>  "
   "\"::combobox::Select $widgets(this) \\[$widgets(listbox) nearest %y\\]; break\"\n"
   "bind $widgets(vsb) <ButtonPress-1>   {continue}\n"
   "bind $widgets(vsb) <ButtonRelease-1> {continue}\n"
   "bind $widgets(listbox) <Any-Motion> {\n"
   "%W selection clear 0 end\n"
   "%W activate @%x,%y\n"
   "%W selection anchor @%x,%y\n"
   "%W selection set @%x,%y @%x,%y\n"
   "}\n"
   "foreach event [list <Up> <Down> <Tab> <Return> <Escape>  "
   "<Next> <Prior> <Double-1> <1> <Any-KeyPress>  "
   "<FocusIn> <FocusOut>] {\n"
   "bind $widgets(entry) $event  "
   "\"::combobox::HandleEvent $widgets(this) $event\"\n"
   "}\n"
   "}\n"
   "proc ::combobox::Build {w args } {\n"
   "variable widgetOptions\n"
   "if {[winfo exists $w]} {\n"
   "error \"window name \\\"$w\\\" already exists\"\n"
   "}\n"
   "namespace eval ::combobox::$w {\n"
   "variable ignoreTrace 0\n"
   "variable oldFocus    {}\n"
   "variable oldGrab     {}\n"
   "variable oldValue    {}\n"
   "variable options\n"
   "variable this\n"
   "variable widgets\n"
   "set widgets(foo) foo  ;# coerce into an array\n"
   "set options(foo) foo  ;# coerce into an array\n"
   "unset widgets(foo)\n"
   "unset options(foo)\n"
   "}\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "set widgets(this)   [frame  $w -class Combobox -takefocus 0]\n"
   "set widgets(entry)  [entry  $w.entry -takefocus 1]\n"
   "set widgets(button) [label  $w.button -takefocus 0]\n"
   "foreach name [array names widgetOptions] {\n"
   "if {[llength $widgetOptions($name)] == 1} continue\n"
   "set optName  [lindex $widgetOptions($name) 0]\n"
   "set optClass [lindex $widgetOptions($name) 1]\n"
   "set value [option get $w $optName $optClass]\n"
   "set options($name) $value\n"
   "}\n"
   "if {[info exists options(-value)]  "
   "&& [string length $options(-value)] == 0} {\n"
   "unset options(-value)\n"
   "}\n"
   "set widgets(frame) ::combobox::${w}::$w\n"
   "pack $widgets(entry)  -side left  -fill both -expand yes\n"
   "pack $widgets(button) -side right -fill y    -expand no\n"
   "array set options $args\n"
   "set widgets(popup)   [toplevel  $w.top]\n"
   "set widgets(listbox) [listbox   $w.top.list]\n"
   "set widgets(vsb)     [scrollbar $w.top.vsb]\n"
   "pack $widgets(listbox) -side left -fill both -expand y\n"
   "$widgets(vsb) configure  "
   "-command \"$widgets(listbox) yview\"  "
   "-highlightthickness 0\n"
   "$widgets(button) configure  "
   "-highlightthickness 0  "
   "-borderwidth 1  "
   "-relief raised  "
   "-width [expr {[winfo reqwidth $widgets(vsb)] - 2}]\n"
   "$widgets(entry) configure  "
   "-borderwidth 0  "
   "-relief flat  "
   "-highlightthickness 0\n"
   "$widgets(popup) configure  "
   "-borderwidth 1  "
   "-relief sunken\n"
   "$widgets(listbox) configure  "
   "-selectmode browse  "
   "-background [$widgets(entry) cget -bg]  "
   "-yscrollcommand \"$widgets(vsb) set\"  "
   "-exportselection false  "
   "-borderwidth 0\n"
   "wm overrideredirect $widgets(popup) 1\n"
   "wm transient        $widgets(popup) [winfo toplevel $w]\n"
   "wm group            $widgets(popup) [winfo parent $w]\n"
   "wm resizable        $widgets(popup) 0 0\n"
   "wm withdraw         $widgets(popup)\n"
   "rename ::$w $widgets(frame)\n"
   "proc ::$w {command args}  "
   "\"eval ::combobox::WidgetProc $w \\$command \\$args\"\n"
   "if {[catch \"::combobox::Configure $widgets(this) [array get options]\" error]} {\n"
   "catch {destroy $w}\n"
   "error $error\n"
   "}\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::HandleEvent {w event} {\n"
   "upvar ::combobox::${w}::widgets  widgets\n"
   "upvar ::combobox::${w}::options  options\n"
   "upvar ::combobox::${w}::oldValue oldValue\n"
   "switch $event {\n"
   "\"<Any-KeyPress>\" {\n"
   "if {$options(-editable)} {\n"
   "$widgets(listbox) see 0\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor 0\n"
   "$widgets(listbox) activate 0\n"
   "}\n"
   "}\n"
   "\"<FocusIn>\" {\n"
   "set oldValue [$widgets(entry) get]\n"
   "}\n"
   "\"<FocusOut>\" {\n"
   "if {![winfo ismapped $widgets(popup)]} {\n"
   "set newValue [$widgets(entry) get]\n"
   "if {$oldValue != $newValue} {\n"
   "CallCommand $widgets(this) $newValue\n"
   "}\n"
   "}\n"
   "}\n"
   "\"<1>\" {\n"
   "set editable [::combobox::GetBoolean $options(-editable)]\n"
   "if {!$editable} {\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "$widgets(this) close\n"
   "return -code break;\n"
   "} else {\n"
   "if {$options(-state) != \"disabled\"} {\n"
   "$widgets(this) open\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "}\n"
   "}\n"
   "\"<Double-1>\" {\n"
   "if {$options(-state) != \"disabled\"} {\n"
   "$widgets(this) toggle\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "\"<Tab>\" {\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "::combobox::Find $widgets(this) 0\n"
   "return -code break;\n"
   "} else {\n"
   "::combobox::SetValue $widgets(this) [$widgets(this) get]\n"
   "}\n"
   "}\n"
   "\"<Escape>\" {\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "$widgets(this) close\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "\"<Return>\" {\n"
   "set newValue [$widgets(entry) get]\n"
   "if {$oldValue != $newValue} {\n"
   "CallCommand $widgets(this) $newValue\n"
   "}\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "::combobox::Select $widgets(this)  "
   "[$widgets(listbox) curselection]\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "\"<Next>\" {\n"
   "$widgets(listbox) yview scroll 1 pages\n"
   "set index [$widgets(listbox) index @0,0]\n"
   "$widgets(listbox) see $index\n"
   "$widgets(listbox) activate $index\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor $index\n"
   "$widgets(listbox) selection set $index\n"
   "}\n"
   "\"<Prior>\" {\n"
   "$widgets(listbox) yview scroll -1 pages\n"
   "set index [$widgets(listbox) index @0,0]\n"
   "$widgets(listbox) activate $index\n"
   "$widgets(listbox) see $index\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor $index\n"
   "$widgets(listbox) selection set $index\n"
   "}\n"
   "\"<Down>\" {\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "tkListboxUpDown $widgets(listbox) 1\n"
   "return -code break;\n"
   "} else {\n"
   "if {$options(-state) != \"disabled\"} {\n"
   "$widgets(this) open\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "}\n"
   "\"<Up>\" {\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "tkListboxUpDown $widgets(listbox) -1\n"
   "return -code break;\n"
   "} else {\n"
   "if {$options(-state) != \"disabled\"} {\n"
   "$widgets(this) open\n"
   "return -code break;\n"
   "}\n"
   "}\n"
   "}\n"
   "}\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::DestroyHandler {w} {\n"
   "if {[string compare [winfo class $w] \"Combobox\"] == 0} {\n"
   "upvar ::combobox::${w}::widgets  widgets\n"
   "upvar ::combobox::${w}::options  options\n"
   "namespace delete ::combobox::$w\n"
   "rename $w {}\n"
   "}\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::Find {w {exact 0}} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "set pattern [$widgets(entry) get]\n"
   "if {[string length $pattern] == 0} {\n"
   "$widgets(listbox) see 0\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor 0\n"
   "$widgets(listbox) activate 0\n"
   "return\n"
   "}\n"
   "set list [$widgets(listbox) get 0 end]\n"
   "set exactMatch -1\n"
   "if {$exact} {\n"
   "set exactMatch [lsearch -exact $list $pattern]\n"
   "}\n"
   "set index -1\n"
   "while {$index == -1 && [string length $pattern]} {\n"
   "set index [lsearch -glob $list \"$pattern*\"]\n"
   "if {$index == -1} {\n"
   "regsub {.$} $pattern {} pattern\n"
   "}\n"
   "}\n"
   "set thisItem [lindex $list $index]\n"
   "if {$index != -1} {\n"
   "set nextIndex [expr {$index + 1}]\n"
   "set nextItem [lindex $list $nextIndex]\n"
   "if {[string match $pattern* $nextItem]} {\n"
   "set marker [string length $pattern]\n"
   "while {$marker <= [string length $pattern]} {\n"
   "set a [string index $thisItem $marker]\n"
   "set b [string index $nextItem $marker]\n"
   "if {[string compare $a $b] == 0} {\n"
   "append pattern $a\n"
   "incr marker\n"
   "} else {\n"
   "break\n"
   "}\n"
   "}\n"
   "} else {\n"
   "set marker [string length $pattern]\n"
   "}\n"
   "} else {\n"
   "set marker end\n"
   "set index 0\n"
   "}\n"
   "if {$exact && $exactMatch == -1} {\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) see $index\n"
   "} elseif {!$exact}  {\n"
   "set oldstate [$widgets(entry) cget -state]\n"
   "$widgets(entry) configure -state normal\n"
   "$widgets(entry) delete 0 end\n"
   "$widgets(entry) insert end $thisItem\n"
   "$widgets(entry) selection clear\n"
   "$widgets(entry) selection range $marker end\n"
   "$widgets(listbox) activate $index\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor $index\n"
   "$widgets(listbox) selection set $index\n"
   "$widgets(listbox) see $index\n"
   "$widgets(entry) configure -state $oldstate\n"
   "}\n"
   "}\n"
   "proc ::combobox::Select {w index} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "catch {\n"
   "set data [$widgets(listbox) get [lindex $index 0]]\n"
   "::combobox::SetValue $widgets(this) $data\n"
   "$widgets(listbox) selection clear 0 end\n"
   "$widgets(listbox) selection anchor $index\n"
   "$widgets(listbox) selection set $index\n"
   "$widgets(entry) selection range 0 end\n"
   "}\n"
   "$widgets(this) close\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::HandleScrollbar {w {action \"unknown\"}} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "if {$options(-height) == 0} {\n"
   "set hlimit $options(-maxheight)\n"
   "} else {\n"
   "set hlimit $options(-height)\n"
   "}\n"
   "switch $action {\n"
   "\"grow\" {\n"
   "if {$hlimit > 0 && [$widgets(listbox) size] > $hlimit} {\n"
   "pack $widgets(vsb) -side right -fill y -expand n\n"
   "}\n"
   "}\n"
   "\"shrink\" {\n"
   "if {$hlimit > 0 && [$widgets(listbox) size] <= $hlimit} {\n"
   "pack forget $widgets(vsb)\n"
   "}\n"
   "}\n"
   "\"crop\" {\n"
   "pack $widgets(vsb) -side right -fill y -expand n\n"
   "}\n"
   "default {\n"
   "if {$hlimit > 0 && [$widgets(listbox) size] > $hlimit} {\n"
   "pack $widgets(vsb) -side right -fill y -expand n\n"
   "} else {\n"
   "pack forget $widgets(vsb)\n"
   "}\n"
   "}\n"
   "}\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::ComputeGeometry {w} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "if {$options(-height) == 0 && $options(-maxheight) != \"0\"} {\n"
   "set nitems [$widgets(listbox) size]\n"
   "if {$nitems > $options(-maxheight)} {\n"
   "$widgets(listbox) configure -height $options(-maxheight)\n"
   "} else {\n"
   "$widgets(listbox) configure -height 0\n"
   "}\n"
   "update idletasks\n"
   "}\n"
   "set bd [$widgets(popup) cget -borderwidth]\n"
   "set height [expr {[winfo reqheight $widgets(popup)] + $bd + $bd}]\n"
   "set width [winfo width $widgets(this)]\n"
   "set screenWidth  [winfo screenwidth $widgets(this)]\n"
   "set screenHeight [winfo screenheight $widgets(this)]\n"
   "set rootx        [winfo rootx $widgets(this)]\n"
   "set rooty        [winfo rooty $widgets(this)]\n"
   "set vrootx       [winfo vrootx $widgets(this)]\n"
   "set vrooty       [winfo vrooty $widgets(this)]\n"
   "set x  [expr {$rootx + $vrootx}]\n"
   "if {0} {\n"
   "set rightEdge [expr {$x + $width}]\n"
   "if {$rightEdge > $screenWidth} {\n"
   "set x [expr {$screenWidth - $width}]\n"
   "}\n"
   "if {$x < 0} {set x 0}\n"
   "}\n"
   "set y [expr {$rooty + $vrooty + [winfo reqheight $widgets(this)] + 1}]\n"
   "set bottomEdge [expr {$y + $height}]\n"
   "if {$bottomEdge >= $screenHeight} {\n"
   "set y [expr {($rooty - $height - 1) + $vrooty}]\n"
   "if {$y < 0} {\n"
   "if {$rooty > [expr {$screenHeight / 2}]} {\n"
   "set y 1\n"
   "set height [expr {$rooty - 1 - $y}]\n"
   "} else {\n"
   "set y [expr {$rooty + $vrooty +  "
   "[winfo reqheight $widgets(this)] + 1}]\n"
   "set height [expr {$screenHeight - $y}]\n"
   "}\n"
   "HandleScrollbar $widgets(this) crop\n"
   "}\n"
   "}\n"
   "if {$y < 0} {\n"
   "set y 0\n"
   "set height $screenheight\n"
   "}\n"
   "set geometry [format \"=%dx%d+%d+%d\" $width $height $x $y]\n"
   "return $geometry\n"
   "}\n"
   "proc ::combobox::DoInternalWidgetCommand {w subwidget command args} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "set subcommand $command\n"
   "set command [concat $widgets($subwidget) $command $args]\n"
   "if {[catch $command result]} {\n"
   "regsub $widgets($subwidget) $result $widgets(this) result\n"
   "switch $subwidget,$subcommand {\n"
   "listbox,index  {regsub \"index\"  $result \"list index\"  result}\n"
   "listbox,insert {regsub \"insert\" $result \"list insert\" result}\n"
   "listbox,delete {regsub \"delete\" $result \"list delete\" result}\n"
   "listbox,get    {regsub \"get\"    $result \"list get\"    result}\n"
   "listbox,size   {regsub \"size\"   $result \"list size\"   result}\n"
   "}\n"
   "error $result\n"
   "} else {\n"
   "return $result\n"
   "}\n"
   "}\n"
   "proc ::combobox::WidgetProc {w command args} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "upvar ::combobox::${w}::oldFocus oldFocus\n"
   "upvar ::combobox::${w}::oldFocus oldGrab\n"
   "set command [::combobox::Canonize $w command $command]\n"
   "set doWidgetCommand  "
   "[list ::combobox::DoInternalWidgetCommand $widgets(this)]\n"
   "if {$command == \"list\"} {\n"
   "set command \"list-[lindex $args 0]\"\n"
   "set args [lrange $args 1 end]\n"
   "}\n"
   "set result \"\"\n"
   "switch $command {\n"
   "bbox -\n"
   "delete -\n"
   "get -\n"
   "icursor -\n"
   "index -\n"
   "insert -\n"
   "scan -\n"
   "selection -\n"
   "xview {\n"
   "set result [eval $doWidgetCommand entry $command $args]\n"
   "}\n"
   "list-get  {set result [eval $doWidgetCommand listbox get $args]}\n"
   "list-index  {set result [eval $doWidgetCommand listbox index $args]}\n"
   "list-size  {set result [eval $doWidgetCommand listbox size $args]}\n"
   "select {\n"
   "if {[llength $args] == 1} {\n"
   "set index [lindex $args 0]\n"
   "set result [Select $widgets(this) $index]\n"
   "} else {\n"
   "error \"usage: $w select index\"\n"
   "}\n"
   "}\n"
   "subwidget {\n"
   "set knownWidgets [list button entry listbox popup vsb]\n"
   "if {[llength $args] == 0} {\n"
   "return $knownWidgets\n"
   "}\n"
   "set name [lindex $args 0]\n"
   "if {[lsearch $knownWidgets $name] != -1} {\n"
   "set result $widgets($name)\n"
   "} else {\n"
   "error \"unknown subwidget $name\"\n"
   "}\n"
   "}\n"
   "curselection {\n"
   "set result [eval $doWidgetCommand listbox curselection]\n"
   "}\n"
   "list-insert {\n"
   "eval $doWidgetCommand listbox insert $args\n"
   "set result [HandleScrollbar $w \"grow\"]\n"
   "}\n"
   "list-delete {\n"
   "eval $doWidgetCommand listbox delete $args\n"
   "set result [HandleScrollbar $w \"shrink\"]\n"
   "}\n"
   "toggle {\n"
   "if {$options(-state) == \"disabled\"} return\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "set result [$widgets(this) close]\n"
   "} else {\n"
   "set result [$widgets(this) open]\n"
   "}\n"
   "}\n"
   "open {\n"
   "if {$options(-editable)} {\n"
   "focus $widgets(entry)\n"
   "$widgets(entry) select range 0 end\n"
   "$widgets(entry) icur end\n"
   "}\n"
   "if {$options(-state) == \"disabled\"} {\n"
   "return 0\n"
   "}\n"
   "set geometry [::combobox::ComputeGeometry $widgets(this)]\n"
   "wm geometry $widgets(popup) $geometry\n"
   "update idletasks\n"
   "if {[winfo ismapped $widgets(popup)]} {\n"
   "return 0\n"
   "}\n"
   "set oldFocus [focus]\n"
   "$widgets(button) configure -relief sunken\n"
   "raise $widgets(popup) [winfo parent $widgets(this)]\n"
   "wm deiconify $widgets(popup)\n"
   "focus -force $widgets(entry)\n"
   "::combobox::Find $widgets(this) 1\n"
   "set status \"none\"\n"
   "set grab [grab current $widgets(this)]\n"
   "if {$grab != \"\"} {set status [grab status $grab]}\n"
   "set oldGrab [list $grab $status]\n"
   "unset grab status\n"
   "grab -global $widgets(this)\n"
   "event generate $widgets(listbox) <B1-Enter>\n"
   "return 1\n"
   "}\n"
   "close {\n"
   "if {![winfo ismapped $widgets(popup)]} {\n"
   "return 0\n"
   "}\n"
   "catch {focus $oldFocus} result\n"
   "catch {grab release $widgets(this)}\n"
   "catch {\n"
   "set status [lindex $oldGrab 1]\n"
   "if {$status == \"global\"} {\n"
   "grab -global [lindex $oldGrab 0]\n"
   "} elseif {$status == \"local\"} {\n"
   "grab [lindex $oldGrab 0]\n"
   "}\n"
   "unset status\n"
   "}\n"
   "$widgets(button) configure -relief raised\n"
   "wm withdraw $widgets(popup)\n"
   "set editable [::combobox::GetBoolean $options(-editable)]\n"
   "if {$editable} {\n"
   "$widgets(entry) selection range 0 end\n"
   "$widgets(button) configure -relief raised\n"
   "}\n"
   "tkCancelRepeat\n"
   "return 1\n"
   "}\n"
   "cget {\n"
   "if {[llength $args] != 1} {\n"
   "error \"wrong # args: should be $w cget option\"\n"
   "}\n"
   "set opt [::combobox::Canonize $w option [lindex $args 0]]\n"
   "if {$opt == \"-value\"} {\n"
   "set result [$widget(entry) get]\n"
   "} else {\n"
   "set result $options($opt)\n"
   "}\n"
   "}\n"
   "configure {\n"
   "set result [eval ::combobox::Configure {$w} $args]\n"
   "}\n"
   "default {\n"
   "error \"bad option \\\"$command\\\"\"\n"
   "}\n"
   "}\n"
   "return $result\n"
   "}\n";
char *ExtensionSource1 =
   "proc ::combobox::Configure {w args} {\n"
   "variable widgetOptions\n"
   "variable defaultEntryCursor\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "if {[llength $args] == 0} {\n"
   "set results {}\n"
   "foreach opt [lsort [array names widgetOptions]] {\n"
   "if {[llength $widgetOptions($opt)] == 1} {\n"
   "set alias $widgetOptions($opt)\n"
   "set optName $widgetOptions($alias)\n"
   "lappend results [list $opt $optName]\n"
   "} else {\n"
   "set optName  [lindex $widgetOptions($opt) 0]\n"
   "set optClass [lindex $widgetOptions($opt) 1]\n"
   "set default [option get $w $optName $optClass]\n"
   "lappend results [list $opt $optName $optClass  "
   "$default $options($opt)]\n"
   "}\n"
   "}\n"
   "return $results\n"
   "}\n"
   "if {[llength $args] == 1} {\n"
   "set opt [::combobox::Canonize $w option [lindex $args 0]]\n"
   "set optName  [lindex $widgetOptions($opt) 0]\n"
   "set optClass [lindex $widgetOptions($opt) 1]\n"
   "set default [option get $w $optName $optClass]\n"
   "set results [list $opt $optName $optClass  "
   "$default $options($opt)]\n"
   "return $results\n"
   "}\n"
   "if {[expr {[llength $args]%2}] == 1} {\n"
   "error \"value for \\\"[lindex $args end]\\\" missing\"\n"
   "}\n"
   "foreach {name value} $args {\n"
   "set name [::combobox::Canonize $w option $name]\n"
   "set opts($name) $value\n"
   "}\n"
   "foreach option [array names opts] {\n"
   "set newValue $opts($option)\n"
   "if {[info exists options($option)]} {\n"
   "set oldValue $options($option)\n"
   "}\n"
   "switch -- $option {\n"
   "-background {\n"
   "$widgets(frame)   configure -background $newValue\n"
   "$widgets(entry)   configure -background $newValue\n"
   "$widgets(listbox) configure -background $newValue\n"
   "$widgets(vsb)     configure -background $newValue\n"
   "$widgets(vsb)     configure -troughcolor $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-borderwidth {\n"
   "$widgets(frame) configure -borderwidth $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-command {\n"
   "set options($option) $newValue\n"
   "}\n"
   "-commandstate {\n"
   "if {$newValue != \"normal\" && $newValue != \"disabled\"} {\n"
   "set options($option) $oldValue\n"
   "set message \"bad state value \\\"$newValue\\\";\"\n"
   "append message \" must be normal or disabled\"\n"
   "error $message\n"
   "}\n"
   "set options($option) $newValue\n"
   "}\n"
   "-cursor {\n"
   "$widgets(frame) configure -cursor $newValue\n"
   "$widgets(entry) configure -cursor $newValue\n"
   "$widgets(listbox) configure -cursor $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-editable {\n"
   "if {$newValue} {\n"
   "$widgets(entry) configure  "
   "-state normal  "
   "-cursor $defaultEntryCursor\n"
   "} else {\n"
   "$widgets(entry) configure  "
   "-state disabled  "
   "-cursor $options(-cursor)\n"
   "}\n"
   "set options($option) $newValue\n"
   "}\n"
   "-font {\n"
   "$widgets(entry) configure -font $newValue\n"
   "$widgets(listbox) configure -font $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-foreground {\n"
   "$widgets(entry)   configure -foreground $newValue\n"
   "$widgets(button)  configure -foreground $newValue\n"
   "$widgets(listbox) configure -foreground $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-height {\n"
   "$widgets(listbox) configure -height $newValue\n"
   "HandleScrollbar $w\n"
   "set options($option) $newValue\n"
   "}\n"
   "-highlightbackground {\n"
   "$widgets(frame) configure -highlightbackground $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-highlightcolor {\n"
   "$widgets(frame) configure -highlightcolor $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-highlightthickness {\n"
   "$widgets(frame) configure -highlightthickness $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-image {\n"
   "if {[string length $newValue] > 0} {\n"
   "$widgets(button) configure -image $newValue\n"
   "} else {\n"
   "$widgets(button) configure -image ::combobox::bimage\n"
   "}\n"
   "set options($option) $newValue\n"
   "}\n"
   "-maxheight {\n"
   "$widgets(listbox) configure -height $options(-height)\n"
   "HandleScrollbar $w\n"
   "set options($option) $newValue\n"
   "}\n"
   "-relief {\n"
   "$widgets(frame) configure -relief $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-selectbackground {\n"
   "$widgets(entry) configure -selectbackground $newValue\n"
   "$widgets(listbox) configure -selectbackground $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-selectborderwidth {\n"
   "$widgets(entry) configure -selectborderwidth $newValue\n"
   "$widgets(listbox) configure -selectborderwidth $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-selectforeground {\n"
   "$widgets(entry) configure -selectforeground $newValue\n"
   "$widgets(listbox) configure -selectforeground $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-state {\n"
   "if {$newValue == \"normal\"} {\n"
   "set editable [::combobox::GetBoolean  "
   "$options(-editable)]\n"
   "if {$editable} {\n"
   "$widgets(entry) configure -state normal\n"
   "$widgets(entry) configure -takefocus 1\n"
   "}\n"
   "} elseif {$newValue == \"disabled\"}  {\n"
   "$widgets(entry) configure -state disabled\n"
   "$widgets(entry) configure -takefocus 0\n"
   "} else {\n"
   "set options($option) $oldValue\n"
   "set message \"bad state value \\\"$newValue\\\";\"\n"
   "append message \" must be normal or disabled\"\n"
   "error $message\n"
   "}\n"
   "set options($option) $newValue\n"
   "}\n"
   "-takefocus {\n"
   "$widgets(entry) configure -takefocus $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-textvariable {\n"
   "$widgets(entry) configure -textvariable $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-value {\n"
   "::combobox::SetValue $widgets(this) $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-width {\n"
   "$widgets(entry) configure -width $newValue\n"
   "$widgets(listbox) configure -width $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "-xscrollcommand {\n"
   "$widgets(entry) configure -xscrollcommand $newValue\n"
   "set options($option) $newValue\n"
   "}\n"
   "}\n"
   "}\n"
   "}\n"
   "proc ::combobox::VTrace {w args} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "upvar ::combobox::${w}::ignoreTrace ignoreTrace\n"
   "if {[info exists ignoreTrace]} return\n"
   "::combobox::SetValue $widgets(this) [set ::$options(-textvariable)]\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::SetValue {w newValue} {\n"
   "upvar ::combobox::${w}::widgets     widgets\n"
   "upvar ::combobox::${w}::options     options\n"
   "upvar ::combobox::${w}::ignoreTrace ignoreTrace\n"
   "upvar ::combobox::${w}::oldValue    oldValue\n"
   "if {[info exists options(-textvariable)]  "
   "&& [string length $options(-textvariable)] > 0} {\n"
   "set variable ::$options(-textvariable)\n"
   "set $variable $newValue\n"
   "} else {\n"
   "set oldstate [$widgets(entry) cget -state]\n"
   "$widgets(entry) configure -state normal\n"
   "$widgets(entry) delete 0 end\n"
   "$widgets(entry) insert 0 $newValue\n"
   "$widgets(entry) configure -state $oldstate\n"
   "}\n"
   "set oldValue $newValue\n"
   "CallCommand $w $newValue\n"
   "return \"\"\n"
   "}\n"
   "proc ::combobox::CallCommand {w newValue} {\n"
   "upvar ::combobox::${w}::widgets widgets\n"
   "upvar ::combobox::${w}::options options\n"
   "if {$options(-commandstate) == \"normal\" &&  "
   "[string length $options(-command)] > 0} {\n"
   "set args [list $widgets(this) $newValue]\n"
   "uplevel \\#0 $options(-command) $args\n"
   "}\n"
   "}\n"
   "proc ::combobox::GetBoolean {value {errorValue 1}} {\n"
   "if {[catch {expr {([string trim $value])?1:0}} res]} {\n"
   "return $errorValue\n"
   "} else {\n"
   "return $res\n"
   "}\n"
   "}\n"
   "proc ::combobox::convert {w args} {\n"
   "set result {}\n"
   "if {![winfo exists $w]} {\n"
   "error \"window \\\"$w\\\" doesn't exist\"\n"
   "}\n"
   "while {[llength $args] > 0} {\n"
   "set option [lindex $args 0]\n"
   "set args [lrange $args 1 end]\n"
   "switch -exact -- $option {\n"
   "-x {\n"
   "set value [lindex $args 0]\n"
   "set args [lrange $args 1 end]\n"
   "set win $w\n"
   "while {[winfo class $win] != \"Combobox\"} {\n"
   "incr value [winfo x $win]\n"
   "set win [winfo parent $win]\n"
   "if {$win == \".\"} break\n"
   "}\n"
   "lappend result $value\n"
   "}\n"
   "-y {\n"
   "set value [lindex $args 0]\n"
   "set args [lrange $args 1 end]\n"
   "set win $w\n"
   "while {[winfo class $win] != \"Combobox\"} {\n"
   "incr value [winfo y $win]\n"
   "set win [winfo parent $win]\n"
   "if {$win == \".\"} break\n"
   "}\n"
   "lappend result $value\n"
   "}\n"
   "-w -\n"
   "-W {\n"
   "set win $w\n"
   "while {[winfo class $win] != \"Combobox\"} {\n"
   "set win [winfo parent $win]\n"
   "if {$win == \".\"} break;\n"
   "}\n"
   "lappend result $win\n"
   "}\n"
   "}\n"
   "}\n"
   "return $result\n"
   "}\n"
   "proc ::combobox::Canonize {w object opt} {\n"
   "variable widgetOptions\n"
   "variable columnOptions\n"
   "variable widgetCommands\n"
   "variable listCommands\n"
   "variable scanCommands\n"
   "switch $object {\n"
   "command {\n"
   "if {[lsearch -exact $widgetCommands $opt] >= 0} {\n"
   "return $opt\n"
   "}\n"
   "set list $widgetCommands\n"
   "foreach element $list {\n"
   "set tmp($element) \"\"\n"
   "}\n"
   "set matches [array names tmp ${opt}*]\n"
   "}\n"
   "{list command} {\n"
   "if {[lsearch -exact $listCommands $opt] >= 0} {\n"
   "return $opt\n"
   "}\n"
   "set list $listCommands\n"
   "foreach element $list {\n"
   "set tmp($element) \"\"\n"
   "}\n"
   "set matches [array names tmp ${opt}*]\n"
   "}\n"
   "{scan command} {\n"
   "if {[lsearch -exact $scanCommands $opt] >= 0} {\n"
   "return $opt\n"
   "}\n"
   "set list $scanCommands\n"
   "foreach element $list {\n"
   "set tmp($element) \"\"\n"
   "}\n"
   "set matches [array names tmp ${opt}*]\n"
   "}\n"
   "option {\n"
   "if {[info exists widgetOptions($opt)]  "
   "&& [llength $widgetOptions($opt)] == 2} {\n"
   "return $opt\n"
   "}\n"
   "set list [array names widgetOptions]\n"
   "set matches [array names widgetOptions ${opt}*]\n"
   "}\n"
   "}\n"
   "if {[llength $matches] == 0} {\n"
   "set choices [HumanizeList $list]\n"
   "error \"unknown $object \\\"$opt\\\"; must be one of $choices\"\n"
   "} elseif {[llength $matches] == 1} {\n"
   "set opt [lindex $matches 0]\n"
   "switch $object {\n"
   "option {\n"
   "set opt [lindex $matches 0]\n"
   "if {[llength $widgetOptions($opt)] == 1} {\n"
   "set opt $widgetOptions($opt)\n"
   "}\n"
   "}\n"
   "}\n"
   "return $opt\n"
   "} else {\n"
   "set choices [HumanizeList $list]\n"
   "error \"ambiguous $object \\\"$opt\\\"; must be one of $choices\"\n"
   "}\n"
   "}\n"
   "proc ::combobox::HumanizeList {list} {\n"
   "if {[llength $list] == 1} {\n"
   "return [lindex $list 0]\n"
   "} else {\n"
   "set list [lsort $list]\n"
   "set secondToLast [expr {[llength $list] -2}]\n"
   "set most [lrange $list 0 $secondToLast]\n"
   "set last [lindex $list end]\n"
   "return \"[join $most {, }] or $last\"\n"
   "}\n"
   "}\n";

RexxFunctionHandler TkCombobox  ;
RexxFunctionHandler TkComboboxClose  ;
RexxFunctionHandler TkComboboxICursor;
RexxFunctionHandler TkComboboxListDelete ;
RexxFunctionHandler TkComboboxListGet ;
RexxFunctionHandler TkComboboxListIndex ;
RexxFunctionHandler TkComboboxListInsert ;
RexxFunctionHandler TkComboboxListSize ;
RexxFunctionHandler TkComboboxOpen  ;
RexxFunctionHandler TkComboboxSelect  ;
RexxFunctionHandler TkComboboxSubwidget  ;
RexxFunctionHandler TkComboboxLoadFuncs    ;
RexxFunctionHandler TkComboboxDropFuncs    ;

/*-----------------------------------------------------------------------------
 * Table of TK Functions. Used to install/de-install functions.
 * If you change this table, don't forget to change the table at the end
 * of this file.
 *----------------------------------------------------------------------------*/
RexxFunction RxPackageFunctions[] = {
   { "TKCOMBOBOXDROPFUNCS"       ,TkComboboxDropFuncs       ,"TkComboboxDropFuncs"       , 1 },
   { "TKCOMBOBOXLOADFUNCS"       ,TkComboboxLoadFuncs       ,"TkComboboxLoadFuncs"       , 0 }, /* don't load this from a DLL */
   { "TKCOMBOBOX"                ,TkCombobox                ,"TkCombobox"                , 1 },
   { "TKCOMBOBOXICURSOR"         ,TkComboboxICursor         ,"TkComboboxICurso"          , 1 },
   { "TKCOMBOBOXLISTDELETE"      ,TkComboboxListDelete      ,"TkComboboxListDelete"      , 1 },
   { "TKCOMBOBOXLISTGET"         ,TkComboboxListGet         ,"TkComboboxListGet"         , 1 },
   { "TKCOMBOBOXLISTINDEX"       ,TkComboboxListIndex       ,"TkComboboxListIndex"       , 1 },
   { "TKCOMBOBOXLISTINSERT"      ,TkComboboxListInsert      ,"TkComboboxListInsert"      , 1 },
   { "TKCOMBOBOXLISTSIZE"        ,TkComboboxListSize        ,"TkComboboxListSize"        , 1 },
   { "TKCOMBOBOXSELECT"          ,TkComboboxSelect          ,"TkComboboxSelect"          , 1 },
   { "TKCOMBOBOXSUBWIDGET"       ,TkComboboxSubwidget       ,"TkComboboxSubwidget"       , 1 },
   { NULL, NULL, NULL, 0 }
};

static char czTclCommand[TCLCOMMANDLEN];
static REXXTKDATA *RexxTkData;
   
#ifdef WIN32
Tcl_Interp *RexxTk_TclCreateInterp(void)
{
   return RexxTkData->Dyn_TclCreateInterp();
}

int RexxTk_TclEval(Tcl_Interp *interp, char *string)
{
   return RexxTkData->Dyn_TclEval( interp, string );
}

int RexxTk_TclInit(Tcl_Interp *interp)
{
   return RexxTkData->Dyn_TclInit( interp );
}

int RexxTk_TkInit(Tcl_Interp *interp)
{
   return RexxTkData->Dyn_TkInit( interp );
}
#endif

/*
 * Rexx/Tk combobox functions start here...
 */

/*
 * Combobox:create pathName ?options?
 * TkCombobox(pathName [,options])
 */
RFH_RETURN_TYPE TkCombobox
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   return rtk_TypeA(RexxTkData,czTclCommand,name,"combobox::combobox", argc, argv, retstr);
}

/*
 * pathName list delete first ?last?
 * TkComboboxListDelete(pathName, first [,last])
 */
RFH_RETURN_TYPE TkComboboxListDelete
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 3 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"list delete", argc, argv, retstr);
}

/*
 * pathName list size
 * TkComboboxListSize(pathName)
 */
RFH_RETURN_TYPE TkComboboxListSize
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 1, 1 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"list size", argc, argv, retstr);
}

/*
 * pathName list index index
 * TkComboboxListIndex(pathName, index)
 */
RFH_RETURN_TYPE TkComboboxListIndex
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"list index", argc, argv, retstr);
}

/*
 * pathName list get first ?last?
 * TkComboboxListGet(pathName, first [,last])
 */
RFH_RETURN_TYPE TkComboboxListGet
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 3 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"list get", argc, argv, retstr);
}

/*
 * pathName list insert index ?text...?
 * TkComboboxListInsert(pathName, index [,args...])
 */
RFH_RETURN_TYPE TkComboboxListInsert
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 0 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"list insert", argc, argv, retstr);
}

/*
 * pathName icursor index
 * TkComboboxICursor(pathName, index)
 */
RFH_RETURN_TYPE TkComboboxICursor
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"icursor", argc, argv, retstr);
}

/*
 * pathName select index
 * TkComboboxSelect(pathName, index)
 */
RFH_RETURN_TYPE TkComboboxSelect
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 2, 2 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"select", argc, argv, retstr);
}

/*
 * pathName subwidget ?name?
 * TkComboboxSelect(pathName [,name])
 */
RFH_RETURN_TYPE TkComboboxSubwidget
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   FunctionPrologue( (char *)name, argc, argv );

   if (RexxTkData->REXXTK_IntCode) ClearIntError( RexxTkData);

   if ( my_checkparam( name, argc, 1, 2 ) )
      return 1;

   return rtk_TypeC(RexxTkData,czTclCommand,name,"subwidget", argc, argv, retstr);
}



RFH_RETURN_TYPE TkComboboxDropFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
 ULONG rc=0;
 int unload=0;

 if ( my_checkparam(name, argc, 0, 1 ) )
    return( 1 );
 if ( argv[0].strlength == 6
 &&   memcmpi( argv[0].strptr, "UNLOAD", 6 ) == 0 )
    unload = 1;
 rc = DeregisterRxFunctions( unload );
 return RxReturnNumber( retstr, rc );
}


RFH_RETURN_TYPE TkComboboxLoadFuncs
   (RFH_ARG0_TYPE name, RFH_ARG1_TYPE argc, RFH_ARG2_TYPE argv, RFH_ARG3_TYPE stck, RFH_ARG4_TYPE retstr)
{
   ULONG rc = 0L;

#if defined(DYNAMIC_LIBRARY)
   if ( !QueryRxFunction( "TKWAIT" ) )
   {
      fprintf(stderr,"The base Rexx/Tk function package must be loaded before this one\n");
      return RxReturnNumber( retstr, 1 );
   }
   /*
    * get the pointer to the tcl Interpreter and the base data from base Rexx/Tk
    * library
    */
   if ( argc == 0 )
   {
      fprintf(stderr,"You must pass the return value from TkGetBaseData() as the one and only argument.\n");
      return RxReturnNumber( retstr, 1 );
   }
   RexxTkData = (REXXTKDATA *)atol(argv[0].strptr);
   rc = InitRxPackage( NULL );
   /* 
    * Register all external functions
    */
   if ( !rc )
   {
      rc = RegisterRxFunctions( );
   }
#endif
   return RxReturnNumber( retstr, rc );
}
   
/*
 * The following functions are used in rxpackage.c
 */

/*-----------------------------------------------------------------------------
 * Execute any initialisation
 *----------------------------------------------------------------------------*/
int InitialisePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   InternalTrace( "InitialisePackage", NULL );

   /*
    * Install the Combobox widget
    */
   if ( Tcl_Eval(RexxTkData->RexxTkInterp, ExtensionSource ) !=TCL_OK) {
      fprintf(stderr, "Tk_Eval for Combobox widget failed miserably at line %d: %s\n", RexxTkData->RexxTkInterp->errorLine, RexxTkData->RexxTkInterp->result);
      return 1;
   }
   if ( Tcl_Eval(RexxTkData->RexxTkInterp, ExtensionSource1 ) !=TCL_OK) {
      fprintf(stderr, "Tk_Eval for Combobox widget failed miserably at line %d: %s\n", RexxTkData->RexxTkInterp->errorLine, RexxTkData->RexxTkInterp->result);
      return 1;
   }
   DEBUGDUMP(fprintf(stderr,"%s-%d: After Tcl_Eval()\n",__FILE__,__LINE__);)
   return 0;
}

/*-----------------------------------------------------------------------------
 * Execute any termination
 *----------------------------------------------------------------------------*/
int TerminatePackage

#ifdef HAVE_PROTO
   ( void )
#else
   ( )
#endif

{
   return 0;
}


#if defined(USE_REXX6000)
/*
 * This function is used as the entry point for the REXX/6000
 * Rexx Interpreter
 * If you change this table, don't forget to change the table at the
 * start of this file.
 */
USHORT InitFunc( RXFUNCBLOCK **FuncBlock )
{
   static RXFUNCBLOCK funcarray[] =
   {
      { "TKCOMBOBOXDROPFUNCS"       ,TkComboboxDropFuncs      ,NULL },
      { "TKCOMBOBOXLOADFUNCS"       ,TkComboboxLoadFuncs      ,NULL },
      { "TKCOMBOBOX"                ,TkCombobox               ,NULL },
      { "TKCOMBOBOXICURSOR"         ,TkComboboxICursor        ,NULL },
      { "TKCOMBOBOXLISTDELETE"      ,TkComboboxListDelete     ,NULL },
      { "TKCOMBOBOXLISTGET"         ,TkComboboxListGet        ,NULL },
      { "TKCOMBOBOXLISTINDEX"       ,TkComboboxListIndex      ,NULL },
      { "TKCOMBOBOXLISTINSERT"      ,TkComboboxListInsert     ,NULL },
      { "TKCOMBOBOXLISTSIZE"        ,TkComboboxListSize       ,NULL },
      { "TKCOMBOBOXSELECT"          ,TkComboboxSelect         ,NULL },
      { "TKCOMBOBOXSUBWIDGET"       ,TkComboboxSubwidget      ,NULL },
      { NULL, NULL, NULL }
   } ;
   *FuncBlock = funcarray;
   return (USHORT)0;
}
#endif
