/*
 *  Copyright (C) 2000 heXoNet Support GmbH, D-66424 Homburg.
 *  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

#include "rfbServer.h"
#include <stdlib.h>
#include <string.h>


namespace rfb {



void computeSubrectDimensions8( unsigned char *, unsigned int &, unsigned int &, unsigned int );
void computeSubrectDimensions16( unsigned char *, unsigned int &, unsigned int &, unsigned int );
void computeSubrectDimensions24( unsigned char *, unsigned int &, unsigned int &, unsigned int );
void computeSubrectDimensions32( unsigned char *, unsigned int &, unsigned int &, unsigned int );



unsigned char* Server::encodeRectangleHextileTile( unsigned int   _x,
                                                   unsigned int   _y,
                                                   unsigned int   _width,
                                                   unsigned int   _height,
                                                   unsigned char* _buffer,
                                                   unsigned int&  _size,
                                                   unsigned int   _maxSize,
						   Framebuffer *cframebuffer )
{
  _size = 0;
  bool visited[_width * _height];
  unsigned int i = 0;
  while ( i < _width * _height )
    visited[i++] = false;
  
  switch ( cframebuffer->pixelFormat.bits_per_pixel ) {

    case 8: {
      if ( _maxSize < _size + 2 ) return NULL;
      *((CARD8 *) (_buffer)) = 0;
      CARD8 *subrectangles = (CARD8 *) _buffer + 1;
      *subrectangles = 0;
      _size += 2;
      unsigned int x, y, w, h;
      unsigned char *src;
      for ( y = 0; y < _height; y++ ) {
        x = 0;
        while ( x < _width ) {
          if ( !visited[y*_width + x] ) {
            src = cframebuffer->data + ((_x + x) * 1 + (_y + y) * cframebuffer->bytesPerLine);
            if ( src[0] ) {  // black background
              if ( _maxSize < _size + 3 ) return NULL;
              (*subrectangles) = (*subrectangles) + 1;
              w = _width - x;
              h = _height - y;
              computeSubrectDimensions8( src, w, h, cframebuffer->bytesPerLine );
              unsigned int x1, y1;
              for ( y1 = y; y1 < h; y1++ )
                for ( x1 = x; x1 < w; x1++ )
                  visited[y1 * _width + x1] = true;
              CARD8 x_y_position = (x << 4) + y;
              CARD8 width_height = ((w-1) << 4) + (h-1);
              memcpy( _buffer + _size + 0, src,           1 );
              memcpy( _buffer + _size + 1, &x_y_position, 1 );
              memcpy( _buffer + _size + 2, &width_height, 1 );
              _size += 3;
              x += w;
            }
            else
              x++;
          }
          else
            x++;
        }
      }
    } break;
    case 16: {
      if ( _maxSize < _size + 3 ) return NULL;
      *((CARD16 *) (_buffer + _size)) = 0;
      CARD8 *subrectangles = (CARD8 *) _buffer + 2;
      *subrectangles = 0;
      _size += 3;
      unsigned int x, y, w, h;
      unsigned char *src;
      for ( y = 0; y < _height; y++ ) {
        x = 0;
        while ( x < _width ) {
          if ( !visited[y*_width + x] ) {
            src = cframebuffer->data + ((_x + x) * 2 + (_y + y) * cframebuffer->bytesPerLine);
            if ( src[0] || src[1] ) {  // black background
              if ( _maxSize < _size + 4 ) return NULL;
              (*subrectangles) = (*subrectangles) + 1;
              w = _width - x;
              h = _height - y;
              computeSubrectDimensions16( src, w, h, cframebuffer->bytesPerLine );
              unsigned int x1, y1;
              for ( y1 = y; y1 < h; y1++ )
                for ( x1 = x; x1 < w; x1++ )
                  visited[y1 * _width + x1] = true;
              CARD8 x_y_position = (x << 4) + y;
              CARD8 width_height = ((w-1) << 4) + (h-1);
              memcpy( _buffer + _size + 0, src,           2 );
              memcpy( _buffer + _size + 2, &x_y_position, 1 );
              memcpy( _buffer + _size + 3, &width_height, 1 );
              _size += 4;
              x += w;
            }
            else
              x++;
          }
          else
            x++;
        }
      }
    } break;
    case 32: {
      if ( _maxSize < _size + 5 ) return NULL;
      *((CARD32 *) (_buffer + _size)) = 0;
      CARD8 *subrectangles = (CARD8 *) _buffer + 4;
      *subrectangles = 0;
      _size += 5;
      unsigned int x, y, w, h;
      unsigned char *src;
      for ( y = 0; y < _height; y++ ) {
        x = 0;
        while ( x < _width ) {
          if ( !visited[y*_width + x] ) {
            src = cframebuffer->data + ((_x + x) * 4 + (_y + y) * cframebuffer->bytesPerLine);
            if ( src[0] || src[1] || src[2] || src[3] ) {  // black background
              if ( _maxSize < _size + 6 ) return NULL;
              (*subrectangles) = (*subrectangles) + 1;
              w = _width - x;
              h = _height - y;
              computeSubrectDimensions32( src, w, h, cframebuffer->bytesPerLine );
              unsigned int x1, y1;
              for ( y1 = y; y1 < h; y1++ )
                for ( x1 = x; x1 < w; x1++ )
                  visited[y1 * _width + x1] = true;
              CARD8 x_y_position = (x << 4) + y;
              CARD8 width_height = ((w-1) << 4) + (h-1);
              memcpy( _buffer + _size + 0, src,           4 );
              memcpy( _buffer + _size + 4, &x_y_position, 1 );
              memcpy( _buffer + _size + 5, &width_height, 1 );
              _size += 6;
              x += w;
            }
            else
              x++;
          }
          else
            x++;
        }
      }
    } break;
    
  }
  return _buffer;
}


#define HEXTILE_RAW                   1
#define HEXTILE_BACKGROUND_SPECIFIED  2
#define HEXTILE_FOREGROUND_SPECIFIED  4
#define HEXTILE_ANY_SUBRECTS          8
#define HEXTILE_SUBRECTS_COLOURED    16



unsigned char* Server::encodeRectangleHextile( unsigned int   _x,
                                               unsigned int   _y,
                                               unsigned int   _width,
                                               unsigned int   _height,
                                               unsigned char* _buffer,
                                               unsigned int&  _size,
                                               unsigned int   _maxSize )
{
    _size = 0;

    Framebuffer *cframebuffer;
    if ( clientPixelFormat ) {
      cframebuffer = new Framebuffer;
      cframebuffer->pixelFormat = (*clientPixelFormat);
      cframebuffer->bytesPerLine = _width * (clientPixelFormat->bits_per_pixel >> 3 );
      cframebuffer->data = (unsigned char*) malloc( cframebuffer->bytesPerLine * _height );
      unsigned int cy;
      unsigned char *src = framebuffer->data
                           + _x * (framebuffer->pixelFormat.bits_per_pixel >> 3 )
                           + _y * framebuffer->bytesPerLine;
      for ( cy = 0; cy < _height; cy++ ) {
        encodePixels( cframebuffer->data + cy * cframebuffer->bytesPerLine, src, _width );
        src += framebuffer->bytesPerLine;
      }
      cframebuffer->data -= ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                            + _y * cframebuffer->bytesPerLine );
    } else {
      cframebuffer = framebuffer;
    }

    int y = 0;
    while ( y < _height ) {
        int h = (_height - y > 16)? 16 : (_height - y);
        int x = 0;
        while ( x < _width ) {
            int w = (_width - x > 16)? 16 : (_width - x);
            if ( _maxSize < _size + 1 )  {
                if ( clientPixelFormat ) {
                  cframebuffer->data += ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                                         + _y * cframebuffer->bytesPerLine );
	          free( cframebuffer->data );
	          free( cframebuffer );
     	        }
                return NULL;
            }
	    CARD8 *subencoding = (CARD8*) (_buffer + _size);
	    _size += 1;
	    switch ( cframebuffer->pixelFormat.bits_per_pixel ) {

	        case 8: {
		    unsigned int size;
		    unsigned int maxSize = w * h;
		    if ( maxSize > _maxSize - _size )
		         maxSize = _maxSize - _size;
		    if ( encodeRectangleHextileTile( x+_x, y+_y, w, h,
		                                     _buffer + _size,
				    		     size, maxSize,
	             		     	             cframebuffer ) ) {
		        _size += size;
			(*subencoding) = HEXTILE_BACKGROUND_SPECIFIED
				       | HEXTILE_ANY_SUBRECTS
				       | HEXTILE_SUBRECTS_COLOURED;
		    } else {
		        if ( maxSize == w * h ) {
			    int l;
			    for ( l = 0; l < h; l++ ) {
			      memcpy( _buffer + _size, cframebuffer->data + (x + _x) + cframebuffer->bytesPerLine * (l + y + _y), w );
			      _size += w;
			    }
			    (*subencoding) = HEXTILE_RAW;
			} else {
                            if ( clientPixelFormat ) {
                                cframebuffer->data += ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                                                    + _y * cframebuffer->bytesPerLine );
	                        free( cframebuffer->data );
	                        free( cframebuffer );
     	                    }
                            return NULL;
			}
		    }
		}
		break;

	        case 16: {
		    unsigned int size;
		    unsigned int maxSize = w * h * 2;
		    if ( maxSize > _maxSize - _size )
		         maxSize = _maxSize - _size;
		    if ( encodeRectangleHextileTile( x+_x, y+_y, w, h,
		                                     _buffer + _size,
				    		     size, maxSize,
	             		     	             cframebuffer ) ) {
		        _size += size;
			(*subencoding) = HEXTILE_BACKGROUND_SPECIFIED
				       | HEXTILE_ANY_SUBRECTS
				       | HEXTILE_SUBRECTS_COLOURED;
		    } else {
		        if ( maxSize == w * h * 2 ) {
			    int l;
			    for ( l = 0; l < h; l++ ) {
			      memcpy( _buffer + _size, cframebuffer->data + 2 * (x + _x) + cframebuffer->bytesPerLine * (l + y + _y), 2 * w );
			      _size += w * 2;
			    }
			    (*subencoding) = HEXTILE_RAW;
			} else {
                            if ( clientPixelFormat ) {
                                cframebuffer->data += ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                                                    + _y * cframebuffer->bytesPerLine );
	                        free( cframebuffer->data );
	                        free( cframebuffer );
     	                    }
                            return NULL;
			}
		    }
		}
		break;

	        case 32: {
		    unsigned int size;
		    unsigned int maxSize = w * h * 4;
		    if ( maxSize > _maxSize - _size )
		         maxSize = _maxSize - _size;
		    if ( encodeRectangleHextileTile( x+_x, y+_y, w, h,
		                                     _buffer + _size,
				    		     size, maxSize,
	             		     	             cframebuffer ) ) {
		        _size += size;
			(*subencoding) = HEXTILE_BACKGROUND_SPECIFIED
				       | HEXTILE_ANY_SUBRECTS
				       | HEXTILE_SUBRECTS_COLOURED;
		    } else {
		        if ( maxSize == w * h * 4 ) {
			    int l;
			    for ( l = 0; l < h; l++ ) {
			      memcpy( _buffer + _size, cframebuffer->data + 4 * (x + _x) + cframebuffer->bytesPerLine * (l + y + _y), 4 * w );
			      _size += 4 * w;
			    }
			    (*subencoding) = HEXTILE_RAW;
			} else {
                            if ( clientPixelFormat ) {
                                cframebuffer->data += ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                                                    + _y * cframebuffer->bytesPerLine );
	                        free( cframebuffer->data );
	                        free( cframebuffer );
     	                    }
                            return NULL;
			}
		    }
		}
		break;

	    }
            x += 16;
        }
        y += 16;
    }
    if ( clientPixelFormat ) {
      cframebuffer->data += ( _x * (cframebuffer->pixelFormat.bits_per_pixel >> 3 )
                              + _y * cframebuffer->bytesPerLine );
      free( cframebuffer->data );
      free( cframebuffer );
    }
    return _buffer;
}


} // namespace rfb
