/*
 *  address-dialog.c     -
 *
 *  Created: 20020112
 *
 *  Copyright (c) 2001, 2002 by Tomasz Trojanowski
 *  All rights reserved
 *
 *  $Id: address-dialog.c,v 1.21 2002/03/19 19:14:36 tomek Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <address-dialog.h>

#include <intl.h>
#include <common.h>

static gchar *country_list_items[] =
{
    N_("Argentine"),
    N_("Australia"),
    N_("Austria"),
    N_("Belgium"),
    N_("Brazil"),
    N_("Bulgaria"),
    N_("Canada"),
    N_("Chile"),
    N_("China"),
    N_("Czech"),
    N_("Denmark"),
    N_("Dominikana"),
    N_("Egypt"),
    N_("France"),
    N_("Germany"),
    N_("Great Britain"),
    N_("Greece"),
    N_("Hong Kong"),
    N_("Hungary"),
    N_("India"),
    N_("Irleand"),
    N_("Island"),
    N_("Israel"),
    N_("Italy"),
    N_("Japan"),
    N_("Mexico"),
    N_("Netherlands"),
    N_("Norway"),
    N_("Poland"),
    N_("Portugal"),
    N_("Russia"),
    N_("USA"),
    N_("Spain"),
    N_("Switzerland"),
    N_("Sweden"),
    N_("Turkey")
};

/* Funkcja tworzca list krajw. */
static GList *country_list()
{
    GList *list = NULL;
    gint i;

    for (i = 0; i < sizeof(country_list_items) / sizeof(char *); i++)
    {
        list = g_list_append(list, (gpointer)gettext(country_list_items[i]));
    }
    
    return list;
}

/* Funkcja obsugujca wcinicie przycisku "OK" okienka dialogowego. */
static void ok_button_clicked(GtkButton *button, contact_data *data)
{
    gpointer entry;
    gboolean modified = FALSE;
    gchar *text;
    guint length;
    
    g_assert(data != NULL);
    
    /* Przepisywanie danych z elementw okna dialogowego do struktury
       danych kontaktu i sprawdzanie czy dane zostay zmodyfikowane,
       a jeeli tak - ustawianie flagi modified. */
    entry = gtk_object_get_data(GTK_OBJECT(data->sub_dialog), "street");
    g_assert(entry != NULL);
    length = gtk_text_get_length(GTK_TEXT(entry));
    if (length > MAX_STREET)
        length = MAX_STREET;
    text = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, length);
    if (strcmp(text, data->address[data->address_type].street) != 0)
    {
        strcpy(data->address[data->address_type].street, text);
	modified = TRUE;
    }
    g_free(text);

    entry = gtk_object_get_data(GTK_OBJECT(data->sub_dialog), "city");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->address[data->address_type].city) != 0)
    {
        strcpy(data->address[data->address_type].city, text);
	modified = TRUE;
    }
    
    entry = gtk_object_get_data(GTK_OBJECT(data->sub_dialog), "state");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->address[data->address_type].state) != 0)
    {
        strcpy(data->address[data->address_type].state, text);
	modified = TRUE;
    }
    
    entry = gtk_object_get_data(GTK_OBJECT(data->sub_dialog), "zip");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->address[data->address_type].zip) != 0)
    {
        strcpy(data->address[data->address_type].zip, text);
	modified = TRUE;
    }
    
    entry = gtk_object_get_data(GTK_OBJECT(data->sub_dialog), "country");
    g_assert(entry != NULL);
    text = gtk_entry_get_text(GTK_ENTRY(entry));
    if (strcmp(text, data->address[data->address_type].country) != 0)
    {
        strcpy(data->address[data->address_type].country, text);
	modified = TRUE;
    }
    
    /* Zamyka okienko dialogowe. */
    gtk_widget_destroy(data->sub_dialog);
    data->sub_dialog = NULL;

    if (modified)
    {
        /* Przepisanie penego adresu do pola tekstowego adresu. */
	create_full_address(data);

        /* Przepisywanie flagi modified do flagi modified struktury danych
           kontaktu. */
        data->address[data->address_type].modified = TRUE;
	data->global_modified = TRUE;	   
    }
}

/* Funkcja tworzca okno dialogowe edycji adresu.

   Struktura okna:
   
     GtkDialog (address_dialog)
      +- GtkVBox (dialog_vbox)
      |   +- GtkFrame (frame)                                    "Szczegy"
      |       +- GtkFixed (fixed)
      |           +- GtkLabel                                        "Ulica"
      |           +- GtkText ("street")
      |           +- GtkLabel                                  "Miejscowo"
      |           +- GtkEntry ("city")
      |           +- GtkLabel                                  "Wojewdztwo"
      |           +- GtkEntry ("state")
      |           +- GtkLabel                                 "Kod pocztowy"
      |           +- GtkEntry ("zip")
      |           +- GtkLabel                                         "Kraj"
      |           +- GtkCombo
      |               +- GtkEntry ("country")
      +- GtkHBox (dialog_action_area)
          +- GtkHButtonBox (hbuttonbox)
	      +- GtkButton                                              "OK"
	      +- GtkButton                                          "Anuluj"

   Klucze zwizane z okienkiem edycji adresu:
   
     street  - wskanik do pola tekstowego "Ulica"
     city    - wskanik do pola edycji "Miejscowo"
     state   - wskanik do pola edycji "Wojewdztwo"
     zip     - wskanik do pola edycji "Kod pocztowy"
     country - wskanik do pola edycji pols wyboru "Kraj"
   
*/
void create_address_dialog(contact_data *data)
{
    GtkWidget *address_dialog;
    GtkWidget *dialog_vbox;
    GtkWidget *frame;
    GtkWidget *fixed;
    GtkWidget *label;
    GtkWidget *text;
    GtkWidget *entry;
    GtkWidget *combo;
    GtkWidget *dialog_action_area;
    GtkWidget *hbuttonbox;
    GtkWidget *button;
    
    GList *items = NULL;
    
    /* Utworzenie okienka dialogowego. */
    address_dialog = gtk_dialog_new();
    data->sub_dialog = address_dialog;

    /* Przypisanie funkcji obsugi sygnaw do okienka dialogowego. */
    gtk_signal_connect(GTK_OBJECT(address_dialog), "delete_event",
                       GTK_SIGNAL_FUNC(dialog_delete), address_dialog);
    gtk_signal_connect(GTK_OBJECT(address_dialog), "key_press_event",
                       GTK_SIGNAL_FUNC(dialog_key_press), address_dialog);

    /* Ustalenie tytuu i rozmiarw okna. */
    gtk_window_set_title(GTK_WINDOW(address_dialog), _("Address"));
    gtk_widget_set_usize(GTK_WIDGET(address_dialog), 288, 284);
    
        /* Obszar wntrza okna dialogowego. */
	dialog_vbox = GTK_DIALOG(address_dialog)->vbox;
	gtk_widget_show(dialog_vbox);
	
	    /* Ramka "Szczegy". */
	    frame = gtk_frame_new(_("Details"));
	    gtk_widget_show(frame);
	    
	    gtk_box_pack_start(GTK_BOX(dialog_vbox), frame, TRUE, TRUE, 0);
	    
	    gtk_container_set_border_width(GTK_CONTAINER(frame), 6);
	    gtk_frame_set_label_align(GTK_FRAME(frame), 0.05, 0.5);
	    
	        /* Wntrze ramki. */
		fixed = gtk_fixed_new();
		gtk_widget_show(fixed);
		
		gtk_container_add(GTK_CONTAINER(frame), fixed);
		
		    /* Etykieta "Ulica". */
		    label = gtk_label_new(_("Street:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 8);
		    gtk_widget_set_uposition(label, 8, 8);
		    gtk_widget_set_usize(label, 88, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole tekstowe "Ulica". */
		    text = gtk_text_new(NULL, NULL);
		    gtk_widget_show(text);
		    
		    gtk_object_set_data(GTK_OBJECT(address_dialog), "street", text);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), text, 96, 8);
		    gtk_widget_set_uposition(text, 104, 8);
		    gtk_widget_set_usize(text, 160, 56);
		    gtk_text_set_editable(GTK_TEXT(text), TRUE);
		    gtk_text_insert(GTK_TEXT(text), NULL, NULL, NULL,
		                    data->address[data->address_type].street,
			            strlen(data->address[data->address_type].street));

		    gtk_widget_grab_focus(text);
		    
		    /* Etykieta "Miejscowo". */
		    label = gtk_label_new(_("City:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 72);
		    gtk_widget_set_uposition(label, 8, 72);
		    gtk_widget_set_usize(label, 88, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Miejscowo". */
		    entry = gtk_entry_new_with_max_length(MAX_CITY);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(address_dialog), "city", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 96, 72);
		    gtk_widget_set_uposition(entry, 104, 72);
		    gtk_widget_set_usize(entry, 160, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->address[data->address_type].city);
		    
		    /* Etykieta "Wojewdztwo". */
		    label = gtk_label_new(_("State/Province:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 104);
		    gtk_widget_set_uposition(label, 8, 104);
		    gtk_widget_set_usize(label, 88, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Wojewdztwo". */
		    entry = gtk_entry_new_with_max_length(MAX_STATE);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(address_dialog), "state", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 96, 104);
		    gtk_widget_set_uposition(entry, 104, 104);
		    gtk_widget_set_usize(entry, 160, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->address[data->address_type].state);
		    
		    /* Etykieta "Kod pocztowy:". */
		    label = gtk_label_new(_("ZIP Code:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 136);
		    gtk_widget_set_uposition(label, 8, 136);
		    gtk_widget_set_usize(label, 88, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole edycji "Kod pocztowy". */
		    entry = gtk_entry_new_with_max_length(MAX_ZIP);
		    gtk_widget_show(entry);
		    
		    gtk_object_set_data(GTK_OBJECT(address_dialog), "zip", entry);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), entry, 96, 136);
		    gtk_widget_set_uposition(entry, 104, 136);
		    gtk_widget_set_usize(entry, 80, 24);
		    gtk_entry_set_text(GTK_ENTRY(entry), data->address[data->address_type].zip);
		    
		    /* Etykieta "Kraj". */
		    label = gtk_label_new(_("Country:"));
		    gtk_widget_show(label);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), label, 8, 168);
		    gtk_widget_set_uposition(label, 8, 168);
		    gtk_widget_set_usize(label, 88, 24);
		    gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
		    
		    /* Pole wyboru "Kraj". */
		    combo = gtk_combo_new();
		    gtk_widget_show(combo);
		    
		    gtk_fixed_put(GTK_FIXED(fixed), combo, 104, 168);
		    gtk_widget_set_uposition(combo, 104, 168);
		    gtk_widget_set_usize(combo, 160, 24);
		    
		        /* Ustalenie listy pola wyboru. */
		        items = country_list();
		        gtk_combo_set_popdown_strings(GTK_COMBO(combo), items);
		        g_list_free(items);
			
			/* Pole edycji pola wyboru "Prefix". */
			entry = GTK_COMBO(combo)->entry;
			gtk_widget_show(entry);
			
			gtk_object_set_data(GTK_OBJECT(address_dialog), "country", entry);
			
			gtk_entry_set_max_length(GTK_ENTRY(entry), MAX_COUNTRY);
			gtk_entry_set_text(GTK_ENTRY(entry),
			                   data->address[data->address_type].country);

        /* Obszar przyciskw okienka dialogowego. */
	dialog_action_area = GTK_DIALOG(address_dialog)->action_area;
	gtk_widget_show(dialog_action_area);
	
	gtk_container_set_border_width(GTK_CONTAINER(dialog_action_area), 10);
	
	    /* Listwa przyciskw. */
	    hbuttonbox = gtk_hbutton_box_new();
	    gtk_widget_show(hbuttonbox);
	    
	    gtk_box_pack_start(GTK_BOX(dialog_action_area), hbuttonbox, TRUE, TRUE, 0);
	    
	    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbuttonbox), GTK_BUTTONBOX_END);
	    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbuttonbox), 10);
	    
	        /* Przycisk "OK". */
		button = gtk_button_new_with_label(_("OK"));
		gtk_widget_show(button);
		
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
		                   GTK_SIGNAL_FUNC(ok_button_clicked), data);
		
		gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
		
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
		
		gtk_widget_grab_default(button);
		
		/* Przycisk "Anuluj". */
		button = gtk_button_new_with_label(_("Cancel"));
		gtk_widget_show(button);
		
		gtk_signal_connect(GTK_OBJECT(button), "clicked",
		                   GTK_SIGNAL_FUNC(dialog_delete), address_dialog);
		
		gtk_container_add(GTK_CONTAINER(hbuttonbox), button);
		
		GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    
    /* Wywietlanie okienka dialogowego. */
    gtk_widget_show(address_dialog);
    
    /* Ustalenie modalnoci okienka. */
    gtk_window_set_modal(GTK_WINDOW(address_dialog), TRUE);
    gtk_window_set_transient_for(GTK_WINDOW(address_dialog), GTK_WINDOW(data->contact_window));
}
