/****h* ROBODoc/Folds
 * FUNCTION
 *   Folds make it possible to divide a large document in smaller
 *   sections.  The divisions can be indicated by the user with the
 *   aid of fold markers.
 * AUTHOR
 *   PetteriK
 *********
 * $Id: folds.c,v 1.14 2003/12/30 17:39:36 gumpu Exp $
 */

#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "robodoc.h"
#include "folds.h"
#include "globals.h"
#include "generator.h"
#include "util.h"


#ifdef DMALLOC
#include <dmalloc.h>
#endif


/* Local functions */

static char         RB_Check_Fold_End( char *cur_char );
static int          RB_Check_Fold_Start( char *cur_char );

long                extra_flags;

#define MAX_FOLD_DEPTH 100
#define MAX_FOLD_NAME 4096

char                foldname[MAX_FOLD_NAME];

/****v* Folds/fold_document_name_stack
 * FUNCTION
 *   Keeps track all the filenames of the nested folds.
 *******
 */

char               *fold_document_name_stack[MAX_FOLD_DEPTH];

/****v* Folds/fold_start_markers
 * NAME
 *   fold_start_markers
 * FUNCTION
 *   Strings for fold start.
 * SOURCE
 */

fold_mark_t         fold_start_markers[] = {
    {"/*{{{", "*/"},            /* C, C++ */
    {"--{{{", "\n"},            /* Occam, line ends with newline */
    {"#{{{", "\n"},             /* Various scripts, line ends with newline */
    {NULL, NULL}
};

/****/


/****v* Folds/fold_end_markers
 * NAME
 *   fold_start_end
 * FUNCTION
 *   Strings for fold end.
 * SOURCE
 */

fold_mark_t         fold_end_markers[] = {
    {"/*}}}", "*/"},
    {"--}}}", "\n"},
    {"#}}}", "\n"},
    {NULL, NULL}
};

/****/

/****v* Folds/fold
* NAME
*   fold
* FUNCTION
*   Fold counter - true global.
* SOURCE
*/

int                 fold = 0;

/****/

/****v* Folds/inside_fold
* NAME
*   in_fold
* FUNCTION
*   Boolean fold indicator. True if processing inside a fold (or nested folds).
* SOURCE
*/

static char         inside_fold = 0;

/****/


int                 fold_id = 0;

/****f* Folds/RB_Fold
 * FUNCTION
 *   Test if we are entering or leaving a fold.
 *****
 * TODO Documentation
 */

char               *
RB_Fold( char *line, char **doc_name, FILE ** dest_doc, char *extension,
         char *charset )
{
    if ( course_of_action & DO_FOLDS )
    {
        if ( RB_Check_Fold_Start( line ) )
        {
            char                number_string[10];
            int                 i;
            int                 j;
            size_t              l;
            char               *name = *doc_name;
            char               *new_doc_name;
            FILE               *new_file;
            FILE               *old_file = *dest_doc;

            /* Save old values */
            fold_document_name_stack[fold] = name;

            /* Create a filename for the fold */
            i = strlen( name );
            for ( ; ( i >= 0 ) && ( name[i] != '/' ); --i );
            assert( i >= 0 );
            l = i;
            l += 6 + 5;         /* enough room for fold_%d */
            l += RB_Get_Len_Extension( extension );     /* and the extension */
            new_doc_name = ( char * ) malloc( l );
            strcpy( new_doc_name, name );
            new_doc_name[i + 1] = '\0';
            sprintf( number_string, "fold__%d", fold_id );
            strcat( new_doc_name, number_string );
            RB_Add_Extension( extension, new_doc_name );

            /* open the new file */
            printf( "opening %s\n", new_doc_name );
            new_file = fopen( new_doc_name, "w" );
            if ( !new_file )
            {
                RB_Panic( "can't open %s!", new_doc_name );
            }

            /* override old values */
            *doc_name = new_doc_name;
            *dest_doc = new_file;

            /* now create a link */
            fprintf( old_file, "    " );
            for ( j = 0; line[j] == ' '; ++j )
            {
                fputc( ' ', old_file );
            }
            RB_Generate_Link( old_file, name, new_doc_name, foldname,
                              number_string );
            fputc( '\n', old_file );
            printf( "closing %s\n", name );
            fclose( old_file );

            /* and the doc start */
            RB_Generate_Doc_Start( new_file, "dummy", new_doc_name, 0,
                                   new_doc_name, charset );
            RB_Generate_Item_Begin( new_file );
            ++fold_id;
            ++fold;
            return foldname;
        }
        else if ( RB_Check_Fold_End( line ) )
        {
            --fold;
            RB_Generate_Item_End( *dest_doc );
            RB_Generate_Doc_End( *dest_doc, *doc_name );
            printf( "closing %s\n", *doc_name );
            free( *doc_name );
            fclose( *dest_doc );
            *doc_name = fold_document_name_stack[fold];
            printf( "opening %s\n", *doc_name );
            *dest_doc = fopen( *doc_name, "a" );
            if ( !( *dest_doc ) )
            {
                RB_Panic( "can't open %s!", doc_name );
            }
            return foldname;
        }
        else
        {
            return NULL;
        }
    }
    return NULL;
}

/****f* Folds/RB_Check_Fold_End [3.0h]
* NAME
*  RB_Check_Fold_End
* AUTHOR
*  PetteriK
* FUNCTION
*  See if a fold end is found.
* RETURN VALUE
*   1 if end mark is found
* SOURCE
*/

static char
RB_Check_Fold_End( char *cur_char )
{
    fold_mark_t        *t = fold_end_markers;
    char                found = 0;

    cur_char = RB_Skip_Whitespace( cur_char );
    while ( found == 0 && t->start != NULL )
    {
        if ( strncmp( t->start, cur_char, strlen( t->start ) ) == 0 )
        {
            found = 1;
            break;
        }
        t++;                    /* try the next fold mark string */
    }
    return found;
}

/*******/

/****f* Folds/RB_Check_Fold_Start
 * NAME
 *   RB_Check_Fold_Start
 * AUTHOR
 *   PetteriK
 * FUNCTION
 *   Check if a fold start is found.
 * RETURN VALUE
 *   1 fold mark found
 *   0 fold mark not found
 * SIDE EFFECTS
 *   Fold name is copied to *foldname.
 * SOURCE
 */

static int
RB_Check_Fold_Start( char *cur_char )
{
    int                 n = 0;
    fold_mark_t        *t = fold_start_markers;
    int                 found = 0;

    cur_char = RB_Skip_Whitespace( cur_char );
    /*{{{ check for fold start string (defined in fold_start_markers) */
    while ( found == 0 && t->start != NULL )
    {
        if ( strncmp( t->start, cur_char, strlen( t->start ) ) == 0 )
        {
            found = 1;
            break;
        }
        t++;                    /* try the next fold mark string */
    }
    /*}}} */
    if ( found == 0 )
    {
        return found;           /* not found, get out of here */
    }
    cur_char += strlen( t->start );     /* skip fold mark */
    /*{{{ get the fold name */
    while ( strncmp( t->end, cur_char, strlen( t->end ) ) != 0 )
    {
        foldname[n++] = *cur_char++;
    }
    /*}}} */
    /*{{{ if fold mark does not end with newline, skip chars...  */
    if ( t->end[0] != '\n' )
    {
        cur_char += strlen( t->end );
    }
    /*}}} */
    foldname[n] = '\0';         /* terminate fold name string */
    /*{{{{ skip chars until newline, not so sure about this */
    while ( *cur_char != '\n' )
    {
        cur_char++;
    }
    /*}}} */
    return found;
}

/*******/

/****f* Folds/RB_Inside_Fold
 * NAME
 *   RB_Inside_Fold
 * AUTHOR
 *   PetteriK
 * FUNCTION
 *   Check if processing currently inside a fold.
 * RETURN VALUE
 *   1 if inside, otherwise 0
 * SOURCE
 */
char
RB_Inside_Fold( void )
{
    return inside_fold;
}

/*******/

/****f* Folds/RB_Enter_Fold
 * NAME
 *   RB_Enter_Fold
 * SOURCE
 */
void
RB_Enter_Fold( void )
{
    inside_fold = 1;
}

/*******/

/****f* Folds/RB_Exit_Fold
 * NAME
 *   RB_Exit_Fold
 * SOURCE
 */
void
RB_Exit_Fold( void )
{
    inside_fold = 0;
}

/*******/
