# Code for the GUI interface to RoutePlanner
#
# Copyright (C) 1996-2001 Chris Lawrence
# This file may be freely distributed under the terms of the RoutePlanner
# license.  A copy should appear as 'LICENSE' in the archive that this
# file was included in.

import string
from gtk import *
from gnome.ui import *

class CityList:
    def __init__(self, wtree):
        self.wtree = wtree
        self.citywin = wtree.get_widget('choosecity')
        self.citylist = wtree.get_widget('citychooser')
        self.cityentry = wtree.get_widget('cityentry')
        self.showallbutton = wtree.get_widget('showall')
        self.visible = []

        dict = {}
        for key in dir(self.__class__):
            dict[key] = getattr(self, key)
        self.wtree.signal_autoconnect(dict)

    def startup(self, city='', db=None, setfun=None, setdata=None):
        self.db = db
        self.cities = map(str, db.cities)

        # Update the CList
        self.update_citylist()
        if city and city in self.cities:
            cnum = self.cities.index(city)
            self.citylist.select_row(cnum, 0)
            self.selected = city
            self.selrow = cnum
        else:
            self.selected = self.selrow = None
            self.cityentry.set_text('')
        
        self.citywin.connect("delete_event", self.on_city_cancel_clicked)
        
        if setfun:
            self.callwhendone = (setfun, setdata)
        else:
            self.callwhendone = None

        self.cityentry.grab_focus()
        self.citywin.show()

    def on_citychooser_select_row(self, *args):
        pos = self.visible[args[1]]
        self.selected = self.cities[pos]
        self.selrow = pos
        self.cityentry.set_text(self.selected)

    def on_citychooser_unselect_row(self, *args):
        self.selected = self.selrow = None
        self.cityentry.set_text('')

    def on_citychooser_button_press_event(self, button, event, *args):
        if event.type == GDK._2BUTTON_PRESS and event.button == 1 and \
           self.citylist.selection:
            self.on_city_ok_clicked()

    def on_cityentry_activate(self, *args):
        text = self.cityentry.get_text()
        matching = self.db.CitiesMatchIndex(text, self.visible)
        if not matching:
            dialog = GnomeWarningDialog('No matching cities', self.citywin)
            dialog.run_and_close()
            return

        if len(matching) == 1:
            match, lmatch = matching[0], self.visible.index(matching[0])
            self.citylist.select_row(lmatch, 0)
            self.citylist.moveto(lmatch)
            if string.lower(text) == string.lower(self.cities[match]):
                self.on_city_ok_clicked()
            return

        self.update_citylist(tuple(matching))

    def on_city_ok_clicked(self, *args):
        if not self.selected:
            self.on_cityentry_activate()
            return

        self.citywin.hide()
        self.citylist.clear()
        if self.callwhendone:
            setfun, setdata = self.callwhendone
            setfun(self.selected, self.selrow, setdata)

    def on_city_cancel_clicked(self, *args):
        self.citywin.hide()
        self.citylist.clear()
        return TRUE

    def on_city_showall_clicked(self, *args):
        self.update_citylist()

    def update_citylist(self, selected=()):
        if not selected:
            selected = range(len(self.cities))

        self.citylist.freeze()
        self.citylist.clear()

        for i in range(len(selected)):
            sel = selected[i]
            self.citylist.append([self.cities[sel]])
            
        self.visible = list(selected)
        self.citylist.thaw()
        self.showallbutton.set_sensitive(len(selected) != len(self.cities))
