/*-----------------------------------------------------------------*-C-*---
 * File:    modules/regex/runmatch.c
 *
 *          Copyright (C)1997 Donovan Kolbly <d.kolbly@rscheme.org>
 *          as part of the RScheme project, licensed for free use.
 *          See <http://www.rscheme.org/> for the latest information.
 *
 * File version:     1.7
 * File mod date:    1997-11-29 23:10:51
 * System build:     v0.7.3.3-b20u, 2004-04-02
 * Owned by module:  regex
 *
 * Purpose:          regex interpreter engine
 *------------------------------------------------------------------------*/

#include "regex_p.h"

UINT_8 **rxmach_save_array;
UINT_8 *rxmach_machine;
UINT_8 *rxmach_start, *rxmach_limit;
UINT_8 *rxmach_on_eos;

void rxmach_internal_error( int code )
{
  scheme_error( "regex-interp: internal error code ~d", 1, int2fx(code) );
}

static UINT_8 *max_range( UINT_8 *str, UINT_8 *limit,
			  UINT_8 *sub_machine );


/*
 *  try to find a match
 *  return NULL if failed
 *  otherwise, return pointer to the place where the match ends
 */

#define FAILED      return NULL
#define EOS         return rxmach_on_eos

#define save_array rxmach_save_array
#define machine rxmach_machine
#define limit rxmach_limit

UINT_8 *run_match( UINT_8 *str, UINT_32 pc )
{
  UINT_8 *min;

  while (1)
    {
      switch (machine[pc++])
	{
	case RXM_MATCH1:
	  if (str < limit)
	    {
	      if (*str++ != machine[pc++])
		FAILED;
	    }
	  else
	    EOS;
	  break;

	case RXM_MATCH_ANY:
	  if (str == limit)
	    EOS;
	  str++;
	  break;

	case RXM_STR_END:
	  if (str != limit)
	    FAILED;
	  break;

	case RXM_STR_START:
	  if (str != rxmach_start)
	    FAILED;
	  break;

	case RXM_MATCH_SET:
	  {
	    UINT_8 ch;

	    if (str == limit)
	      EOS;
	    
	    ch = *str++;
	    if (!(machine[pc+ch/8] & (0x80 >> (ch%8))))
	      FAILED;
	    pc += 32;
	  }
	  break;
	  
	case RXM_MATCHN:
	  {
	    UINT_8 i, n = machine[pc++];

	    while (n > 0)
	      {
		if (str >= limit)
		  EOS;
		else if (*str != machine[pc++])
		  FAILED;
		str++;
		n--;
	      }
	  }
	  break;

	case RXM_ACCEPT:
	  return str;

	case RXM_BRANCH:
	  {
	    UINT_32 subr;
	    UINT_8 *result;

	    subr = (machine[pc] << 8) + machine[pc+1];
	    pc += 2;

	    result = run_match( str, subr );
	    if (result)
	      return result;
	   
	    break;
	  }

	case RXM_SAVE_PLACE:
	  {
	    UINT_8 *p, reg = machine[pc++];

	    p = run_match( str, pc );
	    if (p)
	      {
		save_array[reg] = str;
		return p;
	      }
	    else
	      FAILED;
	  }

	case RXM_MATCH_PLUS:
	  min = str + 1;
	  goto match_kleene;

	case RXM_MATCH_STAR:
	  min = str;
	match_kleene:
	  {
	    UINT_8 *range = max_range( str, 
				       limit,
				       machine 
				       + (machine[pc] << 8)
				       + machine[pc+1] );
	    UINT_8 *result;
	    pc += 2;

	    /* If we can eat everything, then this is
	     * a valid prefix
	     * (although the remainder of the pattern
	     * might mean we are not a valid match)
	     */
	    if ((range == limit) && rxmach_on_eos)
	      EOS;

	    while (range >= min)
	      {
		result = run_match( range, pc );
		if (result)
		  return result;
		range--;
	      }
	    FAILED;
	  }

	case RXM_REJECT:
	  FAILED;

	case RXM_JUMP:
	  pc = (machine[pc] << 8) + machine[pc+1];
	  break;

	default:
	  rxmach_internal_error( RXERR_INT_BAD_OPCODE );
	}
    }
} 


static UINT_8 *max_range( UINT_8 *str, UINT_8 *limit,
			  UINT_8 *sub_machine )
{
  switch (sub_machine[0])
    {
    case RXM_MATCH_SET:
      {
	UINT_8 ch, *vec = sub_machine + 1;

	while (str < limit)
	  {
	    ch = *str;
	    if (!(vec[ch/8] & (0x80 >> (ch%8))))
	      return str;
	    str++;
	  }
	return limit;
      }
    case RXM_MATCH_ANY:
      return limit;
    case RXM_MATCH1:
      {
	UINT_8 ch = sub_machine[1];
	while (str < limit)
	  {
	    if (*str != ch)
	      return str;
	    str++;
	  }
	return limit;
      }
    default:
      rxmach_internal_error(RXERR_INT_BAD_KLEENE);
      return NULL; /* quiet compiler */
    }
}
