#ifndef RUMBA_DESIGN_H
#define RUMBA_DESIGN_H

#include <iosfwd>
#include <sstream>
#include <string>
#include <algorithm>
#include <vector>
#include <rumba/manifoldmatrix.h>

struct Factor;
class Treatment;


/**
  * alternate version of std::find
  */
template <class T> int find_index ( const std::vector<T> & v, T key );

/**
  * Break a token delimited line into separate strings, and store in "result".
  */
void tokenizeCsvLine
( std::vector<std::string>& result, const std::string& line, char delim = ',' );

std::ostream& operator<< (std::ostream& left, const Factor& right);
std::ostream& operator<< (std::ostream& left, const Treatment& right);

void process_header
(
 std::vector<Factor>& result, 
 std::string s, 
 const std::vector<std::string>& headers
 );

/** get a row of an experiment  log file (list of levels in the different 
 * factors) and update levels in factors appropriately.
 */
void process_row ( std::vector<Factor>& result, std::string row);

int treatment_to_column 
( const std::vector<Factor>& factors, const std::vector<std::string>& treatment);






RUMBA::ManifoldMatrix design_matrix 
(
 std::ifstream& fin, std::vector<Factor>& factors
 );

RUMBA::ManifoldMatrix design_matrix2
(
 const std::vector<Treatment>&, std::vector<Factor>& factors
 );

std::vector<Treatment> readfile 
( std::ifstream& fin, vector<Factor>& factors  );


struct Factor
{
	Factor() : name(""),col_num(-1),levels() {}
	Factor(std::string x, int y):name(x),col_num(y){}
	std::string name;
	int col_num;
	std::vector<std::string> levels;
};


/**
  * a primitive treatment is one where all factors are at level 0 except 1.
  *	Hence a primitive treatment is uniquely determined by a factor number
  * and a (non-zero) level for that factor.
  */
int primitiveToColumn(int level, int factor_no, const std::vector<Factor>& factors);


class Treatment 
{
public:
	Treatment(const std::vector<Factor>& F):ptrF(&F){}
	Treatment(const std::vector<Factor>& F, const std::vector<std::string>& v );
	Treatment() : ptrF(0){}
	// is treatment of the form Ai B0 C0 D0 ... 
	bool isPrimitive () const;
	// are all components 0 ?
	bool isZero () const; 

	int level(int i) const
	{
		return T[i];
	}

	std::string stringLevel(int i) const
	{
		return (*ptrF)[i].levels[T[i]];
	}

	int length() const { return T.size(); }

	// get corresponding column from design matrix
	int toColumn() const;



private:
	const std::vector<Factor>* ptrF;
	std::vector<int> T;
};

bool operator<(const Treatment& left, const Treatment& right);
bool operator==(const Treatment& left, const Treatment& right);

template <class TYPE>
int find_index ( const std::vector<TYPE> & v, TYPE key )
{
	int x =  std::find ( v.begin(), v.end(), key ) - v.begin() ;
	return x < v.size() ? x : -1;
}

Treatment column_to_treatment ( int col, const std::vector<Factor> & factors);

RUMBA::ManifoldMatrix 
getDesignMatrixRow ( Treatment t, const std::vector<Factor>& factors );

int count_treatments ( const std::vector<Factor>& factors );

RUMBA::ManifoldMatrix repeatRows(const RUMBA::ManifoldMatrix&, int );


#endif
