#include <list>
#include <fstream>
#include <vector>
#include <string>
#include <algorithm>

#include <rumba/parse.h>
#include <rumba/manifold.h>
#include <rumba/arghandler.h>

using namespace RUMBA;
using std::string;
using std::vector;
using std::list;
using std::ifstream;
using std::getline;

string usage()
{
	return 
	"gen_tmask --infile|-i file1 --outfile|-o file2 [--header-row n]\n"
	"--column-name|-c  name\n"
	"--values val1 val2 val3 ... \n";
}

bool testRow ( const string& line, const string& key, int col )
{
	list<string> tokens_ = tokenizeCsvLine(line,'\t');
	vector<string> tokens(tokens_.begin(),tokens_.end());
	assert (col<tokens.size()&&col>=0);
	return ( tokens[col] == key );
}

string cue(ifstream& fin, int header_row)
{
	string s;
	fin.clear();
	fin.seekg(0,std::ios::beg);
	int count = 1;
	do {
		getline(fin,s);
		++count;	
	}
	while (count < header_row);
	return s;
}

int getColumnByName(const std::string& line, const string & column_header)
{
	vector<string>::const_iterator result;
	list<string> tokens_ = tokenizeCsvLine(line,'\t');
	vector<string> tokens(tokens_.begin(),tokens_.end());
	result = std::find ( tokens.begin(), tokens.end(), column_header );
	return result - tokens.begin();	
	

}

Argument myArgs [] =
{
	Argument("header-row", RUMBA::NUMERIC, '\0', 1, false ),
	Argument("column-name",	RUMBA::ALPHA, 'c', "", true),
	Argument("values", RUMBA::ALPHA, 'v', "", true, true ),
	Argument()
};

int main(int argc,char** argv)
{
	string headline;
	ifstream fin;
	int header_row;
	int colnum;	
	int t =0;
	string colname;
	string current_line;
	string infile;
	string outfile;
	vector<Splodge> values;
	vector<char> result;

	ArgHandler::setRequiredDefaultArg("infile");
	ArgHandler::setRequiredDefaultArg("outfile");


	try	
	{	
		ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			std::cerr <<usage()<< std::endl;
			return 0;
		}
		argh.arg("header-row",header_row);
		argh.arg("column-name",colname);
		argh.arg("outfile",outfile);
		argh.arg("infile",infile);
		values = argh.multiarg("values");
		if (!argh.loose().empty())
			throw RUMBA::Exception(
				string("Invalid argument")
			);

		fin.open(infile.c_str());
		if (!fin)
		{
			throw RUMBA::Exception(string("Couldn't open ") + infile);
		}

		headline = cue (fin,header_row);	
		colnum = getColumnByName(headline,colname);

		while ( getline(fin,current_line))
		{
			t = 0;
			for ( int i =0; i < values.size(); ++i )
				if ( testRow(current_line, values[i].asString(), colnum )) 
					t = 1;
			result.push_back(t);
		}

		Manifold<char> M ( intPoint(1,1,1, result.size() ) );
		std::copy ( result.begin(), result.end(), M.begin() );
		M.save(outfile.c_str());

	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
		std::cerr<<usage()<<std::endl;
		exit(1);
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
		std::cerr<<usage()<<std::endl;
		exit(1);
	}
	catch (RUMBA::MissingArgumentException& s)
	{
		std::cerr << "Missing argument: " << s.error() << std::endl;
		std::cerr << usage()<<std::endl;
		exit(1);
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl; 
		std::cerr<<usage()<<std::endl;
		exit(1);
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
		std::cerr<<usage()<<std::endl;
		exit(1);
	}

	

}
