#include <rumba/manifold.h>
#include <rumba/manifoldFile.h>
#include <rumba/arghandler.h>
#include <rumba/parse.h>
#include <rumba/util.hpp>
#include <iostream>
#include <cstdlib>
#include <vector>
#include <fstream>

#include "../matrixutils/stats.h"

using std::cerr;
using std::endl;
using RUMBA::stream_cast;
using RUMBA::intPoint;

void error()
{
	cerr  << "Usage: manifoldstat -i infile x y z t dx dy dz dt" << endl;
}

int main(int argc, char** argv)
{
//	RUMBA::ArgHandler::setRequiredDefaultArg("infile");

	try 
	{
		RUMBA::ArgHandler argh(argc,argv);
		std::ofstream fout;
		std::ostream* outstream=0;
		std::string infile,outfile;
		intPoint origin,dims;
		RUMBA::BaseManifold* N;
		bool verbose;
		bool debug = true;
		if (argh.arg("help"))
		{
			error();
			exit(0);
		}

		argh.arg_or_loose("infile", infile );

		if(argh.arg("outfile"))
			argh.arg("outfile",outfile );
		std::vector<string> v = argh.loose();


		if ( !v.empty() && v.size () < 8 )
		{
			error();
			exit(EXIT_FAILURE);
		}

		if (!v.empty()) try 
		{
			dims.x() = stream_cast<int> ( v[4] );
			dims.y() = stream_cast<int> ( v[5] );
			dims.z() = stream_cast<int> ( v[6] );
			dims.t() = stream_cast<int> ( v[7] );
			origin.x() = stream_cast<int> ( v[0] );
			origin.y() = stream_cast<int> ( v[1] );
			origin.z() = stream_cast<int> ( v[2] );
			origin.t() = stream_cast<int> ( v[3] );
		}
		catch ( RUMBA::Exception & e )
		{
			std::cerr << "Numeric argument expected" << std::endl;
			if (debug)
				std::cerr << "Debug: " << e.error() << endl;
		}



		RUMBA::ManifoldFile* M = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!M)
			throw RUMBA::Exception ( 
					std::string("Couldn't open input file ") + infile
					);

		if (v.empty())
			N = M->get(intPoint(0,0,0,0),M->extent());
		else 
			N = M->get ( origin, dims );
		RUMBA::StatFunctor f(N);

		if(!outfile.empty())
		{
			fout.open(outfile.c_str());
			if (!fout)
			{
				throw RUMBA::BadFile(std::string("Couldn't open file ")+outfile);
			}
			outstream = &fout;
		}
		else
		{
			outstream = &std::cout;
		}

		*outstream << "Dimensions: " 
			<< M->width() << "x" << M->height() << "x"
			<< M->depth() << "x" << M->timepoints() << std::endl;

		*outstream << "Datatype: " << M->headerData()["normalized_datatype"]
			<< std::endl;
	
		*outstream << "mean: " << f.mean() << std::endl;
		*outstream << "standard deviation: " << f.sd() << std::endl;
		*outstream << "variance: " << f.variance() << std::endl;
		*outstream << "max: " << f.max() << std::endl;
		*outstream << "min: " << f.min() << std::endl;

		delete M;
		delete N;

	
	}
    catch (RUMBA::MissingArgumentException& s)
    {
		cerr << "Error, missing required argument: " << s.error() << endl;
		error();
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		cerr << "Duplicate argument: " << s.error() << endl;
		error();
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		cerr << "Error: " << s.error() << endl;
		error();
	}                   
	catch ( RUMBA::Exception & e)
	{
		cerr << "Fatal exception " << e.error() << endl;	
	}

	return 0;
}
