#include <iostream>
#include <numeric>
#include <algorithm>
#include <complex>
#include <vector>
#include <functional>

#include <rumba/manifoldFile.h>
#include <rumba/manifold.h>
#include <rumba/fft.h>
#include <rumba/arghandler.h>

using namespace std;

template<class TYPE>
void interpolate (std::vector<TYPE> & v, int start_index, int end_index )
{
	double ratio = (end_index - start_index) / (double) v.size();
	double tmp;
	for (int i = v.size()-1; i >= 0; --i )
	{
		tmp = ratio * i;
		v[i] = (tmp - (int)tmp) * (v[(int)(tmp+1)] - v[(int)tmp] )+v[(int)tmp];
	}
}

int next_power_of_2(int x)
{
	int i = 1;
	while (x>i)
		i*=2;
	return i;
}

template <class iter>
void mean_normalize(iter start, iter end)
{
	typedef std::iterator_traits<iter>::value_type value;
	value mean = std::accumulate(start,end, value() ) / (end-start);
	std::transform(start,end,start,std::bind2nd(std::minus<value>(),mean));
}

double abs_and_incr( complex<double>& x, double& y)
{
	return abs(x)*abs(x)+y;
}


void usage()
{
	std::cerr << "powerspectrum -i infile -o outfile --mask maskfile [--zeropad|--subset|--interpolate]" << std::endl;
}

RUMBA::Argument myArgs[] = 
{
 RUMBA::Argument("mask",RUMBA::ALPHA,'m'),
 RUMBA::Argument("zeropad",RUMBA::FLAG,'z'),
 RUMBA::Argument("subset",RUMBA::FLAG,'s'),
 RUMBA::Argument("interpolate",RUMBA::FLAG,'\0'),
 RUMBA::Argument()
};


int main(int argc, char** argv)
{
	std::vector<double> spectrum;
	std::string infile;
	std::string maskfile;
	bool zeropad = true;
	bool subset = false;
	int skipt,skipspace;
	int offset;
	int bufsize;
	RUMBA::ManifoldFile* M = 0;
	RUMBA::Manifold<char>  mask;
	std::vector<complex<double> > buf;

	try
	{

		RUMBA::ArgHandler argh(argc,argv,myArgs);

		zeropad=argh.arg("zeropad");
		if (!zeropad)
			subset = argh.arg("subset");

		if (argh.arg("interpolate"))
		{
			zeropad=subset=false;
		}

		if(argh.arg("help"))
		{
			usage();
			exit(0);
		}

		if (argh.arg("infile"))
			argh.arg("infile", infile);
		else
		{
			std::cerr << "Must supply an input file" << std::endl;
			exit(1);
		}

		M = RUMBA::ManifoldFile::construct(infile.c_str());

		if (!M)
		{
			std::cerr << "Couldn't open input file " << infile.c_str() 
				<< std::endl;
			exit(1);
		}

		if (argh.arg("mask"))
		{
			argh.arg("mask",maskfile);
			mask.load(maskfile.c_str());
		}
		else
		{
			mask = RUMBA::Manifold<char>
				(
				 RUMBA::intPoint(M->width(),M->height(),M->depth(),1)
				);
			std::fill(mask.begin(),mask.end(),1);
		}

		if (mask.pixels()!=M->pixels())
		{
			std::cerr << "Mask dimensions incompatible with input file" 
				<< std::endl;
			delete M;
			exit(1);
		}


		skipt = M->skip().t();
	//	skipspace = M->orientation().Xyzt() ? 1 : M->timepoints();
		skipspace = 1;
		M->setCacheStrategy(RUMBA::CACHE_TIME);

		bufsize = (next_power_of_2(M->timepoints()));
		if (subset)
			bufsize /= 2;
		buf.resize(bufsize);
		spectrum.resize(buf.size());

		for ( int i = 0; i < M->pixels(); ++i )
		{
			offset=i*skipspace;

			if (mask[i])
			{
				std::fill(buf.begin(),buf.end(),0);

				if (subset)
				{
					for (int j = 0; j < bufsize; ++j )
						buf[j] = M->getElementDouble(offset+j*skipt);
				}
				else
				{
					for (int j = 0; j < M->timepoints(); ++j )
						buf[j] = M->getElementDouble(offset+j*skipt);
					mean_normalize(buf.begin(),buf.begin()+M->timepoints());
				}


				if (zeropad)
					std::fill(buf.begin()+M->timepoints(),buf.end(),0.0);
				else 
					interpolate(buf,0,M->timepoints());
				
				RUMBA::generic_fft(buf,buf.size());

				transform(buf.begin(),buf.end(), spectrum.begin(),
					spectrum.begin(), abs_and_incr );

			}

		}

		transform(spectrum.begin(),spectrum.end(),spectrum.begin(),
				bind2nd(divides<double>(), std::count(mask.begin(),mask.end(),1)));

		delete M;

		// we want spectral density.
		std::transform(spectrum.begin(),spectrum.end(),spectrum.begin(),
			std::bind2nd(std::divides<double>(), 
				std::accumulate(spectrum.begin(),spectrum.end(),0.0)
			)
		);

		std::copy(spectrum.begin(),spectrum.end(),
			std::ostream_iterator<double>(std::cout,"\n" ));

	}
	catch (RUMBA::InvalidArgumentException& e)
	{
		std::cerr << "Invalid argument: " << e.error() << std::endl;

	}
	catch(RUMBA::Exception& e)
	{
		std::cerr << "Exception: " << e.error() << std::endl;
	}
	catch(std::exception& e)
	{
		std::cerr << "Fatal: " << e.what() << std::endl;
	}
}
