#include <iostream>
#include <vector>
#include <list>
#include <algorithm>
#include <map>

#include <rumba/arghandler.h>
#include <rumba/exception.h>
#include <rumba/parse.h>
#include <rumba/loadVols.h>
#include <rumba/manifoldmatrix.h>
#include <rumba/matrixio.h>
#include <rumba/matrix.hpp>
#include <rumba/util.hpp>

//template class std::map<std::string, RUMBA::Factory*>;


//RUMBA::Manifold<double> permute_fill(const std::vector<double> &, bool transpose = false);

using namespace RUMBA;

using RUMBA::Splodge;
using RUMBA::ManifoldFile;
using RUMBA::Argument;
using std::cerr;
using std::endl;

void usage()
{
	std::cerr << "Usage: subsample -i infile -o outfile --dimensions x y z t\n";
}

RUMBA::intPoint chk_list(const std::vector<Splodge>& v)
{
	RUMBA::intPoint p;

	if (v.size() != 4) 
		throw RUMBA::Exception("Dimensions must be in form x y z t");

	for ( std::vector<Splodge>::size_type i = 0; i < 4; ++i )
		p[i]=v[i].asInt();

	return p;

}


Argument myArgs [] = {
	Argument("dimensions", RUMBA::NUMERIC, 'd', -1, true, true),
	Argument()
};


int main(int argc,char** argv)
{
	std::string infile , outfile;
	RUMBA::intPoint dims;
	RUMBA::ManifoldFile *fin, *fout;
	RUMBA::intPoint ext;

	try {
		ArgHandler argh ( argc , argv, myArgs );
		if ( argh.arg("help") )
		{
			usage();
			exit(0);
		}
		argh.arg ( "infile" , infile );
		argh.arg ( "outfile" , outfile );
		dims = chk_list(argh.multiarg ("dimensions"));
		fin = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!fin)
		{
			throw RUMBA::Exception(std::string("Couldn't open file ")+infile);
		}
		
		for ( int i = 0; i < 4; ++i )
		{
			ext[i] = fin->extent()[i] / dims[i];
			if ( ext[i] <= 0 )
				throw RUMBA::Exception("Dimensions ill-formed");
		}

		fout = RUMBA::ManifoldFile::construct(
				outfile.c_str(), 
				fin->headerData()["normalized_datatype"].asString(), ext
				);
		if (!fout)
		{
			throw RUMBA::Exception(std::string("Couldn't open file ")+outfile);
		}

		RUMBA::downsample(dims,fin,fout);
		delete fin;
		delete fout;
	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl;
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
	}

}
