/**
  * \file squash.cc
  * \author Donovan Rebbechi, Ben Bly, Giorgio Grasso
  * Copyright Ben Bly
  * This file is released under the artistic license. See the file
  * COPYING for details.
  */

#include <rumba/arghandler.h>
#include "../matrixutils/mask.h"
#include <iostream>
#include <cstdlib>

using RUMBA::Argument;

void usage()
{
	std::cerr << "usage: unsquash -i infile -m mask -o outfile" << std::endl;
}

Argument myArgs [] = {
	Argument ( "mask", RUMBA::ALPHA, 'm', "", true ),
	Argument()
};


int main(int argc,char** argv)
{
	RUMBA::ArgHandler::setRequiredDefaultArg("infile");
	RUMBA::ArgHandler::setRequiredDefaultArg("outfile");
	RUMBA::intPoint dims;
	RUMBA::ManifoldFile* fin; 
	RUMBA::ManifoldFile* fout; 

	std::string infile,outfile,maskfile;

	try {
		RUMBA::ArgHandler argh ( argc , argv, myArgs );
		if ( argh.arg("help") )
		{
		usage();
		exit(0);
		}
		argh.arg ( "infile" , infile );
		argh.arg ( "outfile" , outfile);
		argh.arg ( "mask" , maskfile);


		fin = RUMBA::ManifoldFile::construct(infile.c_str(),std::ios::in);
		if (!fin)
		{
			std::cerr << "Couldn't open input file " << infile << std::endl;
			exit(2);
		}
	
	
		RUMBA::Manifold<char> maskman(maskfile.c_str());
		dims = unSquashSize(maskman, fin);
	
	
		fout = RUMBA::ManifoldFile::construct(outfile.c_str(),
			fin->headerData()["normalized_datatype"].asString(),dims);
	
		if (!fout)
		{
			std::cerr << "Couldn't open output file " << outfile << std::endl;
			exit(2);
		}

		fin->setCacheStrategy(RUMBA::CACHE_TIME);
		fout->setCacheStrategy(RUMBA::CACHE_TIME);
	
	
		unsquash(maskman,fin,fout);
		delete fin;
		delete fout;
		return 0;
	

	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
	catch (RUMBA::MissingArgumentException& s)
	{
		std::cerr << "Missing argument: " << s.error() << std::endl;
		usage();
		exit(1);
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl; 
		usage();
		exit(1);
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
		usage();
		exit(1);
	}

}

