#include <rumba/arghandler.h>
#include <rumba/baseManifold.h>
#include <rumba/manifoldFile.h>
#include <cmath>

#define EPSILON 1e-8

inline double square(const double& x)
{
	return x*x;
}

double variance(int skip, int tp, RUMBA::BaseManifold::iterator it, double m)
{
	double sum=0;
	
	for(int t=0;t<tp;++t,it+=skip)
		sum += square(static_cast<double>(*it)-m);
	return sum/tp;
}

double sd(int skip, int tp, RUMBA::BaseManifold::iterator it, double m)
{
	double sum=0;
	
	for(int t=0;t<tp;++t,it+=skip)
		sum += square(static_cast<double>(*it)-m);
	return std::sqrt(sum/tp);
}



double mean(int skip, int tp, RUMBA::BaseManifold::iterator it)
{
	double sum=0;

	for(int t=0;t<tp;++t,it+=skip)
		sum += static_cast<double>(*it);
	return sum/tp;
}

void usage()
{
	std::cerr << "variance (--infile|-i) infile (--outfile|o) outfile [--sd]\n"
		"[--divide-by-mean|-d]"
		<< std::endl;

}

RUMBA::Argument myArgs [] = 
{
	RUMBA::Argument("sd",RUMBA::FLAG,'s'),
	RUMBA::Argument("divide-by-mean",RUMBA::FLAG,'d'),
	RUMBA::Argument()
};

int main(int argc, char** argv)
{
	std::string infile,outfile;
	int px,tp;
	RUMBA::ManifoldFile* fin=0;
	RUMBA::BaseManifold* fout=0;
	double m=0;
	bool normalize_mean = false;
	bool use_sd = false;

	try 
	{
		RUMBA::ArgHandler argh(argc,argv,myArgs);
		if (argh.arg("help"))
		{
			usage(); 
			return 0;
		}
		argh.arg("infile",infile);
		argh.arg("outfile",outfile);
		normalize_mean = argh.arg("divide-by-mean");
		use_sd = argh.arg("sd");

		fin = RUMBA::ManifoldFile::construct(infile.c_str());
		if (!fin) 
			throw RUMBA::Exception("Fatal: couldn't open file for input");
		RUMBA::intPoint ext = fin->extent(); 
		ext.t()=1;
		fout = RUMBA::ManifoldFile::construct(outfile.c_str(),"float64", ext );
		if (!fout) 
			throw RUMBA::Exception("Fatal: couldn't open file for input");

		px = fin->pixels();
		tp = fin->timepoints();
		fin->setCacheStrategy(RUMBA::CACHE_TIME);
		
		int i = 0;	

		for (RUMBA::BaseManifold::iterator it = fin->begin();
			it!=fin->begin()+px; ++it, ++i)
		{
			m = mean (px,tp,it);
			if (use_sd)
			{
				fout->setElementDouble (
						i,
						sd( px, tp, it,m ) / (normalize_mean ? m : 1)
					 );
			}
			else
			{
				fout->setElementDouble (
						i, 
						variance( px, tp, it,m ) / (normalize_mean ? m : 1)
					 );
			}

		}
	


		delete fin;
		delete fout;
	}
	catch ( RUMBA::InvalidArgumentException& s)
	{
		std::cerr << "Invalid argument: " << s.error() << std::endl;
	}
    catch (RUMBA::DuplicateArgumentException& s)
    {
		std::cerr << "Duplicate argument: " << s.error() << std::endl;
	}
	catch (RUMBA::ArgHandlerException& s)
	{
		std::cerr << "Error: " << s.error() << std::endl;
	}
	catch (RUMBA::Exception& s)
	{
		std::cerr << "Exception:" << s.error() << std::endl;
	}

}
