/* $Id: ip.c,v 1.4 2003/05/30 00:42:40 cgd Exp $ */

/*
 * Copyright 2001, 2003
 * Broadcom Corporation. All rights reserved.
 *
 * This software is furnished under license and may be used and copied only
 * in accordance with the following terms and conditions.  Subject to these
 * conditions, you may download, copy, install, use, modify and distribute
 * modified or unmodified copies of this software in source and/or binary
 * form. No title or ownership is transferred hereby.
 *
 * 1) Any source code used, modified or distributed must reproduce and
 *    retain this copyright notice and list of conditions as they appear in
 *    the source file.
 *
 * 2) No right is granted to use any trade name, trademark, or logo of
 *    Broadcom Corporation.  The "Broadcom Corporation" name may not be
 *    used to endorse or promote products derived from this software
 *    without the prior written permission of Broadcom Corporation.
 *
 * 3) THIS SOFTWARE IS PROVIDED "AS-IS" AND ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING BUT NOT LIMITED TO, ANY IMPLIED WARRANTIES OF
 *    MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE, OR
 *    NON-INFRINGEMENT ARE DISCLAIMED. IN NO EVENT SHALL BROADCOM BE LIABLE
 *    FOR ANY DAMAGES WHATSOEVER, AND IN PARTICULAR, BROADCOM SHALL NOT BE
 *    LIABLE FOR DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 *    OR OTHERWISE), EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "ip.h"
#include "arp.h"
#include "global.h"
#include "eth.h"


/*
 * Data structures
 */

typedef struct {
	uint8_t version_ihl      ;
	uint8_t tos              ;
	uint8_t total_length[2]  ;
	uint8_t id[2]            ;
	uint8_t flags_frag_ofs[2];
	uint8_t ttl              ;
	uint8_t protocol         ;
	uint8_t checksum[2]      ;
	uint8_t src_ip[4]        ;
	uint8_t dest_ip[4]       ;
} ip_header_t;

typedef struct ip_handler_node_struct{
	int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip);
	void *data;
	uint8_t protocol;
	struct ip_handler_node_struct *next;
} ip_handler_node_t;

/*
 * Static data
 */

static ip_handler_node_t *handler_chain;
static uint16_t ip_id = 1;



/* 
 * Static functions
 */

/* 
 * Check if the packet contained in buf is a valid ip
 * packet.  A packet is considered valid if:
 * 
 *  - it contains at least ip_total_length bytes
 *  - The checksum checks out
 */
static int is_valid_ip_packet(mbuf_t *buf)
{
	int chunks;
	uint32_t checksum = 0;
	int i;

	ip_header_t *header = (ip_header_t *)(buf->front_ptr);


	if ((buf->size < sizeof(ip_header_t))
	    || (buf->size < get_uint16_field(header->total_length))) {
		return 0;
	}
	
	chunks = (header->version_ihl & 0xf)<<1;
	for (i = 0; i < chunks; i++) {
		checksum += get_uint16_field(((uint8_t *)header) + (i << 1));
	}
	
	while (checksum > 0xffff) {
		checksum = (checksum >> 16) + (checksum & 0xffff);
	}

	if (checksum != 0xffff) {
		return 0;
	}
	return 1;
}


/*
 * Handler used to grap IP packets from the ethernet layer
 * 
 * Whenever we have handlers for ip packets registered, this
 * function is registered with the ethernet layer to capture
 * all IP packets.  When called, all link-layer header information
 * has been stripped from the packet.
 *
 * Here we check that the packet is a valid ip packet, and
 * if it is, check for a handler in our chain that wants a 
 * crack at it.  If we find a handler that wants it, and
 * who's function returns non zero, we consider the packet
 * handled.
 *
 * Return 1 if the packet was handled, 0 otherwise
 */
static int ip_handler(mbuf_t *buf, void *ptr, uint16_t ignored) 
{
	int retval = 0;
	ip_header_t *header;

	if (is_valid_ip_packet(buf)) {
		ip_handler_node_t *tmp;
		uint32_t src_ip, dest_ip;
		uint8_t prot;


		header = (ip_header_t *)buf->front_ptr;
		src_ip = get_uint32_field(header->src_ip);
		dest_ip = get_uint32_field(header->dest_ip);
		prot = header->protocol;
		
		mbuf_trim_tail(buf, buf->size - get_uint16_field(header->total_length));
		mbuf_trim_head(buf, (header->version_ihl & 0xf)<<2);
		for (tmp = handler_chain; tmp != 0; tmp = tmp->next) {
			if ((!tmp->protocol || (tmp->protocol == prot))
			    && ((*(tmp->handler))(buf, tmp->data, src_ip, dest_ip))) {
				retval = 1;
				break;
			}
		}
	}
	return retval;
}


/*
 * Take the ip packet in header and calculate/update
 * the checksum 
 */
static void ip_calc_checksum(ip_header_t *header)
{
	uint32_t checksum = 0;
	int chunks;
	int i;

	set_uint16_field(header->checksum, 0);
	chunks = (header->version_ihl & 0xf) << 1;
	for (i = 0; i < chunks; i++) {
		checksum += get_uint16_field(((uint8_t *)header) + (i<<1));
	}
	
	while (checksum > 0xffff) {
		checksum = (checksum >> 16) + (checksum & 0xffff);
	}
	
	set_uint16_field(header->checksum, ~checksum);
}



/*
 * Exported functions.  The semantics of these are described in the header 
 */

void ip_send_packet(mbuf_t *mbuf, uint8_t protocol, uint32_t src_ip, uint32_t dest_ip, uint8_t flags)
{
	uint64_t hw_addr;
	ip_header_t *header;
	if (mbuf_prepend_space(mbuf, sizeof(ip_header_t)) != MBUF_SUCCESS) {
		lib_die("Out of memory");
	}
	
	header = (ip_header_t *)mbuf->front_ptr;
	header->version_ihl = (IP_VER_IP<<4)|((sizeof(ip_header_t) + 0x3)>>2);
	header->tos = 0;
	set_uint16_field(header->total_length, mbuf->size);
	set_uint16_field(header->flags_frag_ofs, 0);
	header->flags_frag_ofs[0] |= flags<<5;
	header->ttl = 64;
	header->protocol = protocol;
	set_uint16_field(header->id, ++ip_id);
	set_uint32_field(header->src_ip, src_ip);
	set_uint32_field(header->dest_ip, dest_ip);
	ip_calc_checksum(header);
	
	if (dest_ip != 0xffffffff) {
		/* If we're sending out real packets, we must know our
		   netmask and gateway */
		uint32_t target_ip;
		if (!((net_cfg.ip_addr ^ dest_ip) & net_cfg.ip_net_mask)) {
			target_ip = dest_ip;
		} else {
			target_ip = net_cfg.ip_gateway;
		}
		if (arp_resolve(target_ip, &hw_addr)) {
			lib_die("Out of memory");
		}
	} else {
		/* Broadcast, just broadcast the thing */
		hw_addr = ETH_ADDR_BROADCAST;
	}
	eth_send_packet(mbuf, hw_addr, net_cfg.mac_addr, ETH_FRAME_TYPE_IP);
}


void ip_add_handler(int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip), 
		    void *data, uint8_t protocol)
{
	ip_handler_node_t *tmp = lib_malloc(sizeof(ip_handler_node_t));
	if (!tmp) {
		lib_die("Out of memory");
	}
	if (!handler_chain) {
		/* This is the first thing we're handling.  Set up an ethernet handler */
		eth_add_handler(ip_handler, 0, ETH_FRAME_TYPE_IP);
	}
	tmp->next = handler_chain;
	handler_chain = tmp;
	tmp->handler = handler;
	tmp->data = data;
	tmp->protocol = protocol;
}


void ip_del_handler(int (*handler)(mbuf_t *mbuf, void *data, uint32_t src_ip, uint32_t dest_ip),
		    void *data, uint8_t protocol)
{
	ip_handler_node_t *prev = 0;
	ip_handler_node_t *tmp = handler_chain;
	
	while (tmp) {
		if ((tmp->handler == handler) 
		    && (tmp->data == data)
		    && (tmp->protocol == protocol)) {
			if (prev) {
				prev->next = tmp->next;
			} else {
				handler_chain = tmp->next;
			}
			lib_free(tmp);
			if (handler_chain == 0) {
				/* Not listening for anything.  Remove our
				   ethernet hook */
				eth_del_handler(ip_handler, 0, ETH_FRAME_TYPE_IP);
			}
			return;
		}
		prev = tmp;
		tmp = tmp->next;
	}
	lib_die("ip_del_handler didn't find one to delete\n");
}
