/*
 * ----------------------------------------------------
 *
 * Linux Ethernet tunneling using /dev/net/tun
 * Used from Network interface Emulators
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 *
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#ifdef __linux__
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <asm/types.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <netinet/in.h>
#include <net/if.h>
#include <sys/wait.h>
#include <linux/netlink.h>
#include <linux/if_tun.h>
#include <configfile.h>


static void
exec_ifconfig(const char *ifname,const char *ipaddr) 
{ 
	pid_t pid;
	int status;
	if(!(pid=fork())) {
		execlp("sudo","/sbin/ifconfig","/sbin/ifconfig",ifname,ipaddr,NULL);
		fprintf(stderr,"exec sudo ifconfig failed\n");
		exit(13424);
	}
	waitpid(pid,&status,0);
	if(status!=0)  {
		fprintf(stderr,"ifconfig failed with status %d\n",status);
	}
}

static void
bridge_addif(const char *ifname,const char *bridge) 
{ 
	pid_t pid;
	int status;
	if(!(pid=fork())) {
		execlp("sudo","/sbin/brctl","/sbin/brctl","addif",bridge,ifname,NULL);
		fprintf(stderr,"exec sudo brctl failed\n");
		exit(13774);
	}
	waitpid(pid,&status,0);
	if(status!=0)  {
		fprintf(stderr,"brctl failed with status %d\n",status);
	}
}

static int 
configure_interface(const char *devname,const char *ifname,int ifnr) 
{
	char *ipaddr;
	char *bridge;
	ipaddr=Config_ReadVar(devname,"host_ip");	
	bridge=Config_ReadVar(devname,"bridge");
	if(!ipaddr && !bridge) {
		fprintf(stderr,"No IP-Address and no bridge given for Interface %s(%d)\n",ifname,ifnr);
		return 0;
	}
	if(ipaddr && strlen(ipaddr)) {
		exec_ifconfig(ifname,ipaddr);
	} 
	if(bridge && strlen(bridge)) {
		bridge_addif(ifname,bridge);
	}
	return 0;
}

/*
 * -------------------------------------
 * Create an ethernet tunnel 
 * -------------------------------------
 */

static int 
tap_open(const char *name)
{
    struct ifreq ifr;
    int fd;
    if(!name) {
	return -1;
    }
    if( (fd = open("/dev/net/tun", O_RDWR)) < 0 ) {
		perror("Can not open /dev/net/tun");
		sleep(1);
		return -1;
    }
    memset(&ifr, 0, sizeof(ifr));
    ifr.ifr_flags = IFF_TAP | IFF_NO_PI;
    strncpy(ifr.ifr_name, name, IFNAMSIZ);

    if (ioctl(fd, TUNSETIFF, (void *) &ifr) < 0) {
	  perror("ioctl TUNSETIFF");
          goto failed;
    }
    return fd;
failed:
    sleep(1);
    fprintf(stderr,"Ethernet Tunnel creation failed for \"%s\"\n",name);
    close(fd);
    return -1;
}

static int ifcounter=0;

int 
Net_CreateInterface(const char *devname) {
	int fd;
	int result;
	char *ifname;
	ifname=Config_ReadVar(devname,"host_ifname");	
	if(!ifname) {
		fprintf(stderr,"No host interface name in configfile for interface %s(%d)\n",devname,ifcounter);
		return -1;
	}
	fd=tap_open(ifname);
	if(fd<0)
		return fd;
	result=configure_interface(devname,ifname,ifcounter);
	ifcounter++;
	if(result<0) { 
		return result;
	}
	return fd;
}

#ifdef TAPTEST
int
main() {
	int i;
	tap_create("tap0");
	exec_ifconfig("tap0");
	while(1) {

	}
}
#endif

#endif
