/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: ldap.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $
 */

constant cvs_version="$Id: ldap.pike,v 1.1.1.1 2005/02/23 14:47:21 cvs Exp $";

inherit "/kernel/module";

#include <macros.h>
#include <config.h>
#include <attributes.h>
#include <classes.h>
#include <events.h>

//! This module is a ldap client inside sTeam. It reads configuration
//! parameters of sTeam to contact a ldap server.
//! All the user management can be done with ldap this way. Some
//! special scenario might require to modify the modules code to 
//! make it work.
//!
//! The configuration variables used are:
//! server - the server name to connect to
//! user   - ldap user for logging in
//! password - the password for the user
//! base_dc - ldap base dc, consult ldap documentation
//! objectName - the ldap object nameto be used for the search
//! userAttr - the attribute containing the user login name
//! passwordAttr - the attribute containing the password

static object      oLDAP;
static string sServerURL;
static mapping    config;

static array __restrictedLogins = ({ "postman", "root", "service" });

private static void connect_ldap()
{
   mixed err = catch {
	oLDAP = Protocols.LDAP.client(config["server"]);

	oLDAP->bind("cn="+config["user"] + 
		    (stringp(config->root_dn) && strlen(config->root_dn)>0?","+config->root_dn:""),  config["password"]);

	if ( oLDAP->error_number() > 0 )
	  FATAL("Failed to bind ldap: " + oLDAP->error_string());	
	oLDAP->set_scope(2);
	oLDAP->set_basedn(config["base_dc"]);
   };
    if ( err != 0 )
      FATAL("Failed to build connection to LDAP: %s\n%O", config->server, err);
    if ( objectp(oLDAP) && oLDAP->error_number() > 0 )
      FATAL("Failed to bind ldap: " + oLDAP->error_string());	
}

static void init_module()
{
    config = read_config(Stdio.read_file("config/ldap.txt"), "ldap");
    if ( !mappingp(config) ) {
	MESSAGE("LDAP Service not started - missing configuration !");
	return; // ldap not started !
    }
    MESSAGE("LDAP configuration is %O", config);
    if ( !config->objectName )
      steam_error("objectName configuration missing !");

    connect_ldap();
#if 0
    // if our main dc does not exist - create it
    oLDAP->add(config->base_dc, 
	       ([ 
		 "objectclass": ({ "dcObject", "organization" }),
		 "o": "sTeam Authorization Directory",
		 "dc": "steam"
	       ]));
#endif

}

void load_module()
{
  add_global_event(EVENT_USER_CHANGE_PW, sync_password, PHASE_NOTIFY);
  add_global_event(EVENT_USER_NEW_TICKET, sync_ticket, PHASE_NOTIFY);
}

static mixed map_results(object results)
{
  array result = ({ });
  
  for ( int i = 1; i <= results->num_entries(); i++ ) {
    mapping            data = ([ ]);
    mapping res = results->fetch(i);
    
    foreach(indices(res), string attr) {
      if ( arrayp(res[attr]) ) {
	if ( sizeof(res[attr]) == 1 )
	  data[attr] = res[attr][0];
	else
	  data[attr] = res[attr];
      }
    }
    if ( results->num_entries() == 1 )
      return data;
    result += ({ data });
  }
  return result;
}

static mapping fetch_user(string uname)
{
    mapping udata = ([ ]);
    object results;
    
    if ( !objectp(oLDAP) )
	return 0;


    if ( config->userdn )
      MESSAGE("userdn: %O", oLDAP->set_basedn(config->userdn+","+
					      config->base_dc));
    if ( oLDAP->error_number() )
      MESSAGE("Error: %s", oLDAP->error_string());

    if ( catch(results = 
	       oLDAP->search("("+config->userAttr+"="+uname+")")) )
    {
      connect_ldap(); // try to rebuild connection;
      return 0;
    }
      
    if ( oLDAP->error_number() )
      MESSAGE("Error while searching user: %s", oLDAP->error_string());
    
    if ( results->num_entries() == 0 ) {
        MESSAGE("User %s not found in LDAP directory.", uname);
	return 0;
    }
    udata = map_results(results);
    if ( stringp(udata[config->passwordAttr]) )
      sscanf(udata[config->passwordAttr], 
	     "{crypt}%s", udata[config->passwordAttr]);
    return udata;
}

mapping fetch(string dn, string pattern)
{
  // caller must be module...

  mapping   data;
  object results;

  if ( !_Server->is_module(CALLER) )
    steam_error("Access for non-module denied !");

  oLDAP->set_basedn(dn+"," + config->base_dc);
  
  if ( catch(results = oLDAP->search(pattern)) )
    {
      connect_ldap(); // try to rebuild connection;
      return 0;
    }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    return 0;
  }
  data = map_results(results);
  return data;
}

mapping fetch_group(string gname)
{
  mapping gdata = ([ ]);
  object results;

  if ( !objectp(oLDAP) )
    return gdata;
  
  if ( config->groupdn )
    oLDAP->set_basedn(config->groupdn+"," + config->base_dc);

    if ( catch(results = 
	       oLDAP->search("("+config->groupAttr+"="+gname+")")) )
    {
      connect_ldap(); // try to rebuild connection;
      return 0;
    }
  
  if ( oLDAP->error_number() )
    FATAL("Error: %s", oLDAP->error_string());
  
  if ( results->num_entries() == 0 ) {
    MESSAGE("Group %s not found in LDAP directory.", gname);
    return 0;
  }
  gdata = map_results(results);
  return gdata;
}

static bool check_password(string pass, string user_pw)
{
  if ( !stringp(pass) || !stringp(user_pw) )
    return 0;

  return verify_crypt_md5(pass, user_pw) || crypt(pass, user_pw);
}

bool authorize_ldap(object user, string pass) 
{
  if ( !objectp(oLDAP) || config->authorize != "ldap" )
    return false;

  if ( !objectp(user) )
    steam_error("User object expected for authorization !");

  string uname = user->get_user_name();
  mapping udata = fetch_user(uname);
  if ( mappingp(udata) ) {
      if ( stringp(udata[config->emailAttr]) && 
	   stringp(user->query_attribute(USER_EMAIL)) &&
	   udata[config->emailAttr] != user->query_attribute(USER_EMAIL) )
      {
	  MESSAGE("LDAP: user " + uname + " seems to be a different person "+
		  "email-ldap(%s), email(%s).", 
		  udata[config->emailAttr], user->query_attribute(USER_EMAIL));
	  return false;
      }
      else if ( check_password(pass, udata[config->passwordAttr]) )
      {
	// need to synchronise passwords from ldap if ldap is down ?!
	// this is only done when the password ldap password is received
	if ( udata[config->passwordAttr] != user->get_user_password()) {
	  catch(MESSAGE("Sync PW: %s:%s", udata[config->passwordAttr],
			user->get_user_password()));
	  user->set_user_password(udata[config->passwordAttr], 1);
	}
	MESSAGE("User %s LDAP authorized !", uname);
	return true;
      }
      else {
	MESSAGE("User %s found in LDAP directory - password failed !", uname);
	return false;
      }
  }
  MESSAGE("User " + uname + " was not found in LDAP directory.");
  // if notfound configuration is set to create, then we should create
  // a user.
  if ( config->notfound == "create" ) {
      add_user(uname, pass, user);
  }
  return false;
}

object sync(string name, string pass)
{
    object user;
    mapping udata = fetch_user(name);
    MESSAGE("Sync of ldap user: %O", udata);
    // create locally ?
    if ( mappingp(udata) ) {
        if ( !check_password(pass, udata[config->passwordAttr]) ) 
	{
	    MESSAGE("Password does not match with ldap entry!");
	    return 0; // password does not match
	}

	object factory = get_factory(CLASS_USER);
	user = factory->execute( ([ "name": name,
				  "pw": udata[config->passwordAttr],
				  "email": udata[config->emailAttr],
				  "fullname": udata[config->fullnameAttr],
				  "firstname": udata[config->nameAttr],
				  ]) );
	user->set_user_password(udata[config->passwordAttr], 1);
	user->activate_user(factory->get_activation());
	MESSAGE("LDAP: sync successfull !");
	return user;
    }
    return 0;
}

static void sync_password(int event, object user, object caller)
{
    if ( !mappingp(config) || config->sync != "true" )
	return;
    string crypted = user->get_user_password();
    string name = user->get_user_name();
    MESSAGE("LDAP Password sync for " + user->get_user_name());
    string dn = config->base_dc + " , " + config->userAttr + "=" + name;
    mixed err;
    err=catch(oLDAP->modify(dn, ([ config->passwordAttr: ({ 2,crypted }),])));
}

static void sync_ticket(int event, object user, object caller, string ticket)
{
  mixed err;
  if ( !mappingp(config) || config->sync != "true" )
    return;
  string name = user->get_user_name();
  string dn = config->base_dc + " , " + config->userAttr + "=" + name;
  err=catch(oLDAP->modify(dn, ([ "userCertificate": ({ 2,ticket }),])));
}


mixed query_attribute(string|int key)
{
    mixed res;
    
    if ( key == OBJ_ICON ) {
	object obj = CALLER->this();
	string uname = obj->get_user_name();
	object results = 
	    oLDAP->search("(objectclass="+config["objectName"]+")");
	for ( int i = 1; i <= results->num_entries(); i++ ) {
	    mapping res = results->fetch(i);
	    
	    if ( res[config["userAttr"]][0] == uname ) 
		return res[config->iconAttr][0];
	}
    }
    return ::query_attribute(key);
}

mixed set_attribute(string|int key, mixed val)
{
    array(string)      keywords;
    object obj = CALLER->this();
 
     
    if ( key == OBJ_ICON ) {
	// set the icon
    }
}

bool is_user(string user)
{
    object results = oLDAP->search("("+config["userAttr"]+"="+user+")");
    return (results->num_entries() > 0);
}

static bool add_user(string name, string password, object user)
{
    if ( search(__restrictedLogins, name) >= 0 )
      return false;

    string fullname = user->get_name();
    string firstname = user->query_attribute(USER_FIRSTNAME);
    string email = user->query_attribute(USER_EMAIL);

    mapping attributes = ([
      config["userAttr"]: ({ name }),
      config["fullnameAttr"]: ({ fullname }),
      "objectClass": ({ config["objectName"] }),
      config["passwordAttr"]: ({ make_crypt_md5(password) }),
    ]);
    if ( stringp(firstname) && strlen(firstname) > 0 )
      config["nameAttr"] = ({ firstname });
    if ( stringp(email) && strlen(email) > 0 )
      config["emailAttr"] = ({ email });

    array(string) requiredAttributes =  config["requiredAttr"];

    if ( arrayp(requiredAttributes) && sizeof(requiredAttributes) > 0 ) 
    {
	foreach(requiredAttributes, string attr) {
	    if ( zero_type(attributes[attr]) )
		attributes[attr] = ({ "-" });
	}
    }

    oLDAP->add(config["userAttr"]+"="+name+","+config["base_dc"], attributes);
    int err = oLDAP->error_number();
    if ( err != 0 )
      FATAL("Failed to add user , error number is " + oLDAP->error_string());
    return oLDAP->error_number() == 0;
}

string get_identifier() { return "ldap"; }











