###############################################################################
# This is the common code for tclisem and gxisem (eventually, I'll try to
# merge the common part of tkisem into this file as well.
#
# You should not need to modify anything in this file
###############################################################################

set library "$libdir/tkisem.$lib_ext"
set rom "$libdir/isem_rom"

if {[info exists env(TKISEM_ROM)]} {
    set rom $env(TKISEM_ROM)
}

load $library

#------------------------------------------------------------------------------
# globals
set super_breaks {}
set user_breaks {}
set count_super 0

###############################################################################
# messages
###############################################################################
proc debug {mes} {
    outs "DEBUG:  $mes"
}

proc set_status_message {msg} {
    outs "STATUS:  $msg"
}

proc err_msg {msg} {
    outs "ERROR:  $msg"
}

#------------------------------------------------------------------------------
# loading files
#------------------------------------------------------------------------------
proc load_file name {

    # make sure we can read the file we're trying to read
    if {[file readable $name]==0} {
	err_msg "load_file: Cannot read $name"
	return
    }

    global load_mode
    if {[catch {isem_load $name $load_mode} symbols] != 0} {
	err_msg $symbols
	return
    }

    global disasm_pc
    set disasm_pc -1

    global proc_mode
    upvar \#0 [format {%s_syms} $load_mode] syms
    set syms(x) y
    foreach i [array names syms] {
	unset syms($i)
    }

    foreach sect {abs bss data text} {
	upvar \#0 [format {%s_%s} $load_mode $sect] arr
	set arr(x) y
	foreach i [array names arr] {
	    unset arr($i)
	}
    }

    global text_start_$load_mode text_end_$load_mode
    global data_start_$load_mode data_end_$load_mode

    set text_start_$load_mode [lindex $symbols 0]
    upvar \#0 [format {%s_text} $load_mode] val
    set text_end_$load_mode [lindex $symbols 1]
    set data_start_$load_mode [lindex $symbols 2]
    set data_end_$load_mode [lindex $symbols 3]

    global cur_dump dump_mode dump_sect
    upvar \#0 [format {data_start_%s} $load_mode] datastart
    set cur_dump $datastart
    set dump_mode $load_mode
    set dump_sect "data"

    global user_breaks super_breaks
    set ${load_mode}_breaks {}
    set symbols [lrange $symbols 4 end]

    foreach s $symbols {
	set nam [lindex $s 0]
	if { ![regexp {.*\.o$} $nam] } {
	    set syms($nam) [list [lindex $s 1] [lindex $s 2]]
	    upvar \#0 [format {%s_%s} $load_mode [lindex $s 2]] val
	    set val([lindex $s 1]) $nam
	}
    }

    set xxx $syms(_etext)
    set text_end_$load_mode [lindex $xxx 0]

    set xxx $syms(_edata)
    set data_end_$load_mode [lindex $xxx 0]
    
    set_status_message "$name loaded in the $load_mode region"
}

proc is_breakp {mode addr} {
   global user_breaks super_breaks
   if { [lsearch -exact [set ${mode}_breaks] $addr] != -1} {
	return 1
    } else {
   	return 0
    }
}

#------------------------------------------------------------------------------
# running programs
#------------------------------------------------------------------------------

#..............................................................................
# driver -- run, stop when:
#	1) stop_run is set
#	2) we encounter a breakpoint
#	3) the processor enters an error state
#	4) enough steps have been executed
proc driver {{steps "never"}} {
    global stop_run
    global count_super
    global user_breaks super_breaks proc_mode
    global poll_rate poll_op
    
    set stop_run 0

    if { $steps == "never" } {
	set stop_cycle 0
	set done 0
    } else {
	if {$count_super} {
	    set stop_cycle [expr [isem_tsteps]+$steps]
	    set done [expr [isem_tsteps] >= $stop_cycle]
	} else {
	    set stop_cycle [expr [isem_usteps]+$steps]
	    set done [expr [isem_usteps] >= $stop_cycle]
	}
    } 

    set poll_step [expr [isem_tsteps] + $poll_rate]
    while { !$stop_run && !$done } {
	set proc_mode [isem_step]
	run_devices

	if {$proc_mode=="error"} {
	    err_msg "The processor is in an error state"
	    set stop_run 1
	    return
	}
	if {[llength ${proc_mode}_breaks]} { ;# see if worth checking..
	    set pc [isem_reg get pc]
	    if { [lsearch -exact [set ${proc_mode}_breaks] $pc] != -1 } {
		set_status_message [format {Breakpoint at %s} $pc]
		set stop_run 1
	    }
	}

	set tsteps [isem_tsteps]
	if { $tsteps == $poll_step } {
	    eval $poll_op
	    set poll_step [expr [isem_tsteps] + $poll_rate]
	}

	if { $stop_cycle != 0 } {
	    if {$count_super} {
		set done [expr $tsteps >= $stop_cycle]
	    } else {
		set done [expr ("$proc_mode"=="user") && ([isem_usteps] >= $stop_cycle) ]
	    }
	}
    }
}

###############################################################################
# ---- devices -----
###############################################################################

proc stop_processor {} {
    global stop_run
    set stop_run 1
}

proc eval_command {} {
    global command
    outs "ISEM> $command"
    set temp $command
    set command ""
    if { [catch {eval $temp} stat] != 0 } {
	err_msg $stat
    }
}

#------------------------------------------------------------------------------
# the device run list
#------------------------------------------------------------------------------
set dev_run_list {}
proc run_devices {} {
    global dev_run_list

    foreach dev_proc $dev_run_list {
	$dev_proc
    }
}

proc add_run {dev_proc} {
    global dev_run_list

    lappend dev_run_list $dev_proc
}

proc rmv_run {dev_proc} {
    global dev_run_list

    set index [lsearch -exact $dev_run_list $dev_proc]
    if { $index != -1 } {
	set dev_run_list [lreplace $dev_run_list $index $index]
    }    
}

#------------------------------------------------------------------------------
# the console device
#------------------------------------------------------------------------------
set console_input {}
set console_line ""

proc console {op addr bytemask value} {
    global console_input console_line
    if { $op == "write" } {
	outc [format {%c} $value]
    } else {
	if { $console_line == "" } {
	    if { $console_input == {} } {
		return 0xffffffff
	    } else {
		set console_line [format "%s\n" [lindex $console_input 0]]
		set console_input [lreplace $console_input 0 0]
	    }
	}

	scan $console_line %c res
	set console_line [string range $console_line 1 \
			      [expr [string length $console_line] - 1]]
	return $res
    }
}

isem_device console $console_address $console_mode

#------------------------------------------------------------------------------
# the halt device
#------------------------------------------------------------------------------
proc halt_device {op addr bytemask value} {
    global stop_run
    set stop_run 1
}

isem_device halt_device $halt_address $halt_mode

#------------------------------------------------------------------------------
# the timer device
#------------------------------------------------------------------------------
set timer_period 0x00000000
set timer_count 0x00000000
set timer_interrupt 0

proc timer_tick {} {
    global timer_period timer_count
    if {$timer_period != 0} {
	set timer_count [format {0x%.8x} [expr $timer_count+1]]
	if {$timer_count == $timer_period} {
	    timer_interrupt 1
	    set timer_count 0x00000000
	}
    }
}

proc timer_interrupt {state} {
    global timer_interrupt timer_int_level

    set timer_interrupt $state
    isem_interrupt $timer_int_level $state
}

proc timer {op addr bytemask value} {
    global timer_period

    #puts [format "uartcall %s %d %d %d" $op $addr $bytemask $value]
    timer_interrupt 0
    if {$op == "write"} {
	set_timer_period $value
    } else {
        scan $timer_count 0x%x woof
        return $woof
    }
}

proc set_timer_period {period} {
    global timer_period

    set old_period $timer_period
    set timer_period [format {0x%.8x} $period]
    set timer_count 0x00000000
    if {$timer_period != 0 && $old_period == 0} {
	add_run timer_tick
    }
    if {$timer_period == 0 && $old_period != 0} {
	rmv_run timer_tick
    }
}

isem_device timer $timer_address $timer_mode

###############################################################################
# the interface routines
###############################################################################

proc var { {name "none"} {value ""} } {
    global disasm_loc dump_loc load_mode reg_view console_input count_super
    switch $name {
	"none" {
	    outs "disasm_loc = '$disasm_loc'"
	    outs "dump_loc = '$dump_loc'"
	    outs "load_mode = '$load_mode'"
	    outs "reg_view = '$reg_view'"
	    outs "count_super = '$count_super'"
	    outs "console_input = '$console_input'"
	}
	"count_super" {
	    if { $value == "" } {
		outs "count_super = '$count_super'"
	    } else {
		set count_super $value
	    }
	}
	"reg_view" {
	    if { $value == "" } {
		outs "reg_view = '$reg_view'"
	    } else {
		set reg_view $value
	    }
	}
	"console_input" {
	    outs "console_input = '$console_input'"
	}
	"disasm_loc" {
	    outs "disasm_loc = '$disasm_loc'"
	}
	"disasm_loc" {
	    outs "disasm_loc = '$disasm_loc'"
	}
	default {
	    err_msg "unknown variable $name"
	}
    }
}

proc con { {line ""} } {
    global console_input
    lappend console_input $line
}

proc load {{filename "a.out"} {mode ""}} {
    global load_mode
    if { $mode != "" } {
	set load_mode $mode
    }

    set void [load_file $filename]
}

proc quit {} {
    exit
}

proc cycles {} {
    global total_cycles

    outs [format "Total = %d" $total_cycles]
}

proc run {} {
    global stop_run poll_op
    driver
    eval $poll_op
}

proc step {{nsteps 1}} {
    driver $nsteps
    global proc_mode count_super poll_op
    if {($proc_mode=="user") || $count_super} {
	reg
    }
    eval $poll_op
}

proc trace {{nsteps 1}} {
    global proc_mode count_super poll_op
    for {set i 0} {$i < $nsteps} {incr i} {
	driver 1
	if {($proc_mode=="user") || $count_super} {
	    reg
	}
	eval $poll_op
    }
}

proc halt {} {
    global stop_run
    set stop_run 1
}

proc reg {{register -1} {value "none"}} {
    if { $register == -1 } {
	outs "  ----0--- ----1--- ----2--- ----3--- ----4--- ----5--- ----6--- ----7---"
	set i 0
	foreach grp {G O L I} {
	    set line "$grp "
	    for {set j 0} {$j < 8} {incr j} {
		append line [format "%.8x " [isem_reg get r$i]]
		incr i
	    }
	    outs $line
	}
	set psr [isem_reg get psr]
	set psr_c [expr $psr>>20 &1]
	set psr_v [expr $psr>>21 &1]
	set psr_z [expr $psr>>22 &1]
	set psr_n [expr $psr>>23 &1]
	
	set pc [isem_reg get pc]

	outs [format "  PC: 0x%.8x    nPC: 0x%.8x   PSR: 0x%.8x   N:%d Z:%d V:%d C:%d" \
		  $pc [isem_reg get npc] $psr $psr_n $psr_z $psr_v $psr_c ]
	global proc_mode reg_view
	set instr [isem_disasm $pc $proc_mode $reg_view]
	set instr_lab [lindex $instr 0]
	set instr_op [lindex $instr 1]
	set instr_opnds [lindex $instr 2]

	if { $instr_lab == {} } {
	    outs [format "0x%.8x:\t%s\t%s" $pc $instr_op $instr_opnds]
	} else {
	    outs [format "%s\t%s\t%s" $instr_lab $instr_op $instr_opnds]
	}

    } elseif {$value == "none"} {
	if {[catch {isem_reg get $register} val] != 0} {
	    err_msg $val
	} else {
	    outs [format "register: %s = %s" $register $val]
	}
    } else {
	global load_mode
	if {[catch {map_addr $value $load_mode "text"} value] != 0} {
	    return
	}
	set value [lindex $value 2]
	if {[catch {isem_reg set $register $value} val] != 0} {
	    err_msg $val
	} else {
	    outs [format "register: %s = %s" $register $val]
	}
    }
}

proc symb { {space "none"} } {
    global load_mode

    if { $space == "none" } {
	upvar \#0 [format {%s_syms} $load_mode] syms
    } elseif { $space=="user" || $space=="super" } {
	upvar \#0 [format {%s_syms} $space] syms
    } else {
	err_msg "Unknown mode: $space"
	return
    }

    foreach i [lsort [array names syms]] {
	if { [string length $i] < 8 } {
	    outs [format "%s\t\t%s" $i $syms($i)]
	} else {
	    outs [format "%s\t%s" $i $syms($i)]
	}
    }

}

proc map_addr {addr def_mode def_space} {
    if {[regexp {(.*):(.*)} $addr all orig_space offset]} {
	set mode ""
	if { [scan $orig_space %i space] } {
	    if { $space==8 } {
		set space "text"
		set mode "user"
	    } elseif { $space==9 } {
		set space "text"
		set mode "super"
	    } elseif { $space==10 } {
		set space "data"
		set mode "user"
	    } elseif { $space==11 } {
		set space "data"
		set mode "super"
	    }
	}
    } elseif {[regexp {%.*} $addr]} {
	set mode $def_mode
	set space $def_space
	if {[catch {isem_reg get $addr} offset] != 0} {
	    err_msg $offset
	}
    } else {
	set mode $def_mode
	set space $def_space
	set offset $addr
    }

    if { $mode=="" } {
	err_msg "$orig_space is not a valid address space"
	return -code error
    }
    if {[scan $offset %i value] != 1} {
	upvar \#0 [format {%s_syms} $mode] syms
	if {[info exists syms($offset)]} {
	    set value [lindex $syms($offset) 0]
	} else {
	    err_msg "$offset is not defined in $mode mode"
	    return -code error
	}
    }
    set value [format {0x%.8x} $value]
    return [list $mode $space $value]
}

proc disasm { {start "none"} {end "none"} } {
    global load_mode proc_mode
    global cur_disasm disasm_pc disasm_mode disasm_sect
    if { $start != "none" } {
	if {[catch {map_addr $start $load_mode "text"} start] != 0} {
	    return
	}
    } else {
	set pc [isem_reg get pc]
	if { $disasm_pc != $pc } {
	    set start [list $proc_mode "text" $pc]
	} else {
	    set start [list $disasm_mode $disasm_sect $cur_disasm]
	}
    }

    if { $end != "none" } {
	if {[catch {map_addr $end [lindex $start 0] [lindex $start 1]} end] != 0} {
	    return
	}
    } else {
	set end [list [lindex $start 0] [lindex $start 1] \
		     [expr [lindex $start 2] + 15*4] ]
    }

    if { [lindex $start 0] != [lindex $end 0] \
	    ||  [lindex $start 1] != [lindex $end 1] } {
	err_msg [format "%s %s and %s %s are not compatible spaces" \
		     [lindex $start 0] [lindex $start 1] \
		     [lindex $end 0] [lindex $end 1] ]
	return
    }

    set mode [lindex $start 0]
    set section [lindex $start 1]
    set start [expr (([lindex $start 2]>>2)<<2)]
    set end [lindex $end 2]

    upvar \#0 [format {%s_syms} $mode] syms
    if { $end > [lindex $syms(_e$section) 0] } {
	set end [lindex $syms(_e$section) 0]
    }

    global reg_view

    while { $start <= $end } {
	set instr [isem_disasm $start $mode $reg_view]
	set instr_lab [lindex $instr 0]
	set instr_op [lindex $instr 1]
	set instr_opnds [lindex $instr 2]

	if { $instr_lab == {} } {
	    outs [format "0x%.8x:\t%s\t%s" $start $instr_op $instr_opnds]
	} else {
	    if {[string length $instr_lab] < 8 } {
		outs [format "%s\t\t%s\t%s" $instr_lab $instr_op $instr_opnds]
	    } else {
		outs [format "%s\t%s\t%s" $instr_lab $instr_op $instr_opnds]
	    }
	}
	incr start 4
    }

    set cur_disasm $start
    set disasm_pc [isem_reg get pc]
    set disasm_mode $mode
    set disasm_sect $section
    return
}

proc dump { {start "none"} {end "none"} } {
    global load_mode
    global cur_dump dump_mode dump_sect
    if { $start != "none" } {
	if {[catch {map_addr $start $load_mode "data"} start] != 0} {
	    return
	}
    } else {
	set start [list $dump_mode $dump_sect $cur_dump]
    }

    set start [list [lindex $start 0] [lindex $start 1] [expr ([lindex $start 2]>>2)<<2] ]

    if { $end != "none" } {
	if {[catch {map_addr $end [lindex $start 0] [lindex $start 1]} end] != 0} {
	    return
	}
    } else {
	set end [list [lindex $start 0] [lindex $start 1] \
		     [expr [lindex $start 2] + 128-1] ]
    }

    if { [lindex $start 0] != [lindex $end 0] \
	    ||  [lindex $start 1] != [lindex $end 1] } {
	err_msg [format "%s %s and %s %s are not compatible spaces" \
		     [lindex $start 0] [lindex $start 1] \
		     [lindex $end 0] [lindex $end 1] ]
	return
    }

    set mode [lindex $start 0]
    set section [lindex $start 1]
    set start [lindex $start 2]
    set end [lindex $end 2]

    upvar \#0 [format {%s_syms} $mode] syms
    if { $end > [lindex $syms(_e$section) 0] } {
	set end [lindex $syms(_e$section) 0]
    }

    while { $start <= $end } {
	set line [format {0x%.8x} $start]
	set words [isem_mem_rd $mode $section $start [expr $start+12]]
	set bytes ""
	set chars ""
	for {set i 0} {$i < 4} {incr i} {
	    set word [lindex $words $i]
	    for {set j 0} {$j < 4} {incr j} {
		set byte [expr $word>>24 & 0xff]
		set word [expr $word<<8]
		set bytes [format {%s %.2x} $bytes $byte]
		if {$byte < 32 || $byte > 126 } {
		    set byte 46
		}
		set chars [format {%s%c} $chars $byte]
	    }
	}
	outs "$line $bytes $chars"
	incr start 16
    }

    set cur_dump $start
    set dump_mode $mode
    set dump_sect $section
    return
}

proc break { {addr "none"} {del 0} } {
    global load_mode
    global user_breaks super_breaks
    if { $addr != "none" } {
	if {[catch {map_addr $addr $load_mode "text"} addr] != 0} {
	    return
	}

	set mode [lindex $addr 0]
	set section [lindex $addr 1]
	set addr [lindex $addr 2]

	if {$section != "text"} {
	    err_msg "Can't set breakpoints in the $section section"
	    return
	}

	upvar \#0  [format {%s_breaks} $mode] breaks
	if { $del == 0 } {
	    # add the breakpoint
	    if { [lsearch -exact $breaks $addr] == -1 } {
		lappend breaks $addr
	    }
	} else {
	    # delete the break point
	    set index [lsearch -exact $breaks $addr]
	    if { $index != -1 } {
		set breaks [lreplace $breaks $index $index]
	    } 
	}
    } else {
	# list all breakpoints
	if { $user_breaks != {} } {
	    outs "User Breakpoints"
	    foreach break [lsort $user_breaks] {
		outs "\t$break"
	    }
	}
	if { $super_breaks != {} } {
	    outs "Supervisor Breakpoints"
	    foreach break [lsort $super_breaks] {
		outs "\t$break"
	    }
	}
    }
}


proc help { {topic "help"} } {
    switch $topic {
	"help" {
	    outs "  Syntax:"
	    outs "\thelp \[topic\]"
	    outs ""
	    outs "  The help command provides information on commands and features."
	    outs "  Without an argument, help displays short message."
	    outs ""
	    outs "  You can get additional help on specific commands or topics by using"
	    outs "  an optional argument.  The help command can provide information on the"
	    outs "  following commands and topics:"
	    outs ""
	    outs "  COMMANDS"
	    outs "\thelp"
	    outs "\texit\tquit"
	    outs "\tload\trun\tstep\ttrace\tbreak\thalt"
	    outs "\tdisasm\tdump\treg\tsymb"
	    outs "\tcon\tvar"
	    outs "  TOPICS"
	    outs "\taddress\tvalue"
	    outs "\tmode\tspace\tsyntax\tregister"
	    outs "\tvariable"
	    outs "\tdevices"
	}
	"var" {
	    outs "  Syntax:"
	    outs "\tvar \[name \[value \] \]"
	    outs ""
	    outs "  State variables:"
	    outs "\tdisasm_loc dump_loc load_mode reg_view console_input count_super"
	    outs ""
	    outs "  Examine or set state variables.  Without any arguments, this command"
	    outs "  displays the current values for all of the state variables.  With one"
	    outs "  one argument, this command displays the current value of the named"
	    outs "  argument.  With two arguments, this command sets the named variable"
	    outs "  to the value given as the third argument.  Only  reg_view  and "
	    outs "  count_super  can be set using this command."
	}
	"con" {
	    outs "  Syntax:"
	    outs "\tcon line"
	    outs ""
	    outs "  State variables:"
	    outs "\tconsole_input"
	    outs ""
	    outs "  Enter a line of text for the console input.  If the line of text"
	    outs "  includes spaces, it must be enclosed in double quotes (\")."
	    outs ""
	    outs "  See also:"
	    outs "\tdevices"
	}
	"quit" -
	"exit" {
	    outs "  Syntax:"
	    outs "\tquit"
	    outs "\texit"
	    outs ""
	    outs "  Either of these commands will stop the interpreter."
	}
	"load" {
	    outs "  Syntax:"
	    outs "\tload \[file_name \[mode\]\]"
	    outs ""
	    outs "  State variables:"
	    outs "\tload_mode"
	    outs ""
	    outs "  Loads an executable file.  Without any arguments this command loads"
	    outs "  the file \"a.out\".  Optional arguments can be used to specify the"
	    outs "  name of the file to load and the mode.  When the mode is not"
	    outs "  specified it defaults to the current value of load_mode.  When the"
	    outs "  mode is specified, it is used to set the value of load_mode."
	    outs ""
	    outs "  When a file is loaded, any breakpoints or symbols defined for the"
	    outs "  mode are cleared."
	    outs ""
	    outs "  See also:"
	    outs "\tbreak, symb"
	    outs "\tspace, symbols"
	}
	"step" -
	"trace" -
	"run" {
	    outs "  Syntax:"
	    outs "\trun"
	    outs "\tstep \[count\]"
	    outs "\ttrace \[count\]"
	    outs ""
	    outs "  State variables:"
	    outs "\tcount_super"
	    outs ""
	    outs "  All of these commands execute instructions based on the current value"
	    outs "  of the program counters (pc and npc).  The run command executes"
	    outs "  instructions until a breakpoint in encountered, or until the processor"
	    outs "  is halted.  The step instruction executes the specified number of "
	    outs "  instructions (default 1) and then displays the registers.  The trace"
	    outs "  command is similar to the step command except that the registers are"
	    outs "  displayed after each instruction is executed."
	    outs ""
	    outs "  By default, the step and trace commands do not count instructions"
	    outs "  executed when the processor is in supervisor mode (e.g., when"
	    outs "  handling a trap).  If you want to count supervisor instructions,"
	    outs "  you should set the variable \"count_super\" to 1.  In tcl, this "
	    outs "  is done using the following instruction."
	    outs "\tset count_super 1"
	    outs ""
	    outs "  Note, while the interpreter is executing SPARC instructions, it still"
	    outs "  interprets any commands that you type.  As such, the \"halt\" and \"con\""
	    outs "  commands can be used to halt the interpretation of instructions or to"
	    outs "  enter new lines of console input."
	    outs ""
	    outs "  See also:"
	    outs "\tbreak, reg, halt, con"
	}
	"break" {
	    outs "  Syntax:"
	    outs "\tbreak \[address \[delete\]\]"
	    outs ""
	    outs "  Without any arguments, this command will display the breakpoints"
	    outs "  that have been set.  With a single argument, this command sets a"
	    outs "  breakpoint at the specified address.  With two arguments, this"
	    outs "  command deletes a the breakpoint specified by the first argument."
	    outs ""
	    outs "  See also:"
	    outs "\trun, load"
	    outs "\taddress"
	}
	"halt" {
	    outs "  Syntax:"
	    outs "\thalt"
	    outs ""
	    outs "  Halts the interpretation of SPARC instructions.  This command is"
	    outs "  only meaningful if the interpreter is processing a run, step, or"
	    outs "  trace command (i.e., interpreting SPARC instructions)."
	    outs ""
	    outs "  See also:"
	    outs "\trun, step, trace"
	}
	"disasm" {
	    outs "  Syntax:"
	    outs "\tdisasm \[start \[end\]\]"
	    outs ""
	    outs "  State variables:"
	    outs "\tdisasm_loc, reg_view"
	    outs ""
	    outs "  Without any arguments, this command disassembles and displays"
	    outs "  16 instructions.  If the program counter has not changed since"
	    outs "  the previous invocation, the 16 instructions are those following"
	    outs "  the instructions shown in the previous invocation.  Otherwise,"
	    outs "  the 16 instructions starting from the program counter are displayed."
	    outs "  The two optional arguments can be used to specify the first and"
	    outs "  last instruction displayed."
	    outs ""
	    outs "  By default, the register names are displayed using the names %r0"
	    outs "  through %r31.  If you would like the register names displayed using"
	    outs "  the window names (e.g., %i2), you need to set the variable reg_view"
	    outs "  to \"window\".  The following tcl command will accomplish this:"
	    outs "\tset reg_view \"window\""
	    outs ""
	    outs "  See also:"
	    outs "\taddress"
	}
	"dump" {
	    outs "  Syntax:"
	    outs "\tdump \[start \[end\]\]"
	    outs ""
	    outs "  State variables:"
	    outs "\tdump_loc"
	    outs ""
	    outs "  Without any arguments, this command displays 128 bytes starting"
	    outs "  just after the last bytes displayed.  The two optional arguments"
	    outs "  can be used to specify the first and last instruction displayed."
	    outs ""
	    outs "  See also:"
	    outs "\taddress"
	}
	"reg" {
	    outs "  Syntax:"
	    outs "\treg \[register \[value\]\]"
	    outs ""
	    outs "  Without any arguments, this command displays the current values"
	    outs "  of the integer unit registers and the next instruction.  With a"
	    outs "  single argument, it displays the value of the identified register."
	    outs "  With two arguments, it sets the value of the identified register"
	    outs "  to the specified value."
	    outs ""
	    outs "  See also:"
	    outs "\ttrace, step"
	    outs "\tregister, value"
	}
	"symb" {
	    outs "  Syntax:"
	    outs "\tsymb \[mode\]"
	    outs ""
	    outs "  State variables:"
	    outs "\tload_mode"
	    outs ""
	    outs "  Without any arguments, this command displays the symbols for the"
	    outs "  last executable file loaded.  The optional arguments can be used"
	    outs "  to view the symbols from user or supervisor space."
	    outs ""
	    outs "  See also:"
	    outs "\tload"
	    outs "\tmode"
	}
	"address" {
	    outs "  An address consists of an optional space identifier followed"
	    outs "  by an offset.  When it is present, the space identifier consists"
	    outs "  of an integer followed by ':'.  The offset can be specified by"
	    outs "  an integer or a symbol."
	    outs ""
	    outs "  Integer values can be specified in decimal notation or in"
	    outs "  hexadecimal notation (using the 0x prefix)."
	    outs ""
	    outs "  Examples"
	    outs "  \t9:0x2040, 8:start, end, 0x2020"
	    outs ""
	    outs "  Commands that use addresses:"
	    outs "\tbreak, disasm, dump"
	}
	"value" {
	    outs "  A value can be specified as an address (i.e., a symbol or a integer)"
	    outs "  or by the contents of a register (e.g., %pc)."
	    outs ""
	    outs "  Commands that use values:"
	    outs "\treg"
	}
	"mode" {
	    outs "  A mode can be one of \"user\" or \"super\""
	    outs ""
	    outs "  Commands that use modes:"
	    outs "\tload, symb"
	}
	"register" {
	    outs "  A register name is specified by '%' followed by one of the"
	    outs "  following:"
	    outs "\tg0 g1 g2 g3 g4 g5 g6 g7    o0 o1 o2 o3 o4 o5 o6 o7"
	    outs "\tl0 l1 l2 l3 l4 l5 l6 l7    i0 i1 i2 i3 i4 i5 i6 i7"
	    outs "\tr0  r1  r2  r3  r4  r5  r6  r7  r8  r9  r10 r11 r12 r13 r14 r15"
	    outs "\tr16 r17 r18 r19 r20 r21 r22 r23 r24 r25 r26 r27 r28 r29 r30 r31"
	    outs "\tpc npc psr y wim tbr"
	}
	"space" {
	    outs "  The four address spaces are:"
	    outs "\t 8\tuser text"
	    outs "\t 9\tsuper text"
	    outs "\t10\tuser data"
	    outs "\t11\tsuper data"
	}
	"syntax" {
	    outs "\tbreak \[address \[delete\]\]"
	    outs "\tdisasm \[start \[end\]\]"
	    outs "\tdump \[start \[end\]\]"
	    outs "\texit"
	    outs "\thelp \[topic\]"
	    outs "\tload \[file_name \[mode\]\]"
	    outs "\tquit"
	    outs "\treg \[register \[value\]\]"
	    outs "\trun"
	    outs "\tstep \[count\]"
	    outs "\tsymb \[mode\]"
	    outs "\ttrace \[count\]"
	}
	"variable" {
	    outs "  State variables:"
	    outs "\tdisasm_loc\tmemory location for the  disasm  command"
	    outs "\tdump_loc\tmemory location for the  dump  command"
	    outs "\tload_mode\tmode for the last  load  operation (user or super)"
	    outs "\treg_view\tcontrols how register namess are shown when"
	    outs "\t\t\tinstructions are disassembled (regular or window)"
	    outs "\tcount_super\tcontrols whether instructions executed in supervisor"
	    outs "\t\t\tmode are counted in the  trace  and  step  commands"
	    outs "\tconsole_input\tthe remaining lines of console input."
	    outs ""
	    outs "  All of these variables can be examined using the  var  commmand"
	    outs "  In addition, the variables  reg_view  and  count_super  can be set"
	    outs "  using the  var  command."
	}
	"devices" {
	    outs "  This version of isem supports the timer, the console, and the halt device."
	    outs "  The console device is a bit strange in that its input and output are mixed"
	    outs "  with the interpreter messages and commands.  Messages from the interpreter"
	    outs "  are distinguished by a prefix (STATUS, ERROR, or DEBUG).  Console input is"
	    outs "  provided using the \"con\" command.  To use this command, you must type"
	    outs "  the name of the command (con) followed by the line of input in quotes."
	    outs "  For example,"
	    outs "\tcon \"xxx yyy\""
	    outs "  creates an input line consisting of \"xxx yyy\\n\" (the con command adds"
	    outs "  the newline)."
	}
	default {
	    outs "  No help is available for $topic"
	}
    }    
}

proc reset {} {
    global load_mode stop_run rom release
    global poll_op poll_rate

    debug "Booting supervisor from $rom"
    set load_mode super
    
    set stop_run 0
    
    load_file $rom
    set_status_message "Running supervisor initialization..."
    eval $poll_op
    set poll_rate 1024
    driver
    set load_mode user
    
    set_status_message "tcl ISEM version Release $release ready"
    eval $poll_op
    set poll_rate 256
}

