/* ==================================================== ======== ======= *
 *
 *  ufont.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _ufont_hpp_
#define	_ufont_hpp_
//pragma ident	"@(#)ufont.hpp	ubit:03.06.02"
#include <ubit/uprop.hpp>


/** Active Font Property.
 */
class UFont : public UProp {
public:
  static const int MIN_LSIZE    = 1;
  static const int MEDIUM_LSIZE = 8;
  static const int MAX_LSIZE    = 23;
  ///< minimum, medium and maximum LOGICAL size (not a point size!).

  static const int FONTMAP_SIZE = 4 * (UFont::MAX_LSIZE + 1);
  ///< internals.

  struct FontSize {
    int logsize;		// logical size
    int ptsize;		 	// point size
    const char *natspec;	// native point spec
  };
  ///< internal representation of a font size.

  enum FontStyle {
    BOLD	   = 1<<0,	// DON'T CHANGE VALUE!
    ITALIC	   = 1<<1,	// DON'T CHANGE VALUE!
    FILL           = 1<<2,
    UNDERLINE	   = 1<<3,
    OVERLINE	   = 1<<4,
    STRIKETHROUGH  = 1<<5
  };
  ///< bitmask values used by UFont::setStyle().

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  static UFont normal, bold, italic, fill, underline, overline, strikethrough;
  static UFont _bold, _italic, _fill, _underline, _overline, _strikethrough;
  /**< Font Style Modifiers. 
   * - 'normal' means not bold, not italic, not underlined
   * - prefix '_' means NOT (for example: '_bold' means 'not bold')
   * - Font Mofifiers can be combined (when applicable)
   *   Example: <pre>  ubox( UFont::bold + UFont::italic + "abcd" )  </pre>*/

  static UFont xx_small, x_small, small, medium, large, x_large, xx_large;
  /**< Font Size Modifiers.
   *  'medium' is equivalent to: logical lsize = MEDIUM_LSIZE */

  static UFont standard, inherit, helvetica, courier, times, fixed;
  /**< Font Family Modifiers.
   * - 'standard' is the default Font Family (with 'medium' size)
   * - 'inherit' means that the font is inherited from parents */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UFont(const class UFontFamily& family, int style = 0,	int lsize = 0, u_modes = 0);
  /**< creates a new Font with custom characteristics.
   * arguments:
   * - 'family': the UFontFamily of this new font
   * - 'style' : the font style: an ORed combination of UFont::FontStyle constants
   *   (ex: BOLD|ITALIC or -BOLD|-ITALIC). negative values deactivate these styles
   * - 'lsize' : the LOGICAL size (not the point size!), which must be in range
   *   [UFont::MIN_LSIZE, UFont::MAX_LSIZE] or 0 (in which case the size is
   *   inherited from parents */

  UFont(const UFont& = UFont::standard);
  ///< creates a new Font that is a copy of another Font.

  friend UFont& ufont(const UFont& _f) {return *new UFont(_f);}
  ///< creator shortcut that is equivalent to: *new UFont().

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  UFont& operator=(const UFont&);
  ///< copies font characteristics (same as set() but not virtual).

  virtual void set(const UFont&);
  ///< copies font characteristics (same as operator = but virtual).

  virtual void merge(const UFont&);
  /**< merge font characteristics.
   * sets NON default values and combines styles */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual bool equals(const UFont&) const;
  ///< returns true if font characteristics are the same

  bool isBold() const;
  bool isItalic() const;
  ///< is this font in the Bold or Italic style ?.

  bool isFilled() const;
  bool isUnderlined() const;
  bool isOverlined() const;
  bool isStrikethrough() const;

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void setFamily(const UFont&);
  void setFamily(const class UFontFamily&);
  ///< changes the Font Family of this font object (see: UFontFamily).

  const class UFontFamily* getFamily() const;
  ///< returns the Font Family of this font object (see: UFontFamily).

  void setStyle(const UFont&);
  void setStyle(signed int font_styles, bool update);
  /**< 
   * changes the Font Style of this font object.
   * 'font_styles' is an integer bitmask (values are ORed from enum FontStyle)
   * 'font_styles' can be negative, which means 'disable these styles'
   * (as in: UFont::_italic, etc) */

  void setSize(const UFont &f);
  ///< copies font size from another font object.

  int getLsize() const;
  ///< returns LOGICAL font size (in range: UFont::MIN_LSIZE to UFont::MAX_LSIZE).

  void setLsize(int logical_size, bool update = true);
  ///< changes LOGICAL font size (in range: UFont::MIN_LSIZE to UFont::MAX_LSIZE).

  int getPointSize() const;
  int getPtsize() const;
  ///< returns POINT size.

  void setPointSize(int point_size, bool update = true);
  void setPtsize(int point_size, bool update = true);
  ///< changes POINT size.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void update();
  ///< update parents' graphics.

  bool realize(class UDisp*);
  /**< [impl] allocates physical resources.
   * this function is automatically called when this font is used
   * for the first time. There no need to call it in client code except
   * for loading fonts in advance.
   * - note: the argument can be the UAppli (the Application Context) */

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  friend class UDisp;
  friend class UNatDisp;
  friend class UFontDesc;

  void setImpl(const UFontDesc*, bool update);
  virtual void putProp(class UContext*, class UCtrl*);

private:
  const class UFontFamily *family;
  short lsize;			 // logical font size of this UFont
  short onStyles, offStyles;	 // activated and disactivated styles
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Font Family object.
 * A Font Family represents a collection of fonts for all possible sizes
 * and styles (however, actual fonts are loaded dynamically when necessary)
 */
class UFontFamily {
public:
  /** Predefined font families.
   * - 'standard' is the default font family. 'standard' must always
   *   be physically available and it is preloaded for all styles.
   * - 'any' represents any font family. It is used in UFont descriptions
   *   when changing a style or a size without changing the font family.
   */
  static UFontFamily standard, any, helvetica, times, courier, fixed;

  void init(const char *name,
	      const char *normal_weight = "medium",
	      const char *bold_weight   = "bold", 
	      const char *slant         = "o",   // may be "i"
	      const char *compression   = "normal",
	      const char *encoding      = "1", 
	      const UFont::FontSize *font_sizes = null);
  /**< creates a new FontFamily from a native system font name.
   * !Warnings:
   * - the character strings are NOT copied: they must NOT be freed nor changed!
   * - the 'font_sizes' vector must be MAX_LSIZE long (or null which means
   *   that default values are used)
   */

  int lsizeToPtsize(int logical_size) const;
  int ptsizeToLsize(int point_size) const;
  ///< these functions convert point sizes to logical sizes and vice versa.

  static float getXYScale(short lscale);
  ///< estimates the XY scale  from the logical scale.

  static const UFont::FontSize* getFontSizes();
  ///< returns an array of MAX_LSIZE.

#ifndef NO_DOC
  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  friend class UDisp;
  friend class UNatDisp;
  friend class UFont;
private:
  static const UFont::FontSize *FONT_SIZES;  // array of MAX_LSIZE
  static int family_count;
  unsigned long index;
  const char *name;
  const char *normal_weight, *bold_weight, *slant, *compression, *encoding;
  const UFont::FontSize *font_sizes;   // array of MAX_LSIZE
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** [impl] Internal representation for fonts.
 * should not be used directly
 */
struct UFontDesc {
  const class UFontFamily* family;
  short font_ix, lsize, styles;

  UFontDesc() {}
  UFontDesc(const UFont&);

  void set(const UFont&);
  void set(const UFont&, int delta_scale);

  // merge only sets NON default values (and combines styles)
  void merge(const UFont&, int delta_scale);

  // note: decrements if the value is negative
  void incrScale(int delta_scale);
};

#endif
/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

