/* ==================================================== ======== ======= *
 *
 *  uima.hpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (Subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

#ifndef _uima_hpp_
#define	_uima_hpp_
//pragma ident	"@(#)uima.hpp	ubit:03.05.02"
#include <ubit/uelem.hpp>


/** Image Object.
 * Notes: 
 * - supported formats depend on which packages where linked to the Ubit lib
 *   (XPM and GIF formats are supported by default)
 * - a pathname prefix is implicitely added to the file names except if
 *   they start by / or .
 * - file names can start by a path variable (eg. "$mypath/whatever/file.gif")
 *   see UAppli.setImaPath() for details
 * - background transparency (ie. image shapes) is not supported by UIma
 *   but is supported by subclass UPix for XPM images
 */
class UIma: public UElem {
public:
  UIma(const UStr& filename);
  UIma(const char* filename = null);
  UIma(const char** xpm_data, u_modes = 0);
  /**<
   * creates an emty image and specifies its file name (or XPM data).
   * details:
   * - the image is not loaded immediately. It will be loaded when the
   *   image becomes visible on the screen (as a consequence, images 
   *   that are not shown are never loaded)
   * - use the load() or readFile() functions if you want to load
   *   the image in memory immediately.
   * - 'xpm_data' is <b>not</b> copied internally and must <b>not</b> be freed
   * see also:
   * - class UIma for supported file formats and the syntax of file names
   * - getStatus() : returns the loading status (see UFilestat)
   */

  friend UIma& uima(const UStr& filename)  {return *new UIma(filename);}
  friend UIma& uima(const char* filename)  {return *new UIma(filename);}
  friend UIma& uima(const char** xpm_data) {return *new UIma(xpm_data);}
  ///< creator shortcuts (see UIma::UIma() constructors for details)

  virtual ~UIma();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  virtual void set(const UStr& filename);
  virtual void set(const char* filename);
  virtual void set(const char** xpm_data);
  /**<
   * specifies the file name of an image (or XPM data).
   * details:
   * - the image is not loaded immediately. It will be loaded when the
   *   image becomes visible on the screen (as a consequence, images 
   *   that are not shown are never loaded)
   * - use the load() or readFile() functions if you want to load
   *   the image in memory immediately.
   * - 'xpm_data' is <b>not</b> copied internally and must <b>not</b> be freed
   * see also:
   * - class UIma for supported file formats and the syntax of file names
   * - getStatus() : returns the loading status (see UFilestat)
   */

  virtual int load();
  virtual int readFile(const UStr& filename);
  virtual int readFile(const char* filename);
  virtual int readData(const char** xpm_data);
  /**<
   * loads an image in memory.
   * details:
   * - the display is updated if the image is already visible.
   * - the loading status (see UFilestat) is returned
   * - load() loads the image previously specified by set() or the constructor
   * - use the set() function for postoned loading
   * - 'xpm_data' is <b>not</b> copied internally and must <b>not</b> be freed
   * see also: 
   * - class UIma for supported file formats and the syntax of file names
   * - getStatus() : returns the loading status (see UFilestat)
   * note:
   * - this function is equivalent to: set() + realize() + update() 
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  void copy(const UIma& ima2);
  /**< 
   * copies image content from ima2.
   * No action performed is ima2 is not realized (ie. not loaded)
   */

  void copyResized(const UIma& ima2, u_dim width, u_dim height);
  /**< 
   * copies image content from ima2 and rescales to size: width x height.
   * see: copy(const UIma&) for details
   */

  void copyRescaled(const UIma&, float xscale, float yscale);
   /**< 
   * copies image content from ima2 and rescales according to scaling factors.
   * see: copy(const UIma&) for details
   */

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

  const char* getNameChars() const;
  ///< returns the image name (if any, null otherwise)

  int getStatus() const;
  bool isLoaded() const;
  ///< returns the loading status of this image (see UFilestat)

  bool isRealized() const;
  ///< returns true if the image has been loaded and realized (see: realize()).

  bool isShaped() const;
  bool isTransparent() const {return isShaped();}
  ///< return true if the image background is shaped (so called "transparent").

  virtual u_dim getHeight() const;
  virtual u_dim getWidth()  const;
  ///< returns the image size (unknown if the image has not been loaded!).

  virtual void update(); 
  ///< update parents' views.

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  int realize(class UDisp& disp, bool force_reload = true) const;
  /**< 
   * [impl] allocates physical resources and loads the image in memory.
   * this function is automatically called when this image is displayed
   * for the first time: there no need to call it in client code.
   * details:
   * - returns the loading status  (see UFilestat)
   * - the 'disp' arg. can be the UAppli
   * - the 'force_reload' arg. specifies if the file must be reloaded:
   *   #- if false, previously loaded files are not reloaded
   *   #- if true files are always reloaded
   */

  std::list<class UNatIma*>& getNatImas() const {return natimas;}
  ///< [impl] returns internal implementation.

#ifndef NO_DOC
  friend class UBox;
  friend class UNatIma;
  friend class UPix;

  virtual void setImpl(const char* fname);
  virtual void setImpl(const char** xpm_data);
  virtual void cleanCache();
  virtual void getSize(UContext*, u_dim *w, u_dim *h) const;
  virtual void paint(UWinGraph&, UContext*, const URegion &r) const;

protected:
  mutable std::list<class UNatIma*> natimas;
  char *name;
  const char **data;
  mutable signed char stat;  // can have negative values
  mutable bool read_from_file, read_from_data, show_unknown_ima;

  class UNatIma* findImaInCache(class UNatDisp*, int lscale) const;
  class UNatIma* addImaInCache(class UNatDisp*, int lscale) const;
#endif
};

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

/** Pixmap Image object.
 * Works in the same way as UIma: see this class.
 * <br>Note: background transparency (ie. image shapes) is supported
 * for XPM images
 */
class UPix: public UIma {
public:
  static UPix 
    folder, ofolder, doc, diskette, plus, minus, ellipsis, rball, 
    select, cross, check, zoom, left, right, up, down, turn, question, 
    leftarrow, rightarrow, uparrow, downarrow,
    stop, ray, clock, bomb, windows, book, edit, eyes, colors;    
  /**<
   * Predefined Pixmap Images (small icons).
   * most of thse pixmaps were originally designed by Ludvig A. Norin
   * for the FVWM window manager. Please refer to file ufvwmpix.hh
   * for notice (more pixmaps are also available in this file)
   *
   * Note some of these pixmaps are shaped (= they have a "transparent"
   * background) */

  UPix(const UStr& filename);
  UPix(const char* filename = null);
  UPix(const char** xpm_data, u_modes = 0);
  /**< 
   * initializes a Pixmap Image object (that will be loaded at a later time).
   * see UIma::UIma for details
   * Notes: 
   * - background transparency (ie. image shapes) is supported for XPM images
   * - 'xpm_data' is <b>not</b> copied internally and must <b>not</b> be freed
   */

  friend UPix& upix(const UStr& filename)  {return *new UPix(filename);}
  friend UPix& upix(const char* filename)  {return *new UPix(filename);}
  friend UPix& upix(const char** xpm_data) {return *new UPix(xpm_data);}
  ///< creator shortcuts (see constructor for details)

  virtual ~UPix();

  // - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
  // implementation

  std::vector<class UNatPix*>& getNatPixs() const {return natpixs;}
  //< [impl]

#ifndef NO_DOC
  virtual void cleanCache();
  virtual void getSize(UContext*, u_dim *w, u_dim *h) const;
  virtual void paint(UWinGraph&, UContext*, const URegion &r) const;
#endif

private:
  mutable std::vector<class UNatPix*> natpixs;
};


/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */
#endif
