/* ==================================================== ======== ======= *
 *
 *  uugraph.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//#pragma ident	"@(#)uugraph.cpp		ubit:03.06.04"
#include <vector>
#include <config.h>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ufont.hpp>
#include <ucolor.hpp>
#include <uima.hpp>
#include <ustr.hpp>
#include <uview.hpp>
#include <ugraph.hpp>
#include <uwin.hpp>
#include <uappli.hpp>
#include <ctype.h>
#include <math.h>
#include <unatima.hpp>
#include <unatdisp.hpp>
#ifdef WITH_GL
#  include <GL/gl.h>
#  include <GL/glx.h>
#endif
#include <iostream>
using namespace std;


// polylines and polygons stored in the stack if card <= POLYCARD
// (heap memory used otherwise)
static const u_count POLYCARD   = 100;

#ifdef MONODISP
static bool last_buffer_is_locked = false;
static u_dim  last_buffer_width  = 0;
static u_dim  last_buffer_height = 0;
static UNatDisp* last_buffer_natdisp = null;
static UNatWin*  last_buffer = null;
#endif

/* !REMARQUE concernant X-Window et le parametre gcval.line_width :
 *
 * - width != 0 provoque des resultats quasi aleatoires
 *   a cause de pbms d arrondis pas franchement clairs (voir man pages)
 *   ainsi le dernier point apparait ou pas selon les cas!!!
 *
 * - DE PLUS: si line_width > 1 le trait deborde en epaisseur tantot
 *   a gauche, tantot a droite (s il est horizontal)
 *
 * ==> il est preferable de conserver line_width = 0 et d'imprimer 
 *     plusieurs lignes cote a cote quand on veut savoir PRECISEMENT
 *     ce que l'on va obtenir (en particulier pour les traits
 *     horizontaux et verticaux des widgets ou c'est au pixel pres)
 */
/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

struct UGraphElem {
  class UNatWin *dest;
  UGraphElem* prev;
  UGraph::GType gtype;
  int push_count;
  // true si dest local a ce GraphElem et doit etre desalloue
  bool own_dest;
  // where graphics are drawn. can be a Window or a Pixmap 
  // (when double or transp buffering is on)
  u_pos xwin, ywin;      // (x,y) offset from 'dest' origin 
  URegion clipbegin;
  UX_Rectangle cliprect;

  void setClip(UGraph* g, const URegion &r);
  void setClip(UGraph* g, u_pos x, u_pos y, u_dim width, u_dim height);

  static UGraphElem* pushElem(UGraph:: GType gtype, UGraphElem* _prev_gelem);
  static UGraphElem* popElem(UGraphElem* _gelem);
};

/* ==================================================== ======== ======= */
// pourrait etre optimise...

UGraphElem* UGraphElem::pushElem(UGraph::GType _gtype, 
				 UGraphElem* _previous_gelem) {
  UGraphElem* ge = new UGraphElem();
  ge->gtype = _gtype;
  ge->prev = _previous_gelem;
  ge->push_count = 0;
  ge->own_dest = false;
  return ge;
}

/* ==================================================== ======== ======= */

UGraphElem* UGraphElem::popElem(UGraphElem* _gelem) {
  UGraphElem* ge = _gelem->prev;
  delete _gelem;
  return ge;
}

/* ==================================================== ======== ======= */
#ifdef WITH_GL

inline void setGLClip(const UX_Rectangle& cliprect) {
  GLdouble left[] = {1, 0, 0.0, -cliprect.x};
  glClipPlane(GL_CLIP_PLANE0, left);
  glEnable(GL_CLIP_PLANE0);
  
  GLdouble right[] = {-1, 0, 0.0, cliprect.x + cliprect.width}; 
  glClipPlane(GL_CLIP_PLANE1, right);
  glEnable(GL_CLIP_PLANE1);
  
  GLdouble top[] = {0, -1, 0.0, -cliprect.y}; 
  glClipPlane(GL_CLIP_PLANE2, top);
  glEnable(GL_CLIP_PLANE2);
  
  GLdouble bottom[] = {0, 1, 0.0, cliprect.y + cliprect.height}; 
  glClipPlane(GL_CLIP_PLANE3, bottom);
  glEnable(GL_CLIP_PLANE3);
}

#endif
/* ==================================================== ======== ======= */

void UGraphElem::setClip(UGraph* g, const URegion &r) {
  //cerr <<  "setClip1" << " " << r.x << " " << r.x+r.width 
  //       << " " << -r.y-r.height << " " << -r.y <<  endl;
  cliprect.x = r.x;
  cliprect.y = r.y;
  cliprect.width  = r.width;
  cliprect.height = r.height;
#ifdef WITH_GL
  setGLClip(cliprect);
#else
  XSetClipRectangles(g->natdisp->getXDisplay(), g->natgraph->gc, 
		     0,0, &cliprect, 1, Unsorted);
#endif
}

/* ==================================================== ======== ======= */

void UGraphElem::setClip(UGraph* g, u_pos x, u_pos y, 
			 u_dim width, u_dim height) {
  //cerr <<  "setClip2" << " " << x << " " << x+width 
  //       << " " << -y-height << " " << -y <<  endl;
  cliprect.x = x;
  cliprect.y = y;
  cliprect.width  = width;
  cliprect.height = height;
#ifdef WITH_GL
  setGLClip(cliprect);
#else
  XSetClipRectangles(g->natdisp->getXDisplay(), g->natgraph->gc, 
		     0,0, &cliprect, 1, Unsorted);
#endif
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

// pour natwin (TOP ou SUB_WINDOW)
void UGraph::push(UGraph:: GType gtype, UNatWin* _dest, 
		  const URegion &clip, u_pos _xwin, u_pos _ywin) 
{
  gelem = UGraphElem::pushElem(gtype, gelem);
  //cerr << "((((( push dest = " << _dest << " gelem = " << gelem << "\n\n";
  gelem->dest = _dest;
  gelem->xwin = -_xwin;
  gelem->ywin = -_ywin;
  gelem->setClip(this, clip.x + gelem->xwin, clip.y + gelem->ywin,
  		 clip.width, clip.height);
  gelem->clipbegin.set(clip.x + gelem->xwin, clip.y + gelem->ywin,
		       clip.width, clip.height);
}

// pour DOUBLE ET TRANSP
void UGraph::push(UGraph:: GType gtype, bool alloc_buffer, const URegion &clip) 
{
  gelem = UGraphElem::pushElem(gtype, gelem);

  if (alloc_buffer) {
    gelem->dest = allocBuffer(clip.width, clip.height);
    gelem->own_dest = (gelem->dest != None);
  }
  else if (gelem->prev) {
    gelem->dest = gelem->prev->dest;
    gelem->own_dest = false;
  }
  else  {
    UError::error("internal@UGraph::push",
		  "wrong operation: empty graphic stack");
    push(UGraph::WIN, wg->natwin, clip, 0, 0);
    return;
  }

  //cerr << "((((( push transp/dble (type=" << gtype << ") dest: " << gelem->dest
  //     << " gelem = " << gelem << "\n \n";

  if (gelem->dest != None) {
    gelem->xwin  = -clip.x;
    gelem->ywin  = -clip.y;
    gelem->setClip(this, 0, 0, clip.width, clip.height); 
    gelem->clipbegin.set(0, 0, clip.width, clip.height); 
  }
  else {
    //pas assez de memoire (ou taille nulle) => simple recopie du precedent
    string mesg = "size: " + clip.width; mesg += + "x" + clip.height;

    UError::error("warning@UGraph::push","can't allocate X buffer: ",mesg);
    if (gelem->prev) {
      // essayer de sauver la situation en recopiant le precedent
      *gelem = *(gelem->prev);
      gelem->gtype = gtype;
      gelem->push_count = 0;
      gelem->own_dest = false;
    }
    else {
      gelem = UGraphElem::popElem(gelem);
      push(UGraph::WIN, wg->natwin, clip, 0, 0);
    }
  }
}

void UGraph::pop() {
  if (!gelem) {
    UError::error("internal@UGraph::pop","graphic stack out of range");
    return;
  }

  if (gelem->own_dest && gelem->dest != None)
    releaseBuffer(gelem->dest);

  gelem = UGraphElem::popElem(gelem);

  if (gelem) {      //retablir le clip
#ifdef WITH_GL
    setGLClip(gelem->cliprect);
#else
    XSetClipRectangles(natdisp->xdisplay, natgraph->gc, 
		       0,0, &(gelem->cliprect), 1, Unsorted);
#endif
  }
}

void UGraph::popAll() {
  UGraphElem* ge = gelem;
  while (ge) {ge = UGraphElem::popElem(ge);}
  gelem = null;
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

UGraph::UGraph() {
  boxview   = null;
  gelem     = null;
  natgraph  = null;
  natdisp   = null;
  wg = null;
  blend_mode = OPAQUE;
  alpha = 0.;
}

UGraph::UGraph(UView* view) {
  constructs(view);
  // mod:7aug03: cette fct redefinit le clip deja defini dans construct
  // la consequence est que le graphique sort de la zone visible de la
  // lorsque celle-ci est clippee (par ex. par une scrollpane)
  //if (view) setWinClip(*view);
}

UGraph::UGraph(UEvent* e) {
  if (!e) constructs(null);
  else {
    constructs(e->getView());
    if (e->sp.redrawStatus) setWinClip(e->sp.redrawClip);
  }
}

UGraph::UGraph(UEvent& e) {
  constructs(e.getView());
  if (e.sp.redrawStatus) setWinClip(e.sp.redrawClip);
}

void UGraph::constructs(UView* view) {
  if (!view) {
    boxview  = null;
    gelem    = null;
    natgraph = null;
    natdisp = null;
    wg = null;
    blend_mode = OPAQUE;
    alpha = 0.;
    UError::error("UGraph::UGraph","null UView given as an argument");
    return;	// exception?
  }

  boxview = view;
  wg = &(view->wg());
  if (!wg) {
    UError::error("UGraph::construct","Null wingraph");
    constructs(null);
    return;
  }

  natdisp = wg->disp->natdisp;
  //own_gelem = true;
  blend_mode = wg->blend_mode;
  alpha = wg->alpha;  

  // shared UNatGraphs(s) [one for each UDisp]
  // !att: ne convient pas si multi-threads du reaffichage standard
  // !att egalement aux couleurs et fonts qui peuvent varier suivant displays

  if (wg->disp->natdisp->getClientGraph()->isLocked()) {
    // deja pris => creer un nouveau NatGraph
    natgraph = new UNatGraph(natdisp);
    natgraph->lock(this);
  }
  else {
    // utiliser le NatGraph standard
    natgraph = natdisp->getClientGraph();
    natgraph->lock(this);
    //...reset eventuel du GC
    natgraph->reset();
  }

  // ajuster la  zone de clipping a la PARTIE VISIBLE de la vue
  // et affecter la 'bonne' destination
  UEvent e2(UEvent::search, null, view->getHardwinView(), null);
  e2.locateSource(view);

  gelem = UGraphElem::pushElem(CLIENT, wg->gelem);
  gelem->dest = wg->gelem ? wg->gelem->dest : wg->natwin;

  if (!wg->gelem 
      || wg->gelem->gtype == WIN
      || wg->gelem->gtype == SUBWIN) {
    gelem->xwin = view->x;
    gelem->ywin = view->y;
    if (e2.sp.redrawStatus) {
      gelem->clipbegin = e2.sp.redrawClip;
      gelem->setClip(this, e2.sp.redrawClip);
    }
  }

  else if (wg->gelem->gtype == DOUBLEBUF
	   || wg->gelem->gtype == BLEND) {
    gelem->xwin = wg->gelem->xwin + view->x;
    gelem->ywin = wg->gelem->ywin + view->y;
    if (e2.sp.redrawStatus) {
      gelem->clipbegin.set(0, 0,
			   e2.sp.redrawClip.width, 
			   e2.sp.redrawClip.height);
      gelem->setClip(this, 0, 0,
      		     e2.sp.redrawClip.width, e2.sp.redrawClip.height);
    }
  }

  else {			// ?? sert a rien ????
    cerr << "UGraph test mode" <<endl;
    gelem->xwin = wg->gelem->xwin + view->x;
    gelem->ywin = wg->gelem->ywin + view->y;
    if (e2.sp.redrawStatus) {
      gelem->clipbegin.set(e2.sp.redrawClip.x + wg->gelem->xwin,
			   e2.sp.redrawClip.y + wg->gelem->ywin,
			   e2.sp.redrawClip.width, 
			   e2.sp.redrawClip.height);
      gelem->setClip(this, 
		     e2.sp.redrawClip.x + wg->gelem->xwin,
		     e2.sp.redrawClip.y + wg->gelem->ywin,
		     e2.sp.redrawClip.width, 
		     e2.sp.redrawClip.height);
      //...;
    }
  }
}

UGraph::~UGraph() {
  // a faire en premier car depend de natgraph
  if (gelem) pop();

  // !ATT: pas question de detruire les NatGraphs partages !
  if (natgraph && natgraph != wg->disp->natdisp->getSharedGraph()) {
    if (natgraph == wg->disp->natdisp->getClientGraph()) 
      natgraph->unlock();
    else {
      if (natgraph->gc) XFreeGC(natdisp->xdisplay, natgraph->gc);
      delete natgraph; 
      natgraph = null;
    }
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NOTE POUR UWinGraph UNIQUEMENT:
// - on ne doit JAMAIS changer thickness qui DOIT donc TOUJOURS valoir 0
//   ==> lignes d'epaisseur 1 avec affichage rapide optimise
// - raison: voir note sur pbm de precision en haut de fichier

UWinGraph::UWinGraph(class UWin *win) : UGraph() {
  wg      = this;
  disp    = null;
  natdisp = null;
  hardwin = win;
  natwin  = new UNatWin();
  //own_gelem = false;
  gelem = null;
}

UWinGraph::~UWinGraph() {
  hardwin = null;  // !ATT: surtout ne PAS detruire hardwin!
  popAll();
  gelem = null;

  //NB: le XDestroyWindow est necessaire car ~UNatWin ne detruit pas xwin
  // (xwin etant soit un XPixmap soit une XWindow). De plus PROBLEME !!!!! 
  // la destruction des XWindow est recursive ce qui peut            !!!!!
  // aboutir a des destructions multiples de la meme XWindow         !!!!!

  if (natwin->xwin != None) 
    XDestroyWindow(natdisp->xdisplay, natwin->xwin);
  natwin->xwin = null;
  delete natwin;
  natwin = null;

  // idiot puisque natgraph partage!
  // if (natgraph == natdisp->wingraph)  natgraph->unlock();

  // !att: appel ulterieur de ~UGraph() !
  natgraph = null;
}

// initialize le WinGraph. dot obligatoirement etre appelee avant usage
// NB: cette fct ne cee pas la X Wndow. ce sera fait uterieurement par
// UWin::realize la 1ere fois que la fenetre apparaitra

bool UWinGraph::init(UDisp *d, UView* win_view) {
  if (!d || !win_view) {
    UError::error("fatal@UWinGraph::init",UError::Cant_realize_wingraph);
    return false;
  }
  disp = d;
  natdisp = disp->getNatDisp();

  // !IMPORTANT:
  // Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
  // sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
  // seront incorrectes
  // (noter que par contre, UGraph permet de varier boxview, ce qui permet
  // (de travailler en ccords relatives)
  boxview = win_view;

  // shared UNatGraphs(s) [one for each UDisp]
  // !att: ne convient pas si multi-threads du reaffichage standard
  // !att egalement aux couleurs et fonts qui peuvent varier suivant displays
  natgraph = disp->natdisp->getSharedGraph();
  //natgraph->lock(this); idiot puisque natgraph partage!

  // Note: on ne change pas le Clip: il faudra le faire explicitement
  // sur le natgraph partage !
  return true;
}

bool UWinGraph::isWinRealized() const {
  return (natwin && natwin->isRealized());
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

void UWinGraph::begin(const URegion& clip) {
  // deja en topwin: juste incrementer
  //if (gelem && gelem->gtype == WIN) {gelem->push_count++; return;}
  push(WIN, natwin, clip, 0, 0);
}
 
void UWinGraph::beginSubwin(const URegion& clip, u_pos _xwin, u_pos _ywin) {
  //cas different de TOP_WINDOW: faire push() a chaque fois
  push(SUBWIN, natwin, clip, _xwin, _ywin);
}

void UWinGraph::beginDoublebuf(const URegion& clip) {
#ifdef WITH_GL
  // ATT: PAS de doublebuf en mode GL
  // (ne PAS mettre en mode DOUBLEBUF car traitements et clip incorrects)
  //if (gelem && gelem->gtype == WIN) {gelem->push_count++; return;}
  push(WIN, natwin, clip, 0, 0);
#else
  // deja en doublebuf: juste incrementer
  if (gelem && gelem->gtype == DOUBLEBUF) {gelem->push_count++; return;}
  push(DOUBLEBUF, true, clip);  // true => alloc buffer
#endif
}

void UWinGraph::beginBlend(const URegion& clip, float _alpha) {
#ifdef WITH_GL
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glEnable(GL_BLEND);
  blend_mode = TRUE_BLEND;
  alpha = _alpha;
#else
  // deja en transpbuf: juste incrementer
  if (gelem && gelem->gtype == BLEND) {gelem->push_count++; return;}
  blend_mode = PSEUDO_BLEND;
  alpha = _alpha;
  push(BLEND, true, clip); // true => alloc buffer
#endif
}

void UWinGraph::endBlend() {   // API A REVOIR
#ifdef WITH_GL
  glDisable(GL_BLEND);
#else
  end();
#endif
}

void UWinGraph::blendColor(const URegion& clip, float _alpha) {
#ifdef WITH_GL
  UError::error("warning@UWinGraph::blendColor",
		"this function cant be used in GL mode");
#else
  // deja en transpbuf: juste incrementer
  if (gelem && gelem->gtype == BLEND) {gelem->push_count++; return;}
  blend_mode = UNIFORM_BLEND;
  alpha = _alpha;
  push(BLEND, false, clip);  // false => do not alloc buffer
  end();
#endif
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UWinGraph::end() {
  if (!gelem) {
    UError::error("UGraph::end",UError::Unbalanced_request);
    return;
  }

  switch (gelem->gtype) {

  case NONE:
    break;

  case CLIENT:  // CORRECT???
    break;

  case WIN:
    //if (gelem->push_count > 0) {gelem->push_count--; return;}
    pop();
#ifdef WITH_GL                //[MODE OGL_DOUBLEBUF]
    // att: ne faire le sawp que pour le top begin()/end()
    if (disp->conf.double_buffering && gelem && gelem->prev == null) {
      glXSwapBuffers(natdisp->xdisplay, natwin->xwin);
    }
#endif
    break;

  case SUBWIN:
    //cas different: pop a chaque fois
    pop();
    break;

  case DOUBLEBUF:
#ifdef WITH_GL
    UError::error("internal@UGraph::end",
		  "soft doublebuf: should not occur in GL mode");
#else
    if (gelem->push_count > 0) {gelem->push_count--; return;}

    //own_dest : dest reelement allouee pour ce gelem
    if (gelem->own_dest && gelem->dest->xwin != None) {
      gelem->setClip(this, boxview->getXwin(), boxview->getYwin(), 
		     boxview->getWidth(), boxview->getHeight());
      XCopyArea(natdisp->xdisplay, 
		gelem->dest->xwin,  //from
		natwin->xwin,       //to
		natgraph->gc,
		0, 0, //r.width, r.height,
		gelem->clipbegin.width, gelem->clipbegin.height, 
		-gelem->xwin, -gelem->ywin);
    }
#endif
    pop();
    break;

  case BLEND:{
    if (gelem->push_count > 0) {gelem->push_count--; return;}
    
    // but du jeu: faire un blending de layer1 et layer2 et mettre
    // le resultat dans layer1
    if (!gelem->prev) {
      UError::error("internal@UWinGraph::end", 
		    "soft blend: no first layer");
      pop();
      return;
    }
    UGraphElem* layer1 = gelem->prev;
    UGraphElem* layer2 = gelem;

    if (layer1 == layer2)
      UError::error("internal@UWinGraph::end", 
		    "soft blend: layer1 = layer2!");
    blendLayers(layer1, layer2);
    pop();
  } break;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UWinGraph::blendLayers(UGraphElem* layer1, UGraphElem* layer2) {
    XImage *xima1 = null, *xima2 = null;
  
    //own_dest : dest reelement allouee pour ce gelem
    if (/* gelem->own_dest && pas vrai si scalar_mode*/
	gelem->dest->xwin != None
	&& layer1->dest && layer2->dest) {

      URegion clip2(-layer2->xwin, -layer2->ywin, 
		    layer2->clipbegin.width, layer2->clipbegin.height);
      int clipstat = 0;
      u_pos xshift_in_clip = 0, yshift_in_clip = 0;

      if (layer1->gtype != UGraph::DOUBLEBUF) {
	clipstat = clip2.setInter(boxview); //securite
	xshift_in_clip = boxview->getXwin() - -layer2->xwin;
	yshift_in_clip = boxview->getYwin() - -layer2->ywin;
      }
      else {
	URegion clip1(-layer1->xwin, -layer1->ywin, 
		      layer1->clipbegin.width, layer1->clipbegin.height);
	clipstat = clip2.setInter(clip1);
	clip2.x -= -layer1->xwin;
	clip2.y -= -layer1->ywin;
	if (clip2.x < 0) clip2.x = 0;
	if (clip2.y < 0) clip2.y = 0;
	
	xshift_in_clip = -layer1->xwin - -layer2->xwin; 
	yshift_in_clip = -layer1->ywin - -layer2->ywin; 
      }
      
      if (xshift_in_clip < 0) xshift_in_clip = 0;
      if (yshift_in_clip < 0) yshift_in_clip = 0;
      
      if (clipstat != 0) {
	// UXtry sert a catcher les X Errors safely :
	// le pbm ici c'est que les xwin ne sont pas forcement mappees
	// et completement visible (or X plante dans ces cas)
	UXtry xtry;
	xima1 = XGetImage(natdisp->xdisplay, 
			  layer1->dest->xwin,
			  clip2.x, clip2.y,
			  clip2.width, clip2.height, 
			  AllPlanes, ZPixmap);
      	if (blend_mode != UNIFORM_BLEND)
	xima2 = XGetImage(natdisp->xdisplay,
			  layer2->dest->xwin,
			  xshift_in_clip, yshift_in_clip,
			  clip2.width, clip2.height,
			  AllPlanes, ZPixmap);
      	//if (!xtry.status()) printf("can't get X Image \n");
      } //end of xtry scope

      if (!xima1 || (blend_mode != UNIFORM_BLEND && !xima2)) {
	//uerror('d', "UWinGraph::end", "can't get X Image");
	return;
      }

      //necessaire pour le XPutImage
      //gelem->setClip(this, boxview->getXwin(), boxview->getYwin(), 
      //               boxview->getWidth(), boxview->getHeight());
      layer1->setClip(this, clip2.x, clip2.y, clip2.width, clip2.height);
      /*test    
	XPutImage(natdisp->xdisplay, natwin->xwin, natgraph->gc, 
		xima1,
		0, 0,	   //from in source ima
		0, 0,      //to in target view
		xima1->width, xima1->height);
        XPutImage(natdisp->xdisplay, natwin->xwin, natgraph->gc, 
		xima2,
		0, 0,	     //from in source ima
		xima1->width+10,0,      //to in target view
		xima2->width, xima2->height);
      */
      if (blend_mode == UNIFORM_BLEND)
	UNatIma::blendImage(disp->natdisp, xima1, natgraph->color, alpha);
      else 
	//le resultat du blend est dans xima1
	UNatIma::blendImages(disp->natdisp, xima1, xima2, alpha);

      XPutImage(natdisp->xdisplay, 
		layer1->dest->xwin,
		natgraph->gc, 
		xima1,
		xshift_in_clip, yshift_in_clip,  //from in source ima2
		//-layer2->xwin, -layer2->ywin,    //to in target layer1
		clip2.x, clip2.y,
		clip2.width, clip2.height);      // size
      
      if (xima1) XDestroyImage(xima1);
      if (xima2) XDestroyImage(xima2); 
    }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// !!ATTENTION: pas le meme SetClip pour WinGraph et Graph !!
// note: UWinGraph: clip absolu dans la Window View

void UWinGraph::setClip(const URegion &r) {
  gelem->setClip(this, r.x + gelem->xwin, r.y + gelem->ywin, 
		 r.width, r.height);
}

void UWinGraph::setClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  gelem->setClip(this, x + gelem->xwin, y + gelem->ywin, 
		 width, height);
}

// setClip and setWinClip are similar for this class: 
// -- coordinates are relative to the origin of the WINDOW
// -- the view is no taken into account

void UWinGraph::setWinClip(const URegion &r) { 
  setClip(r);
}

void UWinGraph::setWinClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  setClip(x, y, width, height);
}

/* ==================================================== [Elc:03] ======= */
// UGraph: noter que le Clip est relatif a la View attachee
// - NB: verifier que le Clip est bien inclus dans la View et prendre le min
// -- !ATT: on ne tient pas compte dy cas ou r.x ou r.y  < 0  !!!

void UGraph::setClip(const URegion &r) {
  URegion xrect;
  if (r.x >= boxview->getWidth() || r.y >= boxview->getHeight()) {
    xrect.width = 0; xrect.height = 0;
  }
  else {
    // keep the min
    if (r.x + r.width > boxview->getWidth())
      xrect.width = boxview->getWidth() - r.x;
    else
      xrect.width = r.width;

    if (r.y + r.height > boxview->getHeight())
      xrect.height = boxview->getHeight() - r.y;
    else
      xrect.height = r.height;
  }

  // !ATT: on ne tient pas compte dy cas ou r.x ou r.y  < 0 
  //xrect.x = boxview->getXwin() + r.x;
  //xrect.y = boxview->getYwin() + r.y;

  UGraph::GType gtype = (gelem->prev ? gelem->prev->gtype : UGraph::NONE);
  
  if (gtype == DOUBLEBUF || gtype == BLEND) {
    gelem->setClip(this, r.x + gelem->xwin, r.y + gelem->ywin,
		   xrect.width, xrect.height);
  }
  else {
    gelem->setClip(this, boxview->getXwin() + r.x, boxview->getYwin() + r.y,
		   xrect.width, xrect.height);
  }
}

void UGraph::setClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  URegion r(x, y, width, height);
  setClip(r);
}

void UGraph::setWinClip(const URegion &r) {
  URegion r2(r.x - boxview->getXwin(), r.y - boxview->getYwin(),
	    r.width, r.height);
  setClip(r2);
}

void UGraph::setWinClip(u_pos x, u_pos y, u_dim width, u_dim height) {
  URegion r2(x - boxview->getXwin(), y - boxview->getYwin(),
	    width, height);
  setClip(r2);
}

// ==== Coordinates ======================================================

// conversion from local View coords. to global Window coords.
u_pos UGraph::XToXwin(u_pos x_in_view) {return x_in_view + boxview->getXwin();}
u_pos UGraph::YToYwin(u_pos y_in_view) {return y_in_view + boxview->getYwin();}

// conversion from global Window coords. to local View coords.
u_pos UGraph::XwinToX(u_pos x_in_win) {return x_in_win - boxview->getXwin();}
u_pos UGraph::YwintoY(u_pos y_in_win) {return y_in_win - boxview->getYwin();}

UDisp*  UGraph::getDisp() const  {return wg ? wg->disp : null;}
UAppli* UGraph::getAppli() const {return wg ? &(wg->disp->getAppli()) :null;}

class UNatWin* UGraph::getNatWin() const {return wg ? wg->natwin : null;}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UNatWin* UGraph::allocBuffer(u_dim width, u_dim height) {
#ifdef MONODISP
  if (width == last_buffer_width && height == last_buffer_height
      && !last_buffer_is_locked && natdisp == last_buffer_natdisp) {
    last_buffer_is_locked = true;
    return last_buffer;
  }
  else {
    if (last_buffer_is_locked) {
      return createBuffer(width, height);
    }
    else {
      last_buffer_is_locked = true;
      // att: deleteBuffer(last_buffer); incoherent en multidisp!
      //      (car le natdisp n'est pas tjrs celui de la creation)
      deleteBuffer(last_buffer);
      last_buffer_width   = width;
      last_buffer_height  = height;
      last_buffer_natdisp = natdisp;
      return (last_buffer = createBuffer(width, height));
    }
  }
#else
  return createBuffer(width, height);
#endif
}

void UGraph::releaseBuffer(UNatWin* buf)  {
#ifdef MONODISP
  if (buf == last_buffer) {
    last_buffer_is_locked = false;
  }
  else {
    deleteBuffer(buf);
  }
#else
  deleteBuffer(buf);
#endif
}

/* ==================================================== ======== ======= */

UNatWin* UGraph::createBuffer(u_dim width, u_dim height) {
  if (width <= 0 || height <= 0) return null;
  //cerr << "createBuffer " << width << " "<< height
  //     << " xwin " << altbuf->xwin << " natdisp " << getNatDisp() << endl;

  UNatWin *altbuf = new UNatWin();
  if (!altbuf) return null;

  altbuf->xwin = XCreatePixmap(natdisp->xdisplay, natdisp->getXWindow(),
			       width, height, natdisp->getDepth());
  if (altbuf->xwin == None) {
    delete altbuf;
    return null;
  }
  else return altbuf;
}

void UGraph::deleteBuffer(UNatWin* altbuf)  {
  if (!altbuf) return;
  //cerr << "deleteBuffer " << " xwin " << altbuf->xwin
  //     << " natdisp " << getNatDisp() << endl;

  //NB: le XFreePixmap est necessaire car ~UNatWin ne detruit pas xwin
  // (xwin etant soit un XPixmap soit une XWindow)
  if (altbuf->xwin != None) {
    // NOTE:                                   !!! TEMPORAIRE : A REVOIR !!!
    XFreePixmap(natdisp->xdisplay, altbuf->xwin);
    altbuf->xwin = None;
    delete altbuf;
  } 
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// PaintMode is the default (overwrites pixels)

void UGraph::setPaintMode() {
  natgraph->xor_mode = false;
#ifdef WITH_GL
  //ca ne change pas le mode du GC! possible en GL ???
  natdisp->setGLColor(natgraph->color);
#else
  XGCValues gcval;
  gcval.foreground = natgraph->color;
  gcval.function = GXcopy;
  XChangeGC(natdisp->xdisplay, natgraph->gc, GCFunction|GCForeground, &gcval);
#endif
}

/* ==================================================== ======== ======= */
// setXORMode initiates drawing in XOR kode

bool UGraph::isXORMode() {return natgraph->xor_mode;}

void UGraph::setXORMode() {
  natgraph->xor_mode = true;
#ifdef WITH_GL
  //ca ne change pas le mode du GC! possible en GL ???
  natdisp->setGLColor(natgraph->color ^ natgraph->bgcolor);
#else
  // seul le champ 'gcval.foreground' sert pour le XOR
  // cette valeur est elle meme un XOR de foreground ET alternate_color
  XGCValues gcval;
  gcval.foreground = natgraph->color ^ natgraph->bgcolor;
  gcval.function = GXxor;
  XChangeGC(natdisp->xdisplay, natgraph->gc, GCFunction|GCForeground, &gcval);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::setColor(const UColor* c) {
  if (c) setColor(*c);
}

void UGraph::setColor(const UColor& c) {
  if (!wg) return;
  natgraph->color = wg->disp->natdisp->getColor(c);

#ifdef WITH_GL
  if (blend_mode == TRUE_BLEND) {
    GLubyte alphab = (GLubyte)(alpha * 255);
    if (natgraph->xor_mode) 
      natdisp->setGLColor(natgraph->color ^ natgraph->bgcolor, alphab);
    else natdisp->setGLColor(natgraph->color, alphab);
  }
  else {
    //ca ne change pas le mode du GC! possible en GL ???
    if (natgraph->xor_mode) 
      natdisp->setGLColor(natgraph->color ^ natgraph->bgcolor);
    else natdisp->setGLColor(natgraph->color);
  }

#else
  XGCValues gcval;
  if (natgraph->xor_mode) {
    gcval.foreground = natgraph->color ^ natgraph->bgcolor;
  }
  else gcval.foreground = natgraph->color;
  XChangeGC(natdisp->xdisplay, natgraph->gc, GCForeground, &gcval);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::setBgcolor(const UColor* c) {
  if(c) setBgcolor(*c);
}

void UGraph::setBgcolor(const UColor& c) {
  if (!wg) return;
  natgraph->bgcolor = wg->disp->natdisp->getColor(c);

#ifdef WITH_GL
  //ca ne change pas le mode du GC! possible en GL ???
  if (natgraph->xor_mode) 
    natdisp->setGLColor(natgraph->color ^ natgraph->bgcolor);

#else
  if (natgraph->xor_mode) {
    XGCValues gcval;
    gcval.foreground = natgraph->color ^ natgraph->bgcolor;
    XChangeGC(natdisp->xdisplay, natgraph->gc, GCForeground, &gcval);
  }
  else  {              // NEW: Aug02: oublie... (cas else seulement?)
    XGCValues gcval;
    gcval.background = natgraph->bgcolor;
    XChangeGC(natdisp->xdisplay, natgraph->gc, GCBackground, &gcval);
  }
#endif
}

/* ==================================================== ======== ======= */

void UGraph::setCursor(const UCursor* c) {
  if (c) setCursor(*c);
  else {       // devrait definir UCursor::none ou normal
    if (!wg) return;
    XDefineCursor(natdisp->xdisplay, wg->natwin->xwin, None);
  }
}

void UGraph::setCursor(const UCursor& c) {
  if (!wg) return;
  Cursor curs = wg->disp->natdisp->getCursor(c);
  // NB: curs =eventuellement None
  XDefineCursor(natdisp->xdisplay, wg->natwin->xwin, curs);
}

/* ==================================================== ======== ======= */

int UGraph::getThickness() {return natgraph->thickness;}

// Changes line thickness when drawing lines, etc.
// - the default value is 0, which sets thickness to 1 pixel
//   and optimizes drawing routines
// !!a revoir avec UPen !!

void UGraph::setThickness(int th) {
#ifdef WITH_GL
  //??? a FAIRE         !!!!
#else
  XGCValues gcval;// !!a revoir avec UPen !!
  natgraph->thickness = th;
  gcval.line_width = th;
  XChangeGC(natdisp->xdisplay, natgraph->gc, GCLineWidth, &gcval);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::setFont(const UFont* f) {
  if (f) setFont(*f);
}

void UGraph::setFont(const UFont& f) {
  UFontDesc fd(f);
  setFont(fd);
}

void UGraph::setFont(const UFontDesc& fd) {
  if (!wg) return;

  natgraph->font = wg->disp->natdisp->getFont(fd);
  if (!natgraph->font) {
    UError::error("UGraph::setFont", UError::Null_argument);
    return;
  }
  natgraph->font_styles = fd.styles;
  
#ifdef WITH_GL
  natgraph->glFontList = wg->disp->natdisp->getGLFont(fd);
  if (!natgraph->glFontList) {
    UError::error("UGraph::setFont", UError::Null_argument);
    return;
  }
#else
  XGCValues gcval;
  gcval.font = natgraph->font->fid;
  XChangeGC(natdisp->xdisplay, natgraph->gc, GCFont, &gcval);
#endif
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */
// NOTE: UWinGraph en coords ABSOLUES dans la Window
//       et UGraph en coords LOCALES de la View

// !IMPORTANT:
// Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
// sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
// seront incorrectes (c'est le cas de clearRect, drawPolyline, drawPolygon..)

void UGraph::drawPoint(u_pos x, u_pos y) {
#ifdef WITH_GL
  glBegin(GL_POINTS);
  glVertex2i(x, -y);
  glEnd();
#else
  XDrawPoint(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
	     gelem->xwin + x, gelem->ywin + y);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawLine(u_pos x1, u_pos y1, u_pos x2, u_pos y2) {
#ifdef WITH_GL
  glBegin(GL_LINE_STRIP);
  glVertex2i(gelem->xwin + x1, -(gelem->ywin + y1));
  glVertex2i(gelem->xwin + x2, -(gelem->ywin + y2));
  glEnd();
#else
  XDrawLine(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
	    gelem->xwin + x1, gelem->ywin + y1, 
	    gelem->xwin + x2, gelem->ywin + y2);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawString(const UStr* s, u_pos x, u_pos y) {
  if (s) drawString(s->chars(), s->length(), x, y);
}

void UGraph::drawString(const UStr& s, u_pos x, u_pos y) {
  drawString(s.chars(), s.length(), x, y);
}

void UGraph::drawString(const char* str, int str_len, u_pos x, u_pos y) {
  if (!str || str_len <= 0) return;

  if (str[str_len-1] == '\n') {  // ne pas afficher le \n final
    if (str_len == 1) return;
    else str_len--;
  }

  // draw underline, overline, strike-through
  
  if (natgraph->font_styles > UFont::BOLD+UFont::ITALIC) {
    u_pos ypos = 0;
    u_pos x1 = gelem->xwin + x;
    u_pos x2 = gelem->xwin + x + XTextWidth(natgraph->font, str, str_len);
    
    if (natgraph->font_styles & UFont::UNDERLINE) {
      ypos = gelem->ywin + y + natgraph->font->ascent + natgraph->font->descent/2;
      drawLine(x1, ypos, x2, ypos);
    }

    if (natgraph->font_styles & UFont::OVERLINE) {
      ypos = gelem->ywin + y;
      drawLine(x1, ypos, x2, ypos);
    }
    
    if (natgraph->font_styles & UFont::STRIKETHROUGH) {
      ypos = gelem->ywin + y + 3 * natgraph->font->ascent / 4;
      drawLine(x1, ypos, x2, ypos);
    }
  }

#ifdef WITH_GL

  // ATT: UFont::FILL a revoir !!
  if (natgraph->font_styles & UFont::FILL)
    setColor(UColor::red);

  int charpos_begin, charpos_end;
  u_pos xpos_begin, xpos_end;

  if (!getClippedText(str, str_len,
		      gelem->xwin + x,
		      -(gelem->ywin + y + natgraph->font->ascent),
		      charpos_begin, charpos_end, xpos_begin, xpos_end))
    return;
  
  if (charpos_begin < 0 || charpos_end < 0 ||charpos_end < charpos_begin)
    return;

  glPushAttrib(GL_LIST_BIT | GL_CURRENT_BIT);
  glRasterPos2i(xpos_begin, 
		-(gelem->ywin + y + natgraph->font->ascent));
  glListBase(natgraph->glFontList);
  glCallLists(charpos_end - charpos_begin + 1, 
	      GL_UNSIGNED_BYTE, str + charpos_begin);
  glPopAttrib();  

#else

  if (natgraph->font_styles & UFont::FILL) {
    XDrawImageString(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		     gelem->xwin + x, 
		     gelem->ywin + y + natgraph->font->ascent, 
		     str, str_len); 
  }
  else {
    XDrawString(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		gelem->xwin + x, 
		gelem->ywin + y + natgraph->font->ascent, 
		str, str_len); 
  }
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawRect(u_pos x, u_pos y, u_dim w, u_dim h) {
#ifdef WITH_GL
  glBegin(GL_LINE_LOOP);
  glVertex2i(gelem->xwin + x    , -(gelem->ywin + y));
  glVertex2i(gelem->xwin + x + w, -(gelem->ywin + y));
  glVertex2i(gelem->xwin + x + w, -(gelem->ywin + y + h));
  glVertex2i(gelem->xwin + x    , -(gelem->ywin + y + h));
  glEnd();
#else
  XDrawRectangle(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
		 gelem->xwin + x, gelem->ywin + y, w, h);
#endif
}

void UGraph::fillRect(u_pos x, u_pos y, u_dim w, u_dim h) {
#ifdef WITH_GL
  glRecti(gelem->xwin + x    , -(gelem->ywin + y), 
  	  gelem->xwin + x + w, -(gelem->ywin + y + h));
#else
  XFillRectangle(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
		 gelem->xwin + x, gelem->ywin + y, w, h);
#endif
}

// remplit le border ?
void UGraph::fillView() {
  fillRect(0, 0, boxview->getWidth(), boxview->getHeight());
}

/* ==================================================== ======== ======= */

void UGraph::drawEllipse(u_pos x, u_pos y, u_dim w, u_dim h) {
  drawArc(x, y, w, h, 0., 360.);
}

void UGraph::drawArc(u_pos x, u_pos y, u_dim w, u_dim h,
		     float angle1, float angle2) {
#ifdef WITH_GL
  float rw    = w/2.0;
  float rh    = h/2.0;
  float delta = 1.0*M_PI/180 * (angle1-angle2);
  float alpha = 1.0+M_PI/180 * angle1;
  int step    = int(((w+h) * (w+h) * (angle1-angle2)) / (90*w*h));
  if (step < 0) step = -step;

  glBegin(GL_LINE_STRIP);
  for(int i=0; i<=step; i++)
    glVertex2f(rw*(cos(alpha+delta*i/step)+1) + gelem->xwin+x, 
	       rh*(sin(alpha+delta*i/step)-1) - gelem->ywin-y);
  glEnd();
#else
  XDrawArc(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
	   gelem->xwin + x, gelem->ywin + y, w, h, 
	   (int)(angle1 * 64.), (int)(angle2 * 64.));     // !!att: au *64 !
#endif
};


void UGraph::fillArc(u_pos x, u_pos y, u_dim w, u_dim h,
		     float angle1, float angle2) {
#ifdef WITH_GL
  float rw    = w/2.0;
  float rh    = h/2.0;
  float delta = 1.0*M_PI/180 * (angle1-angle2);
  float alpha = 1.0+M_PI/180 * angle1;
  int step    = int(((w+h) * (w+h) * (angle1-angle2)) / (90*w*h));
  if (step<0) step = -step;

  glBegin(GL_POLYGON);
  for(int i=0; i<=step; i++)
    glVertex2f(rw*(cos(alpha+delta*i/step)+1)+x, 
	       -(rh*(sin(alpha+delta*i/step)-1)+y));
  glEnd();
#else
  XFillArc(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
	   gelem->xwin + x, gelem->ywin + y, w, h, 
	   (int)(angle1 * 64.), (int)(angle2 * 64.));  // !!att: au *64 !
#endif
};

/* ==================================================== [Elc:03] ======= */
// ==== Clearing routines ================================================
// Note: these functions use the background Color
//       and are NOT altered by the XOR mode

void UGraph::clearView(bool repaint_view) {
  if (repaint_view) {
    if (boxview) boxview->updatePaint();
  }
  else clearRect(0, 0, boxview->getWidth(), boxview->getHeight());
}

/* ==================================================== ======== ======= */

void UGraph::clearRect(u_pos x, u_pos y, u_dim w, u_dim h, 
		       bool repaint_view) {
  if (repaint_view) {
    URegion rr(x, y, w, h);
    if (boxview) boxview->updatePaint(&rr);
  }
  else {
#ifdef WITH_GL
    natdisp->setGLColor(natgraph->bgcolor);   
    glRecti(gelem->xwin + x    , -(gelem->ywin + y), 
	    gelem->xwin + x + w, -(gelem->ywin + y + h));
    
    // retour a la normale
    if (natgraph->xor_mode) 
      natdisp->setGLColor(natgraph->color ^ natgraph->bgcolor);
    else natdisp->setGLColor(natgraph->color);
#else
    // sets the Paint mode in any case
    XGCValues gcval;
    gcval.foreground = natgraph->bgcolor;
    gcval.function   = GXcopy;
    XChangeGC(natdisp->xdisplay, natgraph->gc, 
	      GCFunction|GCForeground, &gcval);
    
    XFillRectangle(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
		   gelem->xwin + x, gelem->ywin + y, w, h);
    
    // retour a la normale
    if (natgraph->xor_mode) {
      gcval.foreground = natgraph->color ^ natgraph->bgcolor;
      gcval.function   = GXxor;
      XChangeGC(natdisp->xdisplay, natgraph->gc, 
		GCFunction|GCForeground, &gcval);
    }
    else {
      gcval.foreground = natgraph->color;
      XChangeGC(natdisp->xdisplay, natgraph->gc, GCForeground, &gcval);
    }
#endif
  }
}

/* ==================================================== ======== ======= */

void UGraph::copyArea(u_pos x, u_pos y, u_dim w, u_dim h,
		      u_dim delta_x, u_dim delta_y,
		      bool generate_refresh_events_when_obscured) {
  XGCValues gcval;

  if (generate_refresh_events_when_obscured) {
    gcval.graphics_exposures = true;
    XChangeGC(natdisp->xdisplay, natgraph->gc, GCGraphicsExposures, &gcval);
  }

  XCopyArea(natdisp->xdisplay, gelem->dest->xwin, 
	    gelem->dest->xwin, natgraph->gc,
	    gelem->xwin + x, gelem->ywin + y, 
	    w, h,
	    gelem->xwin + x + delta_x, 
	    gelem->ywin + y + delta_y);

  if (generate_refresh_events_when_obscured) {
    gcval.graphics_exposures = false;
    XChangeGC(natdisp->xdisplay, natgraph->gc, GCGraphicsExposures, &gcval);
  }
#if WITH_GL
  XFlush(natdisp->xdisplay);
#endif
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// !IMPORTANT:
// Pour UWinGraph on doit TOUJOURS avoir: boxview = win_view
// sinon les fonctions de drawing qui ne sont pas redefinies par UWinGraph
// seront incorrectes (c'est le cas de drawPolyline, drawPolygon, fillPolygon)
// drawPolyline draws a sequence of connected lines:

void UGraph::drawPolyline(u_pos xpoints[], 
			  u_pos ypoints[], 
			  u_count card) {
#ifdef WITH_GL
  glBegin(GL_LINE_STRIP);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin,
	       -(ypoints[k] + gelem->ywin));
  glEnd();
#else
  // ATT: le resultat obtenu avec XDrawLines differe d;une boucle 
  // avec XDrawLine dans certains cas (en part. si l'epaisseur est > 1
  // et en mode XOR)
  XPoint points[POLYCARD];
  for (u_count shift = 0 ; shift < card; shift += POLYCARD) {
    u_count subcard = (card-shift <= POLYCARD) ? card-shift : POLYCARD;
    for (u_count k = 0; k < subcard; k++) {
      points[k].x = xpoints[k + shift] + gelem->xwin;
      points[k].y = ypoints[k + shift] + gelem->ywin;
    }
    XDrawLines(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
	       points, subcard, CoordModeOrigin);
  }
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawPolyline(vector<u_pos> xpoints, 
			  vector<u_pos> ypoints, 
			  u_count card) {
#ifdef WITH_GL
  glBegin(GL_LINE_STRIP);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin, -(ypoints[k] + gelem->ywin));
  glEnd();
#else
  // ATT: le resultat obtenu avec XDrawLines differe d;une boucle 
  // avec XDrawLine dans certains cas (en part. si l'epaisseur est > 1
  // et en mode XOR)
  XPoint points[POLYCARD];
  for (u_count shift = 0 ; shift < card; shift += POLYCARD) {
    u_count subcard = (card-shift <= POLYCARD) ? card-shift : POLYCARD;
    for (u_count k = 0; k < subcard; k++) {
      points[k].x = xpoints[k + shift] + gelem->xwin;
      points[k].y = ypoints[k + shift] + gelem->ywin;
    }
    XDrawLines(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
	       points, subcard, CoordModeOrigin);
  }
#endif
}

/* ==================================================== ======== ======= */
// drawPolygon draws a polygon (which is automatically CLOSED):

void UGraph::drawPolygon(u_pos xpoints[], 
			 u_pos ypoints[],
			 u_count card) {
#ifdef WITH_GL
  glBegin(GL_LINE_LOOP);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin,
	       -(ypoints[k] + gelem->ywin));
  glEnd();
#else  
  drawPolyline(xpoints, ypoints, card);
  drawLine(xpoints[0], ypoints[0], xpoints[card-1], ypoints[card-1]);
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawPolygon(vector<u_pos> xpoints, 
			 vector<u_pos> ypoints, 
			 u_count card) {
#ifdef WITH_GL
  glBegin(GL_LINE_LOOP);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin, -(ypoints[k] + gelem->ywin));
  glEnd();
#else  
  drawPolyline(xpoints, ypoints, card);
  drawLine(xpoints[0], ypoints[0], xpoints[card-1], ypoints[card-1]);
#endif
}

/* ==================================================== ======== ======= */
// fillPolygon fills a closed polygon (which is automatically CLOSED):
// NB: 'mode' can be set for optimisation purpose. 
// - default, mode = 0 = Complex : the polygon can have an arbitrary shape
// - mode = 1 = Nonconvex : assumes a non Convex but non Intersecting polygon.
// - mode =2  = Convex    : assumes a Convex polygon. 

// GL WARNING !!!
// with openGL, others modes than 2 are not supported for the moment  !!!
// be carefull when using this function !!!

void UGraph::fillPolygon(u_pos xpoints[], 
			 u_pos ypoints[],
			 u_count card, int mode) {
#ifdef WITH_GL
  // if (mode != 2) {
  //  uerror('w', "UGraph::FillPolygon", 
  //	     "non convex polygons not supported with OpneGLn");
  //}
  glBegin(GL_POLYGON);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin,
	       -(ypoints[k] + gelem->ywin));
  glEnd();
#else
  if (card <= POLYCARD) {
    XPoint points[POLYCARD+1];
    for (u_count k = 0; k < card; k++) {
      points[k].x = xpoints[k] + gelem->xwin;
      points[k].y = ypoints[k] + gelem->ywin;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
  }
  else {
    XPoint *points = (XPoint*)malloc((card+1) * sizeof(XPoint));
    for (u_count k = 0; k < card; k++) {
      points[k].x = xpoints[k] + gelem->xwin;
      points[k].y = ypoints[k] + gelem->ywin;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
    free(points);
  }
#endif
}

void UGraph::fillPolygon(std::vector<u_pos> xpoints, 
			 std::vector<u_pos> ypoints, 
			 u_count card, int mode) {
#ifdef WITH_GL
  // if (mode != 2) {
  //  uerror('w', "UGraph::FillPolygon", 
  //	     "non convex polygons not supported with OpneGLn");
  //}
  glBegin(GL_POLYGON);
  for (u_count k=0; k<card; k++)
    glVertex2i(xpoints[k] + gelem->xwin,
	       -(ypoints[k] + gelem->ywin));
  glEnd();
#else
  if (card <= POLYCARD) {
    XPoint points[POLYCARD+1];
    for (u_count k = 0; k < card; k++) {
      points[k].x = xpoints[k] + gelem->xwin;
      points[k].y = ypoints[k] + gelem->ywin;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
  }
  else {
    XPoint *points = (XPoint*)malloc((card+1) * sizeof(XPoint));
    for (u_count k = 0; k < card; k++) {
      points[k].x = xpoints[k] + gelem->xwin;
      points[k].y = ypoints[k] + gelem->ywin;
    }
    points[card].x = points[0].x; // close the figure
    points[card].y = points[0].y;
    XFillPolygon(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc,
		 points, card+1, mode, CoordModeOrigin);
    free(points);
  }
#endif
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

static int charDefaultWidth(UX_Font fs) {
  // va savoir pourquoi : parfois default_char vaut n'importe quoi !
  if (fs->default_char >= fs->min_char_or_byte2 
      && fs->default_char <= fs->max_char_or_byte2)
    return fs->per_char[fs->default_char].width;
  else
    return fs->per_char[0].width;
}

bool UGraph::getCharDefaultSize(const UFont& font, int& w, int& h) const{
  UFontDesc fd(font);
  return getCharDefaultSize(fd, w, h);
}

bool UGraph::getCharDefaultSize(const UFontDesc& font, int& w, int& h)const {
  if (!wg) return false;
  UX_Font fs = wg->disp->natdisp->getFont(font);
  if (!fs) return false;
  
  if (fs->default_char >= fs->min_char_or_byte2
      && fs->default_char <= fs->max_char_or_byte2) {
    w = fs->per_char[fs->default_char].width;
    h = fs->ascent + fs->descent;
  }
  else {
    w = fs->per_char[0].width;
    h = fs->ascent + fs->descent;
  }
  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

u_dim UGraph::getTextHeight(const UFont& font) const {
  UFontDesc fd(font);
  return getTextHeight(fd);
}

u_dim UGraph::getTextHeight(const UFontDesc& font) const {
  UX_Font fs = wg->disp->natdisp->getFont(font);
  if (!fs) {
    UError::error("UGraph::getTextHeight", UError::Cant_retreive_font);
    return 0;
  }
  else return fs->ascent + fs->descent;
}

/* ==================================================== ======== ======= */

u_dim UGraph::getTextWidth(const UFont& font, const UStr& s) const {
  UFontDesc fd(font);
  return getTextWidth(fd, s.chars(), s.length());
}

u_dim UGraph::getTextWidth(const UFontDesc& font, const char*s, int len) const {
  UX_Font fs = wg->disp->natdisp->getFont(font);
  if (!fs) {
    UError::error("UGraph::getTextWidth", UError::Cant_retreive_font);
    return 0;
  }
  else return XTextWidth(fs, s, len);
}

/* ==================================================== ======== ======= */

void UGraph::getTextSize(const UFont& font, const UStr& s,
			 u_dim& w, u_dim& h) const{
  UFontDesc fd(font);
  getTextSize(fd, s.chars(), s.length(), w, h);
}

void UGraph::getTextSize(const UFontDesc& font, const char*s, int len, 
			 u_dim& w, u_dim& h) const{
  UX_Font fs;
  if (!(fs = wg->disp->natdisp->getFont(font))) {
    w = 0; h = 0;
    UError::error("UGraph::getTextSize", UError::Cant_retreive_font);
    return;
  }
  w = XTextWidth(fs, s, len);
  h = fs->ascent + fs->descent;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UGraph::getCharPos(const UFontDesc& font, 
		       const char *s, int len, u_pos x) const{
  if (!wg) return 0;
  int charpos = 0;
  UX_Font fs = wg->disp->natdisp->getFont(font);
  if (!fs) return 0;

  XCharStruct *cs;
  int width;
  int xpos = 0;

  for (charpos = 0; charpos < len; charpos++) {
    // !!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[charpos];
    
    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) width = cs->width;
      else width = charDefaultWidth(fs);
    }
    else width = charDefaultWidth(fs);

    //printf("--cwidth=%d\n\n",  width);
    if (x <= xpos + width / 2)  break;
    else xpos += width;
  }

  // lie a getSubTextSize: si on tombe sur un seprateur a la fin
  // alors il faut renvoyer position precedente car ce separateur n'est 
  // pas affiche car il y a un saut de ligne implicite
  if (len > 0 && isspace(s[len-1])) charpos--;
  return charpos;
}

/* ==================================================== ======== ======= */

int UGraph::getXPos(const UFontDesc& font, 
		    const char *s, int len, int charpos) const{
  if (!wg) return 0;
  UX_Font fs = wg->disp->natdisp->getFont(font);
  if (!fs) return 0;

  if (charpos > len || charpos < 0) charpos = len;
  XCharStruct *cs = null;
  int xpos = 0;

  for (int k = 0; k < charpos; k++) {
    // !!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[k];

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) xpos += cs->width;
      else xpos += charDefaultWidth(fs);
    }
    else xpos += charDefaultWidth(fs);
  }
  return xpos;
}

/* ==================================================== ======== ======= */
// utilise par drawString en mode OpenGL pour eviter de dessiner
// ce qui sort de la zone de clipping

bool UGraph::getClippedText(const char*s, int len, u_pos x0, u_pos y0,
			    int& charpos_begin, int& charpos_end,
			    u_pos& xpos_begin, u_pos& xpos_end) const {
  charpos_begin = -1; // -1 means undef or invalid
  charpos_end = -1;
  xpos_begin = 0;
  xpos_end = 0;

  UX_Font fs = natgraph->font;
  /*
  UX_Font fs;
  if (!(fs = wg->disp->natdisp->getFont(font))) {
    UError::error("UGraph::getClippedText", UError::Cant_retreive_font);
    return false;
  }
  */
  //h = fs->ascent + fs->descent;

  u_pos xc = x0;
  UX_Rectangle& cliprect = gelem->cliprect;

  for (int pos = 0; pos < len; pos++) {

    if (charpos_begin < 0 && xc >= cliprect.x) {
      charpos_begin = pos; xpos_begin = xc;
    }

    // !att: il FAUT unsigned char sinon valeur <0 pour lettres accentuees
    unsigned char c = s[pos];
    int cw;

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      XCharStruct *cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) cw = cs->width;
      else cw = charDefaultWidth(fs);
    }
    else cw = charDefaultWidth(fs);

    xc += cw;

    // <= sinon le dernier char est jete dans certains cas
    if (xc <= cliprect.x + cliprect.width) {
      charpos_end = pos; xpos_end = xc;
    }
  }

  return true;
}

/* ==================================================== ======== ======= */

bool UGraph::getSubTextSize(const UFontDesc& font, const char*s, int len, 
			    u_dim& w, u_dim& h, u_dim available_width, 
			    int& sublen, int& change_line) const{
  change_line = 0;
  w = 0;
  h = 0;

  UX_Font fs;
  if (!(fs = wg->disp->natdisp->getFont(font))) {
    w = 0; h = 0; sublen = 0;
    UError::error("UGraph::getSubTextSize", UError::Cant_retreive_font);
    return false;
  }

  //int cw_moyen = (fs->min_bounds.width + fs->max_bounds.width) / 2;
  h = fs->ascent + fs->descent;

  int last_pos = -1;
  u_dim lw = 0, last_lw = 0;

  for (int pos = 0; pos < len; pos++) {
    // !att: il FAUT unsigned char sinon valeur <0 pour lettres accentuees
    unsigned char c = s[pos];
    int cw;

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      XCharStruct *cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) cw = cs->width;
      else cw = charDefaultWidth(fs);
    }
    else cw = charDefaultWidth(fs);

    lw += cw;

    if (c == '\n') {
      w = lw;
      sublen = pos+1;
      change_line = 2;
      return true;
    }

    else if (isspace(c)) {
      last_lw = lw;
      last_pos = pos;
    }

    else if (lw > available_width) {
      if (last_pos >= 0) {
	w = last_lw;
	sublen = last_pos+1;
	change_line = 1;
	return true;
      }
    }
  }

  // sinon: tout retourner
  w = lw;
  sublen = len;
  change_line = (lw > available_width ? 1 : 0);
  return true;
}

/* ==================================================== ======== ======= */
// ancienne version imbitable a foutre a la poubelle qunad plus personne
// ne l'utilisera
// invraisemblable; chw retourne est <0 si retour a la ligne !

int UGraph::getSubTextSize(const UFontDesc *font, const char *s, int len, 
			   u_dim maxwidth, u_dim chw,
			   u_dim *w, u_dim *h) const{
  UX_Font fs;
  if (!(fs = wg->disp->natdisp->getFont(*font))) {
    *w = 0; *h = 0;
    UError::error("UGraph::getSubTextSize", UError::Cant_retreive_font);
    return 0;
  }

  *h = fs->ascent + fs->descent;
  *w = 0;
  int charcount = 0;
  int currentw = 0, lastw = 0;
  //int cw_moyen = (fs->min_bounds.width + fs->max_bounds.width) / 2;
  int cw = 0;

  for (int charpos = 0; charpos < len; charpos++) {
    // !!att: il FAUT unsigned char sinon valeur negative 
    // pour lettres accentuees!
    unsigned char c = s[charpos];
    //int cw = cw_moyen;
    lastw = currentw;

    if (c >= fs->min_char_or_byte2 && c <= fs->max_char_or_byte2) {
      XCharStruct *cs = &(fs->per_char[c - fs->min_char_or_byte2]);
      if (cs) cw = cs->width;
      else cw = charDefaultWidth(fs);
    }
    else cw = charDefaultWidth(fs);

    currentw += cw;
    chw += cw;

    //if (c == ' ' || c == '\n' || c == '\t' || c == 13) {
    if (isspace(c)) {

      if (c == 13) {   ///????
	if (chw-cw != 0) {
	  // en theorie, 0, en pratique, il faut le retour a la ligne !
	  if (charcount == len-1) *w = -1;
	  else *w = -currentw-1;
	  charcount = charpos + 1;
	  return charcount;
	} 
	else {
	  lastw -= cw;
	  currentw -= cw;
	  chw -= cw;
	  charcount = charpos + 1;
	}
      }

      else {      
	// on n'a pas encore depasse' la limite => continuer, sauf si '\n'
	if (currentw < maxwidth) {
	  *w = currentw;
	  charcount = charpos+1;
	  if (c == '\n') {
	    //cerr << "subtext: NEWLINE chrpos="<< charpos<<endl;
	    *w = -currentw-1;	// negatif pour le retour a la ligne
	    return charcount;
	  }
	}
	else {
	  // enlever le separ : si ca depasse encore ET si on a deja
	  // trouve un separ precedent alors renvoyer le precedent
	  if (charcount > 0 && lastw >= maxwidth) 
	    return charcount;
	  else {			// sinon renvoyer separ courant
	    *w = currentw;
	    charcount = charpos+1;
	    return charcount;
	  }
	}
      }
    } // endif(c == ...)
  } // endfor

  // cas ou on n'a trouve aucun separateur : on renvoit tout
  *w = currentw;
  return charcount = len;
}

/* ==================================================== [Elc:02] ======= */
// ==== Images and Pixmaps =============================================

#ifdef WITH_GL
static void drawGLTex(UGraphElem *gelem, GLuint glima, 
		      u_dim width, u_dim height, u_pos x, u_pos y) {
  GLfloat x1 = gelem->xwin + x;
  GLfloat y1 = -(gelem->ywin + y);
  GLfloat x2 = x1 + width;
  GLfloat y2 = y1 - height;

  glPushAttrib(GL_CURRENT_BIT);
  glEnable(GL_TEXTURE_2D);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA); // ex (faux): GL_ONE
  glEnable(GL_BLEND);

  glBindTexture(GL_TEXTURE_2D, glima);
  glColor4f(1.0, 1.0, 1.0, 1.0);
  glBegin(GL_POLYGON);
  glTexCoord2f(0.0, 0.0); glVertex2f(x1, y1);     
  glTexCoord2f(1.0, 0.0); glVertex2f(x2, y1);     
  glTexCoord2f(1.0, 1.0); glVertex2f(x2, y2);      
  glTexCoord2f(0.0, 1.0); glVertex2f(x1, y2);
  glEnd();

  glDisable(GL_BLEND);
  glDisable(GL_TEXTURE_2D);  
  glPopAttrib();
}
#endif

/* ==================================================== ======== ======= */

void UGraph::getImaSize(class UNatIma *nima, u_dim &w, u_dim &h) {
  w = nima->getWidth();
  h = nima->getHeight();
}

void UGraph::getPixSize(class UNatPix *nima, u_dim &w, u_dim &h) {
  w = nima->getWidth();
  h = nima->getHeight();
}

/*
void UGraph::drawIma(const class UIma *ima, u_pos x, u_pos y, int scale) {
  const UPix *pix = dynamic_cast<const UPix*>(ima);
  if (pix && pix->getNatPix())
    drawIma(pix->getNatPix(), x, y);
  else
    drawIma(ima->getNatIma(), x, y);
}
*/
/* ==================================================== ======== ======= */

void UGraph::drawIma(const UNatPix *pix, u_pos x, u_pos y) 
{
#ifdef WITH_GL
  drawGLTex(gelem, pix->glIma, pix->width, pix->height, x, y);

#else
  // calculer la zone de clip du pixmap sur la destination
  // (c'est en particulier necessaire s'il y a un mask car XSetClipMask 
  // prend le mask entier sans tenir compte du cliprect

  int clip_x = gelem->cliprect.x - (x + gelem->xwin);
  int clip_y = gelem->cliprect.y - (y + gelem->ywin);

  // mask => effet de transparence
  if (pix->xpixshape != None) {
    XSetClipOrigin(natdisp->xdisplay, natgraph->gc, 
		   gelem->xwin + x, gelem->ywin + y);
    XSetClipMask(natdisp->xdisplay, natgraph->gc, pix->xpixshape);
  }

  XCopyArea(natdisp->xdisplay, pix->xpix, gelem->dest->xwin, natgraph->gc,
	    clip_x, clip_y,	                            // from
	    gelem->cliprect.width, gelem->cliprect.height,  // size
	    gelem->cliprect.x, gelem->cliprect.y);          // to

  // !att: reset du Clip pour la suite
  if (pix->xpixshape != None) {
    XSetClipRectangles(natdisp->xdisplay, natgraph->gc, 
		       0,0, &gelem->cliprect, 1, Unsorted);
  }
#endif
}

/* ==================================================== ======== ======= */

void UGraph::drawIma(const UNatIma *ima, u_pos x, u_pos y) 
{
#ifdef WITH_GL
    drawGLTex(gelem, ima->glIma, ima->xima->width, ima->xima->height, x, y);
#else
  // optimisation du clipping: 
  // on ne peut pas se contenter d'utiliser le GC pour le clipping
  // car il y a un transfert client -> serveur de l'image
  // et il faut donc clipper l'image source AVANT le transfert 
  // pour minimiser la quaantite de donnees a transferer

  // ima rect: rectangle de la location de l'ima dans target view
  // si l'image n'etait pas clippee
  URegion imarect(x, y, ima->xima->width, ima->xima->height);

  // intersection d'imarect avec le clip courant
  bool clipstat = imarect.setInter(gelem->cliprect.x - gelem->xwin, 
				   gelem->cliprect.y - gelem->ywin,
				   gelem->cliprect.width, 
				   gelem->cliprect.height);
  // rien a afficher
  if (!clipstat) return;

  // calcul du decalage dans l'image d'origine:
  // eg.: si imarect.x == x apres le clip, il faut afficher l'image
  // depuis son origine, sinon il y a une partie a ne pas afficher
  // d'ou un decalage

  //static int no_dima = 0;
  //no_dima++;
  //cerr << "drawIma ima " << ima 
  //     << " size " << imarect.width << " " << imarect.height 
  //     <<  " # " << no_dima << endl;

  XPutImage(natdisp->xdisplay, gelem->dest->xwin, natgraph->gc, 
            ima->xima, 
	    imarect.x - x,	              //from in source ima
	    imarect.y - y,
	    imarect.x+ gelem->xwin,      //to in target view 
	    imarect.y+ gelem->ywin,
	    imarect.width, imarect.height);  // clipped size of source
#endif
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

void UGraph::flush() {
  if (natgraph) {
#ifdef WITH_GL 
    glFlush();
#else
    XFlush(natdisp->xdisplay);
#endif
  }
}

// physical grab on this window (only for menus!)
int UGraph::grabPointer(class UCursor *c, bool confine_to_window) {
  if (!UAppli::getDefaults().menu_grab) {
    return true;
  }
  else {
    return XGrabPointer(natdisp->xdisplay, 
			wg->natwin->getXWindow(),
			//owner_events: si True, les windows sous le pointeur
			//recoivents les Events comme d'habitude (et les coords
			//sont relatives a celles-ci). Si False, c'est la grab-window
			//qui recoit tous les events (et les coords sont relatives
			//a la grab-window)
			True,
			ButtonPressMask|ButtonReleaseMask|PointerMotionMask,
			GrabModeAsync,    // pointer_mode
			GrabModeAsync,    // keyboard_mode
			(confine_to_window ? wg->natwin->getXWindow() : None),
			None,             // cursor 
			CurrentTime);     // time
  }
}

void UGraph::ungrabPointer() {
  if (natgraph) XUngrabPointer(natdisp->xdisplay, CurrentTime);
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */
