/* ==================================================== ======== ======= *
 *
 *  uumsclient.cpp : UMS [Ubit Mouse Server] client
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uumsclient.cpp	ubit:03.06.04"
#include <iostream>
#include <unistd.h>
#include <stdlib.h>
#include <netdb.h>
#include <netinet/tcp.h>   // NEW: A VERIFIER PARTOUT !!
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/utsname.h>
#include <X11/Xlib.h>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ubox.hpp>
#include <uwin.hpp>
#include <ustr.hpp>
#include <unatdisp.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <umsproto.hpp>
#include <umsclient.hpp>
using namespace std;

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UMSclient::UMSclient(UAppli& a) {
  constructs(&a, a.getCommandName());
}

UMSclient::UMSclient(const UStr& _client_name) {
  constructs(null, _client_name);
}

void UMSclient::constructs(UAppli* a, const UStr& _client_name) {
  appli = a;
  client_name = _client_name;
  remote_host = null;
  remote_port = 0;
  rhost       = new sockaddr_in;
  com_sock    = UMSrequest::CloseCnx;
  stat        = NotOpened;
  input = null;
}

UMSclient::~UMSclient() {
  close();
  delete rhost;
}

void UMSclient::close() {
  if (com_sock > UMSrequest::CloseCnx) {
    ::shutdown(com_sock, 2);
    ::close(com_sock);
  }
  if (appli && input) {
    appli->closeInput(input); 
    input = null;
  }
  remote_host = null;
  remote_port = 0;
  com_sock    = UMSrequest::CloseCnx;
  stat        = NotOpened;
}

int UMSclient::open(const UStr& _host, int _port) {
  close();
  remote_host = _host;
  remote_port = (_port != 0 ? _port : UMSprotocol::UMS_DEFAULT_PORT);

  // creer l'adresse du remote host a partir de son nom (getipnodebyname)
  struct hostent* hostinfo = gethostbyname(remote_host.chars());
  if (!hostinfo) {
    //UError::error("warning@UMSclient::UMSclient",
    //		  "remote host not found; host:", remote_host);
    return stat = HostNotFound;
  }

  rhost->sin_family = AF_INET;
  rhost->sin_port = htons(remote_port);
  memcpy(&rhost->sin_addr, hostinfo->h_addr_list[0], hostinfo->h_length);

  if (openCom(client_name, true) <= UMSrequest::CloseCnx) {
    return stat = CannotOpen;
  }

  stat = Opened;

  if (appli) {
    input = appli->openInput(com_sock);
    input->onAction(ucall(this, &UMSclient::receive));
  }

  return stat;
}

int UMSclient::getStatus() const {return stat;}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// returns UMSrequest status and initializes com_sock
// keeps connection if keep_cnx_request is true and the server accepted
// to keep it. 

int UMSclient::openCom(const UStr& cnx_name, bool keep_cnx) {
  char receive_str[1024];
  int stat = UMSrequest::Ok;

  com_sock = ::socket(AF_INET, SOCK_STREAM, 0);

  // turn off TCP coalescence for INET sockets  // NEW!!
  // NB: necessaire (au moins) pour MacOS, sinon delai de transmission
  int tmp = 1;
  ::setsockopt(com_sock, IPPROTO_TCP, TCP_NODELAY,(char*)&tmp, sizeof(int));
  
  // creation du socket et connexion vers le remote host
  if (com_sock < 0
      || ::connect(com_sock, (struct sockaddr*)rhost, sizeof(*rhost)) < 0
      ){
    //uerror('d', "UMSclient::openComSocket", 
    //	   "Can't connect port %d of remote host %s", remote_port, remote_server);
    return UMSrequest::Error;
  }

  // attendre message de debut de communication depuis serveur
  if (read(com_sock, receive_str, sizeof(receive_str)) < 0) {
    UError::error("warning@UMSclient::openCom", "can't read socket") ;
    stat = UMSrequest::Error;
    goto CLOSE_CNX;
  }

  if (cnxOpened(cnx_name) == false) {       // NB: uses com_sock
    //uerror("UMSclient::openCom", "[warning] Can't write on socket");
    stat = UMSrequest::Error;
    goto CLOSE_CNX;
  }

  if (!keep_cnx) {    // utile ?
    ::shutdown(com_sock, 1);	// fermer la connexion dans le sens output
  }

  // lire la reponse du serveur
  if (read(com_sock, receive_str, sizeof(receive_str)) < 0) {
    UError::error("warning@UMSclient::openCom", "can't read socket");
    stat = UMSrequest::Error;
    goto CLOSE_CNX;
  }

  if (keep_cnx) {   // check if the server accepted to keep the connection
    char* p = strstr(receive_str, "KEEP_CNX=");
    if (p) {
      p += 9; 
      if (*p == '1') stat = UMSrequest::KeepCnx;
    }
  }

 CLOSE_CNX:
  if (stat == UMSrequest::KeepCnx) return UMSrequest::KeepCnx;
  else {
    ::shutdown(com_sock, 2);
    ::close(com_sock);
    return (stat == UMSrequest::Error) ? UMSrequest::Error : UMSrequest::CloseCnx;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UMSclient::sendRequest(UMSrequest& req) {
  if (com_sock <= UMSrequest::CloseCnx) {
    UError::error("warning@UMSclient::sendRequest", "can't write on socket");
    return false;
  }

  // req.data is the octal size (must be * by 8)
  // (so, we always send 8 byte data blocks)
  req.data[0] = req.count/8 + req.count%8;

  if (::write(com_sock, &req, req.data[0]*8) < 0) return false;
  else return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UMSclient::receive() {
  XEvent xev;
  int total = 0;
  int stat = UMSrequest::Ok;

  // reception des donnees envoyees via le socket
  // attention, plusieurs read() peuvent etre necessaires

  int n = 0;
  while (1) {
    n = ::read(com_sock, &xev + total, sizeof(xev));
    total += n;

    if (n < 0) {
      UError::error("warning@UMSclient::receive", "can't read socket");
      stat = UMSrequest::Error;
      break; 
    } 
    else if (n == 0) {
      //UError::error("warning@UMSclient::receive", "end of connection");
      stat = UMSrequest::CloseCnx;
      break;
    } 
    else if (total >= int(sizeof(xev))) {
      stat = UMSrequest::Ok;
      break;
    }
  }

  if (stat == UMSrequest::Error || stat == UMSrequest::CloseCnx) {
    UError::error("warning@UMSclient::receive", "connection closed");
    ::shutdown(com_sock, 2);
    ::close(com_sock);
    if (appli && input) {
      appli->closeInput(input);
      input = null;
    }
    return;
  }

  // le champ 'display' doit etre celui de l'appli et non de l'UMS
  xev.xany.display = appli->getNatDisp()->getXDisplay();
  appli->getNatDisp()->dispatchEvents(&xev);
}

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */

bool UMSclient::moveMouse(int mouse_event_flow, 
			  u_pos x, u_pos y, bool abs_coords) {
  UMSrequest req(UMSrequest::MOUSE_CTRL);
  req.writeEvent(MotionNotify, mouse_event_flow, x, y, abs_coords);
  return sendRequest(req);
}

bool UMSclient::pressMouse(int mouse_event_flow, u_id mouse_btn) {
  UMSrequest req(UMSrequest::MOUSE_CTRL);
  req.writeEvent(ButtonPress, mouse_event_flow, 0, 0, mouse_btn);
  return sendRequest(req);
}

bool UMSclient::releaseMouse(int mouse_event_flow, u_id mouse_btn) {
  UMSrequest req(UMSrequest::MOUSE_CTRL);
  req.writeEvent(ButtonRelease, mouse_event_flow, 0, 0, mouse_btn);
  return sendRequest(req);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UMSclient::sendMousePress(const char* target,
			       u_pos x, u_pos y, u_id mouse_btn) {
  UMSrequest req(UMSrequest::SEND_EVENT);
  req.writeEvent(ButtonPress, /*flow*/0, x, y, mouse_btn);
  req.writeString(target);
  return sendRequest(req);
}

bool UMSclient::sendMouseRelease(const char* target,  
				 u_pos x, u_pos y, u_id mouse_btn) {
  UMSrequest req(UMSrequest::SEND_EVENT);
  req.writeEvent(ButtonRelease, /*flow*/0, x, y, mouse_btn);
  req.writeString(target);
  return sendRequest(req);
}

bool UMSclient::sendMouseClick(const char* target, 
			       u_pos x, u_pos y, u_id mouse_button) {
  return 
    sendMousePress(target, x, y, mouse_button)
    && sendMouseRelease(target, x, y, mouse_button);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UMSclient::sendMessage(const char* target, const UStr& message) {
  return sendMessage(target, message.chars());
}

bool UMSclient::sendMessage(const char* target, const char* message) {
  if (!message) message = "";

  if (!target) {
    UError::error("warning@UMSclient::sendMessage","null target");
    return false;
  }

  UMSrequest req(UMSrequest::SEND_MESSAGE);
  req.writeString(target);
  //req.writeChar(' ');
  if (!req.writeString(message)) {
    UError::error("warning@UMSclient::sendMessage", 
		  "message is too long (truncated)");
  }

  return sendRequest(req);
}

/* ==================================================== ======== ======= */

bool UMSclient::winOpened(u_id win_id) {
  UMSrequest req(UMSrequest::OPEN_WIN);
  req.writeLong(win_id);
  return sendRequest(req);
}

bool UMSclient::winClosed(u_id win_id) {
  UMSrequest req(UMSrequest::CLOSE_WIN);
  req.writeLong(win_id);
  return sendRequest(req);
}

bool UMSclient::cnxOpened(const UStr& cnx_name) {
  UMSrequest req(UMSrequest::OPEN_CNX);
  req.writeString(cnx_name.chars());
  return sendRequest(req);
}

bool UMSclient::cnxClosed() {
  UMSrequest req(UMSrequest::CLOSE_CNX);
  return sendRequest(req);
}

/* ==================================================== ======== ======= */
/*
void testMouseServer(UAppli* appli, int cas) {
  if (!serv) {
    cerr << "creating server com " << endl;
    serv = new UMSclient("odyssey", MOUSE_SERVER_PORT);
  }
  Window win_appli = appli->getNatAppli()->getXWindow();
  char s[100];
  sprintf(s, "OPEN_CNX %ld", win_appli);

  switch (cas) {
  case 1:
    stat = serv->openComSocket(s, short(strlen(s)+1), false);
    break;
  case 2:
    stat = serv->openComSocket(s, short(strlen(s)+1), true);
    break;
  case 3:
    if (serv->com_socket <= UMSrequest::CloseCnx) {
      stat = serv->openComSocket(s, short(strlen(s)+1), true);
    }
    else
      serv->writeComSocket(serv->com_socket, s, short(strlen(s)+1));
    break;
  }
  cerr << "com_socket "<< serv->com_socket<<endl;
}
*/

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

