#include "config.h"
#if defined (HAVE_LIBGIF) || defined(HAVE_LIBUNGIF) 

//pragma ident	"@(#)uungif.cc ubit:03.04.00"

/* uungif.cc : glue with the UNGIF library (i.e. libungif.so)
 *
 * file adapted from the UNGIF "util" program "gif2x11.c" by Gershon Elber.
 *
 * Note: as the GIF image format is based on the (patented) LZW compression 
 * algorithm, this file does not contain code for compressing GIF images.
 * Code which does gif decoding only seems not to be covered by the patent:
 * check the ungif web site (or your own lawyer :-) for details.
 *
 * Latest versions of libungif are available from:
 *    ftp://prtr-13.ucsc.edu/pub/libungif/
 *
 * You can see a web page with late breaking announcements at:
 *    http://prtr-13.ucsc.edu/~badger/software/libungif.shtml
 */

/*****************************************************************************
*   "Gif-Lib" - Yet another gif library.				     *
*									     *
* Written by:  Gershon Elber				Ver 0.1, Jul. 1989   *
******************************************************************************/

#if defined (__MSDOS__) && !defined(__DJGPP__) && !defined(__GNUC__)
#include <graphics.h>
#include <stdlib.h>
#include <alloc.h>
#include <io.h>
#include <dos.h>
#include <bios.h>
#include <sys\stat.h>
#else
#include <sys/types.h>
#include <sys/stat.h>
#endif 

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <fcntl.h>

extern "C" {
#include <gif_lib.h>
}
#include <ubit.hpp>
#include <unatima.hpp>
#include <unatdisp.hpp>

#ifdef __MSDOS__
extern unsigned int
    _stklen = 16384;			     /* Increase default stack size. */
#endif /* __MSDOS__ */

/* ==================================================== ======== ======= */

#define GIF_COLORTABLE_SIZE 256

struct UngifAttributes {
  int from_stdin;
  int width, height;
  unsigned char **buffer;
  // number of colors that are actually used - always <= 256
  int    colorCount;
  XColor colorTable[256];
};

static int
InterlacedOffset[] = { 0, 4, 2, 1 },  /* The way Interlaced image should. */
InterlacedJumps[]  = { 8, 8, 4, 2 };   /* be read - offsets and jumps... */


/******************************************************************************
* Interpret the command line and scan the given GIF file.		      *
******************************************************************************/

static int readGifFile(const char *filename, UngifAttributes *attr)
{
  int i, j, Size, Row, Col, Width, Height, ExtCode, Count;
  GifRecordType RecordType;
  GifByteType *Extension;
  GifRowType  *ScreenBuffer;
  GifFileType *GifFile;

  if (attr->from_stdin == 0) {
    if (!filename || !(GifFile = DGifOpenFileName(filename))) {
      // PrintGifError();
      return UFilestat::CannotOpen;
    }
  }
  else {			/* Use the stdin instead: */
#ifdef __MSDOS__
    setmode(0, O_BINARY);
#endif /* __MSDOS__ */
    if ((GifFile = DGifOpenFileHandle(0)) == NULL) {
      // PrintGifError();
      return UFilestat::CannotOpen;
    }
  }

  /* Allocate the screen as vector of column of rows. We cannt allocate    */
  /* the all screen at once, as this broken minded CPU can allocate up to  */
  /* 64k at a time and our image can be bigger than that:		   */
  /* Note this screen is device independent - its the screen as defined by */
  /* the GIF file parameters itself.					   */
  if ((ScreenBuffer = 
       (GifRowType *)malloc(GifFile->SHeight * sizeof(GifRowType *))) == NULL){
    // UGIF_EXIT("Failed to allocate memory required, aborted.");
    return UFilestat::NoMemory;
  }

  Size = GifFile->SWidth * sizeof(GifPixelType);/* Size in bytes one row.*/
  if ((ScreenBuffer[0] = (GifRowType) malloc(Size)) == NULL) { /* First row. */
    // UGIF_EXIT("Failed to allocate memory required, aborted.");
    return UFilestat::NoMemory;
  }

  for (i = 0; i < GifFile->SWidth; i++)  /* Set its color to BackGround. */
    ScreenBuffer[0][i] = GifFile->SBackGroundColor;

  for (i = 1; i < GifFile->SHeight; i++) {
    /* Allocate the other rows, and set their color to background too: */
    if ((ScreenBuffer[i] = (GifRowType) malloc(Size)) == NULL) {
      // UGIF_EXIT("Failed to allocate memory required, aborted.");
      return UFilestat::NoMemory;
    }
    memcpy(ScreenBuffer[i], ScreenBuffer[0], Size);
  }
  
  /* Scan the content of the GIF file and load the image(s) in: */
  do {

    if (DGifGetRecordType(GifFile, &RecordType) == GIF_ERROR) {
      // PrintGifError();
      return UFilestat::InvalidData;
    }

    switch (RecordType) {
    case IMAGE_DESC_RECORD_TYPE:
      if (DGifGetImageDesc(GifFile) == GIF_ERROR) {
	// PrintGifError();
	return UFilestat::InvalidData;
      }
      Row = GifFile->Image.Top; /* Image Position relative to Screen. */
      Col = GifFile->Image.Left;
      Width = GifFile->Image.Width;
      Height = GifFile->Image.Height;
      
      if (GifFile->Image.Left + GifFile->Image.Width > GifFile->SWidth ||
	  GifFile->Image.Top + GifFile->Image.Height > GifFile->SHeight) {
	// UGIF_EXIT("Image is not confined to screen dimension, aborted");
	return UFilestat::InvalidData;
      }

      if (GifFile->Image.Interlace) {
	/* Need to perform 4 passes on the images: */
	for (Count = i = 0; i < 4; i++)
	  for (j = Row + InterlacedOffset[i]; j<Row + Height;
	       j += InterlacedJumps[i]) {
	    if (DGifGetLine(GifFile, &ScreenBuffer[j][Col], Width) == GIF_ERROR) {
	      // PrintGifError();
	      return UFilestat::InvalidData;
	    }
	  }
      }
      else {
	for (i = 0; i < Height; i++) {
	  if (DGifGetLine(GifFile, &ScreenBuffer[Row++][Col], Width) == GIF_ERROR) {
	    // PrintGifError();
	    return UFilestat::InvalidData;
	  }
	}
      }
      break;

    case EXTENSION_RECORD_TYPE:
      /* Skip any extension blocks in file: */
      if (DGifGetExtension(GifFile, &ExtCode, &Extension) == GIF_ERROR) {
	// PrintGifError();
	return UFilestat::InvalidData;
      }
      while (Extension != NULL) {
	if (DGifGetExtensionNext(GifFile, &Extension) == GIF_ERROR) {
	  // PrintGifError();
	  return UFilestat::InvalidData;
	}
      }
      break;

    case TERMINATE_RECORD_TYPE:
      break;

    default:		    /* Should be traps by DGifGetRecordType. */
      break;
    }
  }
  while (RecordType != TERMINATE_RECORD_TYPE);

  attr->width  = GifFile->SWidth;
  attr->height = GifFile->SHeight;
  attr->buffer = ScreenBuffer;

  /* X specific stuff goes here. XColorTable will hold the GIF image colors */
  /* Note the table has 256 entry which is the maximum allowed in GIF format*/

  ColorMapObject *colorMap = 
    (GifFile->Image.ColorMap ? GifFile->Image.ColorMap : GifFile->SColorMap);

  attr->colorCount = colorMap->ColorCount;
  for (i = 0; i < attr->colorCount; i++) {
    /* Prepare color entry in X format. */
    attr->colorTable[i].red   = colorMap->Colors[i].Red << 8;
    attr->colorTable[i].green = colorMap->Colors[i].Green << 8;
    attr->colorTable[i].blue  = colorMap->Colors[i].Blue << 8;
    attr->colorTable[i].flags = DoRed | DoGreen | DoBlue;
  }

  //printf("BackGround = %d\n", GifFile->SBackGroundColor);

  if (DGifCloseFile(GifFile) == GIF_ERROR) {
    // PrintGifError();
    return UFilestat::MiscError;  
  }
  else return UFilestat::Opened; 
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

int UNatIma::gifFileReader(UNatDisp *nd, const char *fpath, UNatIma*& natima)
{
  natima = null;

  UngifAttributes attr;
  attr.from_stdin = 0;
  attr.buffer = null;

  int stat = readGifFile(fpath, &attr);

  if (stat < UFilestat::Opened) {
    // faudrait detruire attr.buffer[] !!!
    return stat;
  }
 
  UX_Image ima = createEmptyImage(nd, attr.width, attr.height, nd->getDepth());
  if (!ima) {
    // faudrait detruire attr.buffer[] !!!
    if (attr.buffer) free(attr.buffer);
    return UFilestat::NoMemory;
  }
  
  // Note the table has 256 entry which is the maximum allowed in GIF format
  unsigned long* convtable = allocXColors(nd, attr.colorTable, attr.colorCount);
  
  for (int y = 0; y < attr.height; y++) {
    for (int x = 0; x < attr.width; x++)
      XPutPixel(ima, x, y, convtable[attr.buffer[y][x]]);
    free(attr.buffer[y]);
  }
  free(attr.buffer);
  delete[] convtable;

  //width  = attr.width; height = attr.height;
  natima = new UNatIma(nd, ima, null);    // shapeima = null
  return UFilestat::Opened;
}

/* ========================================================================= */
/* ========================================================================= */
#endif
