/* ========================================================================== */
/* === UMF_transpose ======================================================== */
/* ========================================================================== */

/* -------------------------------------------------------------------------- */
/* UMFPACK Version 3.2 (Jan. 1, 2002), Copyright (c) 2002 by Timothy A.       */
/* Davis, University of Florida, davis@cise.ufl.edu.  All Rights Reserved.    */
/* See README, umfpack.h, or type "umfpack_details" in Matlab for License.    */
/* -------------------------------------------------------------------------- */

/*  Not user-callable.  Computes a permuted transpose, C = (A (P,Q(1:nq)))' in
	Matlab notation, where B is in column-form.  Alternatively, this routine
	can be viewed as constructing the row-form of B = A(P,Q (1:nq)).  The
	matrix A must have sorted columns.  B does have sorted rows
	(equivalently, C has sorted columns).  A is square, B is n-by-nq, where
	nq < n.  A may be singular.
*/

#include "umf_internal.h"
#include "umf_is_permutation.h"

/* ========================================================================== */

GLOBAL Int UMF_transpose
(
    Int n,
    const Int Ap [ ],		/* size n+1 */
    const Int Ai [ ],		/* size nz = Ap [n] */
    const double Ax [ ],	/* size nz, pattern only if Ax and/or Bx */
				/* not present */

    const Int P [ ],	/* P [k] = i means original row i is kth row in A(P,Q)*/
			/* P is identity if not present */
			/* size n, if present */

    const Int Q [ ],	/* Q [k] = j means original col j is kth col in A(P,Q)*/
			/* Q is identity if not present */
			/* size nq, if present */
    Int nq,		/* size of Q, ignored if Q is (Int *) NULL */

			/* output matrix: Bp, Bi, and Bx: */
    Int Bp [ ],		/* size n+1 */
    Int Bi [ ],		/* size max (n,nz) */
    double Bx [ ],	/* size nz, if present */

    Int W [ ],		/* size n workspace */

    Int check		/* if true, then check inputs */
)
{

    /* ---------------------------------------------------------------------- */
    /* local variables */
    /* ---------------------------------------------------------------------- */

    Int i, j, k, p, bp, *RowCount, nz, newj, lastj, ilast ;

    /* ---------------------------------------------------------------------- */
    /* check inputs */
    /* ---------------------------------------------------------------------- */

    if (check)
    {
	/* UMFPACK_symbolic skips this check */
	/* UMFPACK_transpose always does this check */

	if (!Ai || !Ap || !Bi || !Bp || !W)
	{
	    return (UMFPACK_ERROR_argument_missing) ;
	}

	if (n <= 0)		/* n must be > 0 */
	{
	    return (UMFPACK_ERROR_n_nonpositive) ;
	}

	nz = Ap [n] ;
	if (nz < 0)		/* nz must be >= 0 */
	{
	    return (UMFPACK_ERROR_nz_negative) ;
	}

	if (!UMF_is_permutation (P, W, n, n) || !UMF_is_permutation (Q, W, n,n))
	{
	    return (UMFPACK_ERROR_invalid_permutation) ;
	}

	if (Ap [0] != 0)
	{
	    return (UMFPACK_ERROR_Ap0_nonzero) ;
	}

	for (j = 0 ; j < n ; j++)
	{
	    if (Ap [j] > Ap [j+1])
	    {
		return (UMFPACK_ERROR_col_length_negative) ;
	    }
	}

	for (j = 0 ; j < n ; j++)
	{
	    ilast = -1 ;
	    for (p = Ap [j] ; p < Ap [j+1] ; p++)
	    {
		i = Ai [p] ;
		if (i < 0 || i >= n)
		{
		    return (UMFPACK_ERROR_row_index_out_of_bounds) ;
		}
		if (i <= ilast)
		{
		    return (UMFPACK_ERROR_jumbled_matrix) ;
		}
		ilast = i ;
	    }
	}
    }

    /* ---------------------------------------------------------------------- */
    /* count the entries in each row of A */
    /* ---------------------------------------------------------------------- */

    /* use Bi as workspace for RowCount */
    RowCount = Bi ;

    for (i = 0 ; i < n ; i++)
    {
	RowCount [i] = 0 ;
	Bp [i] = 0 ;
    }

    if (Q)
    {
	lastj = nq ;
    }
    else
    {
	lastj = n ;
    }

    for (newj = 0 ; newj < lastj ; newj++)
    {
	if (Q)
	{
	    j = Q [newj] ;
	}
	else
	{
	    j = newj ;
	}
	DEBUG3 (("Old col "ID" new col "ID"\n", j, newj)) ;
	for (p = Ap [j] ; p < Ap [j+1] ; p++)
	{
	    RowCount [Ai [p]]++ ;
	}
    }

    /* ---------------------------------------------------------------------- */
    /* compute the row pointers for B = A (P,Q) */
    /* ---------------------------------------------------------------------- */

    if (P)
    {
	Bp [0] = 0 ;
	for (k = 0 ; k < n ; k++)
	{
	    Bp [k+1] = Bp [k] + RowCount [P [k]] ;
	}
	for (k = 0 ; k < n ; k++)
	{
	    W [P [k]] = Bp [k] ;
	}
    }
    else
    {
	Bp [0] = 0 ;
	for (i = 0 ; i < n ; i++)
	{
	    Bp [i+1] = Bp [i] + RowCount [i] ;
	}
	for (i = 0 ; i < n ; i++)
	{
	    W [i] = Bp [i] ;
	}
    }

    /* ---------------------------------------------------------------------- */
    /* construct the row form of B */
    /* ---------------------------------------------------------------------- */

    if (Q)
    {
	lastj = nq ;
    }
    else
    {
	lastj = n ;
    }

    for (newj = 0 ; newj < lastj ; newj++)
    {
	if (Q)
	{
	    j = Q [newj] ;
	}
	else
	{
	    j = newj ;
	}
	DEBUG3 (("Old col "ID" new col "ID"\n", j, newj)) ;

	for (p = Ap [j] ; p < Ap [j+1] ; p++)
	{
	    bp = W [Ai [p]]++ ;
	    DEBUG4 (("    old row "ID"  bp "ID"\n", Ai [p], bp)) ;
	    Bi [bp] = newj ;
	    if (Ax && Bx)
	    {
		Bx [bp] = Ax [p] ;
	    }
	}
    }

#ifndef NDEBUG
    for (k = 0 ; k < n ; k++)
    {
	if (P)
	{
	    i = P [k] ;
	}
	else
	{
	    i = k ;
	}
	DEBUG3 ((ID":  W[i] "ID" Bp[k+1] "ID"\n", i, W [i], Bp [k+1])) ;
	ASSERT (W [i] == Bp [k+1]) ;
    }
#endif

    return (UMFPACK_OK) ;
}

