#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <ctype.h>
#include <sys/types.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "view.h"
#include "printwin.h"

#include "vmacfg.h"
#include "vmacfglist.h"
#include "vmainstall.h"
#include "vma.h"
#include "vmautils.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


/* Preference initializers. */
int VMAInstallDefaultLocalDataDirs(
	const char *local_data_dir
);
void VMAInstallDefaultPreferences(
	vma_cfg_item_struct *parm_list
);


#define RADTODEG(r)     ((r) * 180 / PI)
#define DEGTORAD(d)     ((d) * PI / 180)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))


/*
 *	Creates default local data directories with respect to the
 *	given local data directory.
 *
 *	This call will update members in the global dname structure if
 *	successful.
 *
 *	Returns 0 on success or non-zero on error.
 */
int VMAInstallDefaultLocalDataDirs(
        const char *local_data_dir
)
{
	mode_t m_dir = (	S_IRUSR | S_IWUSR | S_IXUSR |
				S_IRGRP | S_IXGRP |
				S_IROTH | S_IXOTH
	);
	char *strptr;
	const char *cstrptr;
	struct stat stat_buf;


	if(local_data_dir == NULL)
	    return(-1);

	/* Create local data directory as needed. */
	if(stat(local_data_dir, &stat_buf))
	{
	    /* Local data directory does not exist, try and create it. */
	    if(rmkdir(local_data_dir, m_dir))
		return(-1);
	}
	else
	{
	    /* Local data directory does exist, but make sure it is
	     * actually a directory.
	     */
	    if(!S_ISDIR(stat_buf.st_mode))
	    {
		CDialogGetResponse(
"Invalid Object!",
"Specified local data directory is not an object of type\n\
directory, cannot continue with installation.",
"The local data directory parsed from either the\n\
default installation location or the specified\n\
configuration file is not a directory. You may\n\
want to try and specify an alternative location\n\
by specifying a different location for the\n\
configuration file with the -f <configuration_file>\n\
argument.",
		    CDIALOG_ICON_ERROR,
		    CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		    CDIALOG_BTNFLAG_OK
		);
		return(-1);
	    }
	}

	/* Update dname structure. */
	/* Global data dir. */
        strptr = dname.data_global;
        strncpy(strptr, VMADefaultDataGlobalDir(), PATH_MAX);
        strptr[PATH_MAX - 1] = '\0';

	/* Local data dir. */
	strptr = dname.data_local;
	strncpy(strptr, local_data_dir, PATH_MAX);
	strptr[PATH_MAX - 1] = '\0';

	/* User's home dir. */
	cstrptr = getenv("HOME");
        strptr = dname.home;
        strncpy(
	    strptr, (cstrptr == NULL) ? "/" : cstrptr, PATH_MAX
	);
        strptr[PATH_MAX - 1] = '\0';

	/* Tempory dir (with user's name included). */
	strptr = NULL;
        cstrptr = (const char *)PrefixPaths(
            VMA_DEF_TMP_DIR, "vertex-"
        );
        strptr = strcatalloc(
            strptr, (cstrptr == NULL) ? VMA_DEF_TMP_DIR : cstrptr
        );
        cstrptr = (const char *)getenv("USERNAME");
        if(cstrptr == NULL)
            cstrptr = (const char *)getenv("USER");
        strptr = strcatalloc(
            strptr, (cstrptr == NULL) ? "unknown" : cstrptr
        );
        strncpy(
            dname.tmp,
	    (strptr == NULL) ? VMA_DEF_TMP_DIR : strptr,
	    PATH_MAX
        );
        dname.tmp[PATH_MAX - 1] = '\0';
	free(strptr); strptr = NULL;


	/* Last models directory. */
	strptr = dname.fb_last_models;
	strncpy(strptr, dname.home, PATH_MAX);
	strptr[PATH_MAX - 1] = '\0';

	/* Last textures directory. */
        strptr = dname.fb_last_textures;
        strncpy(strptr, dname.home, PATH_MAX);
        strptr[PATH_MAX - 1] = '\0';

	/* Last view background image directory. */
        strptr = dname.fb_last_view_background_image;
        strncpy(strptr, dname.home, PATH_MAX);
        strptr[PATH_MAX - 1] = '\0';


	/* Local and global preset models directories. */
	strptr = dname.preset_models_global;
	cstrptr = (const char *)PrefixPaths(
	    dname.data_global, VMA_DEF_PRESET_MODELS_DIR
	);
	strncpy(
	    strptr, (cstrptr == NULL) ? dname.data_global : cstrptr, PATH_MAX
	);
        strptr[PATH_MAX - 1] = '\0';

        strptr = dname.preset_models_local;
        cstrptr = (const char *)PrefixPaths(
            dname.data_local, VMA_DEF_PRESET_MODELS_DIR
        );
        strncpy(
            strptr, (cstrptr == NULL) ? dname.data_local : cstrptr, PATH_MAX  
        );
        strptr[PATH_MAX - 1] = '\0';


        /* Local and global preset primitives directories. */
        strptr = dname.preset_primitives_global;
        cstrptr = (const char *)PrefixPaths(
            dname.data_global, VMA_DEF_PRESET_PRIMITIVES_DIR
        );  
        strncpy(
            strptr, (cstrptr == NULL) ? dname.data_global : cstrptr, PATH_MAX
        );
        strptr[PATH_MAX - 1] = '\0';
          
        strptr = dname.preset_primitives_local;
        cstrptr = (const char *)PrefixPaths(
            dname.data_local, VMA_DEF_PRESET_PRIMITIVES_DIR
        );
        strncpy(
            strptr, (cstrptr == NULL) ? dname.data_local : cstrptr, PATH_MAX
        );
        strptr[PATH_MAX - 1] = '\0';


	/* Local and global plugins directories. */
        strptr = dname.plugins_global;
        cstrptr = (const char *)PrefixPaths(
            dname.data_global, VMA_DEF_PLUGINS_DIR
        );
        strncpy(
            strptr, (cstrptr == NULL) ? dname.data_global : cstrptr, PATH_MAX
        );
        strptr[PATH_MAX - 1] = '\0';

        strptr = dname.plugins_local;
        cstrptr = (const char *)PrefixPaths(
            dname.data_local, VMA_DEF_PLUGINS_DIR
        );
        strncpy(
            strptr, (cstrptr == NULL) ? dname.data_local : cstrptr, PATH_MAX
        );
        strptr[PATH_MAX - 1] = '\0';


	/* *********************************************************** */

	/* Create local preset models directory. */
	rmkdir(dname.preset_models_local, m_dir);

        /* Create local preset primitives directory. */
	rmkdir(dname.preset_primitives_local, m_dir);

	/* Create local plug-ins directory. */
        rmkdir(dname.plugins_local, m_dir);

	return(0);
}


/*
 *	Resets the given configuration options list to default values,
 *	this function should be called by VMADoInstall().
 */
void VMAInstallDefaultPreferences(
        vma_cfg_item_struct *parm_list
)
{
	const gchar *cfg_parm_name;
	gpointer value;
	gchar *strptr;
	const gchar *cstrptr;
	u_int8_t ui8;
	int32_t i32;
	u_int32_t ui32;
	gdouble d;
	vma_color_struct view_color;


	if(parm_list == NULL)
	    return;

#define DO_ADD_PARM	\
{ \
 VMACFGItemListMatchSetValue( \
  parm_list, \
  cfg_parm_name, \
  value, \
  FALSE \
 ); \
}


	/* Version. */
	cfg_parm_name = VMA_CFG_PARM_VERSION_MAJOR;
	ui32 = PROG_VERSION_MAJOR;
	value = &ui32;
	DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VERSION_MINOR;
        ui32 = PROG_VERSION_MINOR;
        value = &ui32;
	DO_ADD_PARM


	/* Locations. */
	/* Local and global data directories. */
	cfg_parm_name = VMA_CFG_PARM_DIR_LOCAL;
	strptr = strdup(dname.data_local);
	value = strptr;
	DO_ADD_PARM
	free(strptr); strptr = NULL;

        cfg_parm_name = VMA_CFG_PARM_DIR_GLOBAL;
        strptr = strdup(dname.data_global);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

	/* Local and global preset models directories. */
        cfg_parm_name = VMA_CFG_PARM_DIR_PRESET_MODELS_LOCAL;
        strptr = strdup(dname.preset_models_local);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        cfg_parm_name = VMA_CFG_PARM_DIR_PRESET_MODELS_GLOBAL;
        strptr = strdup(dname.preset_models_global);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        /* Local and global preset primitives directories. */
        cfg_parm_name = VMA_CFG_PARM_DIR_PRESET_PRIMITIVES_LOCAL; 
        strptr = strdup(dname.preset_primitives_local);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        cfg_parm_name = VMA_CFG_PARM_DIR_PRESET_PRIMITIVES_GLOBAL;
        strptr = strdup(dname.preset_primitives_global);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        /* Local and global preset primitives directories. */
        cfg_parm_name = VMA_CFG_PARM_DIR_PLUGINS_LOCAL;
        strptr = strdup(dname.plugins_local);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        cfg_parm_name = VMA_CFG_PARM_DIR_PLUGINS_GLOBAL;
        strptr = strdup(dname.plugins_global);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

	/* Tempory directory. */
        cfg_parm_name = VMA_CFG_PARM_DIR_TMP;
        strptr = strdup(dname.tmp);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        /* Scratch pad file. */
        cfg_parm_name = VMA_CFG_PARM_FILE_SCRATCHPAD;
        strptr = strdup(fname.scratchpad);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        /* Plugins file. */
        cfg_parm_name = VMA_CFG_PARM_FILE_PLUGINS;
        strptr = strdup(fname.plugins);
        value = strptr;
        DO_ADD_PARM
        free(strptr); strptr = NULL;

        /* Last directories. */
        cfg_parm_name = VMA_CFG_PARM_LAST_DIR_MODELS;
        value = dname.fb_last_models;
        DO_ADD_PARM
        
        cfg_parm_name = VMA_CFG_PARM_LAST_DIR_TEXTURES;
        value = dname.fb_last_textures;
        DO_ADD_PARM
        
        cfg_parm_name = VMA_CFG_PARM_LAST_DIR_VIEW_BACKGROUND_IMAGE;
        value = dname.fb_last_view_background_image;
        DO_ADD_PARM   


	/* Last export file. */
        cfg_parm_name = VMA_CFG_PARM_LAST_FILE_EXPORT;
        value = strdup("");
        DO_ADD_PARM
        free(strptr); strptr = NULL;

	/* Last export extension. */
        cfg_parm_name = VMA_CFG_PARM_LAST_EXT_EXPORT;
        value = strdup("");
        DO_ADD_PARM
        free(strptr); strptr = NULL;


	/* Pointer emulate as 2 button. */
	cfg_parm_name = VMA_CFG_PARM_POINTER_EMULATE_2BUTTON;
	ui8 = FALSE;
	value = &ui8;
	DO_ADD_PARM


	/* Maximum undos (and redos). */
	cfg_parm_name = VMA_CFG_PARM_UNDO_MAX;
        ui32 = VMA_EDITOR_DEF_UNDO_MAX;
        value = &ui32;
        DO_ADD_PARM


	/* Fonts. */
	/* Editable. */
	cfg_parm_name = VMA_CFG_PARM_FONT_NAME_EDITABLE;
        value = strdup(VMA_GTK_FONT_NAME_TEXT_EDITABLE);
        DO_ADD_PARM
        free(strptr); strptr = NULL;

	/* Terminal. */
        cfg_parm_name = VMA_CFG_PARM_FONT_NAME_TERMINAL;
        value = strdup(VMA_GTK_FONT_NAME_TEXT_TERMINAL);
        DO_ADD_PARM
        free(strptr); strptr = NULL;


	/* View. */
	/* View colors. */
	cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_BACKGROUND;
	view_color.a = 1.0;
	view_color.r = 0.0;
	view_color.g = 0.0;
	view_color.b = 0.0;
	value = &view_color;
	DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_POINT;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_LINE;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_LINE_STRIP;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84; 
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_LINE_LOOP;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_TRIANGLE;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_TRIANGLE_STRIP;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_TRIANGLE_FAN;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_QUAD;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_QUAD_STRIP;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_POLYGON;
        view_color.a = 1.0;
        view_color.r = 0.84;
        view_color.g = 0.84;
        view_color.b = 0.84;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_TEXTURE_OUTLINE;
        view_color.a = 1.0;
        view_color.r = 0.0;
        view_color.g = 0.75;
        view_color.b = 1.0;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_TEXTURE_OUTLINE_SELECTED;
        view_color.a = 1.0;
        view_color.r = 0.5; 
        view_color.g = 1.0;
        view_color.b = 1.0; 
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_HEIGHTFIELD;
        view_color.a = 1.0;
        view_color.r = 0.8;
        view_color.g = 0.65;
        view_color.b = 0.6;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_GRID;
        view_color.a = 1.0;
        view_color.r = 0.5;
        view_color.g = 0.5;
        view_color.b = 0.5;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_SELECTED;
        view_color.a = 1.0;
        view_color.r = 1.0;
        view_color.g = 1.0;
        view_color.b = 0.0;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_SELECTED_VERTEX;
        view_color.a = 1.0;
        view_color.r = 1.0;
        view_color.g = 1.0;
        view_color.b = 1.0;
        value = &view_color;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_CURSORY;
        view_color.a = 1.0;
        view_color.r = 1.0;
        view_color.g = 1.0;
        view_color.b = 1.0;
        value = &view_color;
        DO_ADD_PARM

	cfg_parm_name = VMA_CFG_PARM_VIEW_COLOR_NORMAL_VECTOR;
        view_color.a = 1.0;
        view_color.r = 1.0;
        view_color.g = 0.8;
        view_color.b = 0.2;
        value = &view_color;
        DO_ADD_PARM

	/* Size of editor toplevel. */
	cfg_parm_name = VMA_CFG_PARM_WIDTH;
	ui32 = VMA_EDITOR_DEF_WIDTH;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_HEIGHT;
        ui32 = VMA_EDITOR_DEF_HEIGHT;
        value = &ui32;
        DO_ADD_PARM

        /* Size of editor's text dialog (when pulled out). */
        cfg_parm_name = VMA_CFG_PARM_TDIALOG_WIDTH;
        ui32 = VMA_EDITOR_TDIALOG_DEF_WIDTH;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TDIALOG_HEIGHT;
        ui32 = VMA_EDITOR_TDIALOG_DEF_HEIGHT;
        value = &ui32;
        DO_ADD_PARM

	/* Editor panel sizes. */
	/* Vertical length of vbox holding list and view paned widgets. */
        cfg_parm_name = VMA_CFG_PARM_LIST_AND_VIEW_PANED_SIZE;
        ui32 = 480;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL0_SIZE;
        ui32 = 160;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL1_SIZE;
        ui32 = 490;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL2_SIZE;
        ui32 = 276;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL_MODELS_LIST_SIZE;
        ui32 = 155;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL_VIEW_UPPER_SIZE;
        ui32 = 220;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL_VIEW0_SIZE;
        ui32 = 250;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL_VIEW2_SIZE;
        ui32 = 250;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PANEL_VALUES_LIST_SIZE;
        ui32 = 180;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_EDITOR_VALUES_CHWIDTH0;
        ui32 = 135;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_EDITOR_VALUES_CHWIDTH1;
        ui32 = 110;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_EDITOR_VALUES_CHWIDTH2;
        ui32 = 110;
        value = &ui32;
        DO_ADD_PARM


	/* Rotate primitives dialog last values. */
        cfg_parm_name = VMA_CFG_PARM_ROTATE_PRIMITIVE_ANGLE;
        d = DEGTORAD(0.0);
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_ROTATE_PRIMITIVE_ORBITAL_AXIS;
        ui32 = 3;		/* Z axis. */
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_ROTATE_PRIMITIVE_ROTATE_NORMALS;
        ui8 = 1;		/* Yes. */
        value = &ui8;
        DO_ADD_PARM


	/* Translate primitives. */
        cfg_parm_name = VMA_CFG_PARM_TRANSLATE_PRIMITIVE_X;
        d = 0.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TRANSLATE_PRIMITIVE_Y;
        d = 0.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TRANSLATE_PRIMITIVE_Z;
        d = 0.0;
        value = &d;
        DO_ADD_PARM


	/* Scale primitives. */
	cfg_parm_name = VMA_CFG_PARM_SCALE_PRIMITIVE_SCALE_TYPE;
        ui32 = 0;	/* 0 = uniform, 1 = compoent */
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCALE_PRIMITIVE_SCALE_COEFF;
        d = 1.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCALE_PRIMITIVE_SCALE_X;
        d = 1.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCALE_PRIMITIVE_SCALE_Y;
        d = 1.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCALE_PRIMITIVE_SCALE_Z;
        d = 1.0;
        value = &d;
        DO_ADD_PARM


	/* Mirror primitive. */
        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_MIRROR_AXIS;
        ui32 = 2;	/* 1 = X, 2 = Y, 3 = Z */
        value = &ui32;
        DO_ADD_PARM

	cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_AXIS_OFFSET;
        d = 0.0;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_MIRROR_NORMALS;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_MIRROR_TEXCOORDS;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_FLIP_WINDING;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_WIND_NORMALS;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_MIRROR_PRIMITIVE_WIND_TEXCOORDS;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM


	/* Snap primitive. */
	cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_VERTICES;
	ui8 = 1;
	value = &ui8;
	DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_NORMALS;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_TEXCOORDS;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_AXIS_X;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_AXIS_Y;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_AXIS_Z;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

	cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_DIGITS;
	ui8 = 0;		/* 1 = digits, 0 = decimals. */
        value = &ui8;
        DO_ADD_PARM

	cfg_parm_name = VMA_CFG_PARM_SNAP_PRIMITIVE_RESOLUTION;
	ui32 = 2;
        value = &ui32;
        DO_ADD_PARM


	/* Texture browser. */
	cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_WIDTH;
	ui32 = VMA_TEXBROWSER_DEF_WIDTH;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_HEIGHT;
        ui32 = VMA_TEXBROWSER_DEF_HEIGHT;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_PREVIEW_WIDTH;
        ui32 = 220;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_PREVIEW_HEIGHT;
        ui32 = 165;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_INDEX_CHWIDTH;
        ui32 = 45;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_NAME_CHWIDTH;
        ui32 = 120;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_PATH_CHWIDTH;
        ui32 = 180;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXBROWSER_PRIORITY_CHWIDTH;
        ui32 = 20;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_TEXTURE_BASE_DIR;
	cstrptr = getenv("HOME");
	strptr = strdup((cstrptr == NULL) ? "/" : cstrptr);
        value = strptr;
        DO_ADD_PARM
	free(strptr); strptr = NULL;

        cfg_parm_name = VMA_CFG_PARM_TEXTURE_BASE_DIR_OVERRIDE;
        ui8 = 0;	/* No override, let model file define base dir. */
        value = &ui8;
        DO_ADD_PARM


	/* Color selector. */
        cfg_parm_name = VMA_CFG_PARM_CLRSEL_UPDATE_PREVIEW;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_CLRSEL_PREVIEW_QOBJ_TYPE;
        ui32 = 0;		/* Quadric object type, 0 = sphere. */
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_CLRSEL_LIGHT_ORBIT_POSITION;
        d = DEGTORAD(225);
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_CLRSEL_LIGHT_BRIGHTNESS;
        d = 1.0;
        value = &d;   
        DO_ADD_PARM   


	/* Tool bar style. */
	cfg_parm_name = VMA_CFG_PARM_TOOLBAR_STYLE;
        ui8 = 2;	/* Pictures and text. */
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SHOW_TOOLTIPS;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_RECORD_LAST_POS_AND_SIZE;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SHOW_TIPOFDAY;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM


	/* Resolution, vertex winding, and rendering. */
	cfg_parm_name = VMA_CFG_PARM_DATA_DECIMALS_POSITION;
        ui32 = 3;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_DATA_DECIMALS_ANGLE;
        ui32 = 2;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_DECIMALS_POSITION;
        ui32 = 2;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_DECIMALS_ANGLE;
        ui32 = 2;
        value = &ui32;
        DO_ADD_PARM


        cfg_parm_name = VMA_CFG_PARM_VIEW_RENDER;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_CULL_FACES;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_CULL_DIRECTION;
        ui8 = 1;	/* Counter-clockwise. */
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_TRANSLATIONS_STATE;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_ENABLE_ALPHA_CHANNEL;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_CAM_CLIP_NEAR;
        d = 0.0001;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_CAM_CLIP_FAR;
        d = 1000000.0;
        value = &d;  
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_VIEW_CAM_FOV;
        d = DEGTORAD(40.0);
        value = &d;
        DO_ADD_PARM


	/* V3D format options. */
        cfg_parm_name = VMA_CFG_PARM_V3DFMT_PROMPT_ON_SAVE;
        ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_V3DFMT_OPTIMIZATION;
        ui32 = 2;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_V3DFMT_STRIP_EXTRAS;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM


	/* Backup. */
        cfg_parm_name = VMA_CFG_PARM_BACKUP_ON_OPEN;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_BACKUP_ON_SAVE;
        ui8 = 0;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_BACKUP_PERIODIC_INT;
        ui32 = 0;	/* In minutes, 0 for no periodic backups. */
        value = &ui32;
        DO_ADD_PARM

	cfg_parm_name = VMA_CFG_PARM_BACKUP_NEW_INDEX_HIGHEST;
	ui8 = 1;
        value = &ui8;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_BACKUP_MAX;
        ui32 = 5;
        value = &ui32;
        DO_ADD_PARM


	/* Scratch pad. */
        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_WIDTH;
        ui32 = VMA_SCRATCHPAD_DEF_WIDTH;
        value = &ui32;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_HEIGHT;
        ui32 = VMA_SCRATCHPAD_DEF_HEIGHT;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_CLIST_HEIGHT;
        ui32 = VMA_SCRATCHPAD_DEF_HEIGHT * 0.6;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_CH0;
        ui32 = 150;
        value = &ui32;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_CH1;
        ui32 = 150;
        value = &ui32;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_SCRATCHPAD_CH2;
        ui32 = 100;
        value = &ui32;
        DO_ADD_PARM


	/* Print parameters. */
	cfg_parm_name = VMA_CFG_PARM_PRINT_PRINT_TO;
	ui32 = 0;	/* 0 = print to printer, 1 = print to file. */
	value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_COMMAND;
	value = (void *)strdup(DEF_PRINT_COMMAND);
        DO_ADD_PARM
	free(value); value = NULL;

        cfg_parm_name = VMA_CFG_PARM_PRINT_FILE;
        value = (void *)strdup(DEF_PRINT_FILE);
        DO_ADD_PARM
        free(value); value = NULL;

        cfg_parm_name = VMA_CFG_PARM_PRINT_VIEW;
        ui32 = 3;       /* Corresponds to view index. */
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_ORIENTATION;
        ui32 = 0;       /* 0 = portrait, 1 = landscape. */
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_VISUAL;
        ui32 = PRINT_WIN_VISUAL_COLOR;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_PPI_X;
        d = VMA_DEF_PRINT_PPI_X;
        value = &d;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_PRINT_PPI_Y;
        d = VMA_DEF_PRINT_PPI_Y;
        value = &d;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_PAPER_SIZE;
        ui32 = PRINT_WIN_PAPER_SIZE_LETTER;
        value = &ui32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_OUTPUT_OFFSET_X;
        i32 = 0;
        value = &i32;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_PRINT_OUTPUT_OFFSET_Y;
        i32 = 0;
        value = &i32;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_PRINT_OUTPUT_WIDTH;
        ui32 = 640;
        value = &ui32;
        DO_ADD_PARM
        cfg_parm_name = VMA_CFG_PARM_PRINT_OUTPUT_HEIGHT;
        ui32 = 480;
        value = &ui32;
        DO_ADD_PARM


	/* Last directories. */
	cfg_parm_name = VMA_CFG_PARM_LAST_DIR_MODELS;
	value = dname.fb_last_models;
	DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_LAST_DIR_TEXTURES;
        value = dname.fb_last_textures;
        DO_ADD_PARM

        cfg_parm_name = VMA_CFG_PARM_LAST_DIR_VIEW_BACKGROUND_IMAGE;
        value = dname.fb_last_view_background_image;
        DO_ADD_PARM



#undef DO_ADD_PARM

	return;
}
