/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define  DECLARE_GLOBALS			/* force definition of global variables	*/

/*--------------------------------------------------------------------------*/

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <dirent.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "vmelilo.h"

/*--------------------------------------------------------------------------*/
/* Print a Message unless --quiet option specified.
 */

void
message
(	const char	*fmt,
	...
)
{	va_list		ap;

	if (!f_quiet)
	{
		va_start(ap, fmt);
		vfprintf(stdout, fmt, ap);
		va_end(ap);
	}
}

/*--------------------------------------------------------------------------*/
/* Print an Error Message and Exit
 */

void
die
(	const char	*fmt,
	...
)
{	va_list		ap;

	fflush(stdout);
	fputc('\n', stderr);
	va_start(ap, fmt);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	fputc('\n', stderr);

	exit(1);
}

/*--------------------------------------------------------------------------*/

void
error_nomemory
(	void
)
{
	die("Can't allocate memory\n");
}

/*--------------------------------------------------------------------------*/

void
error_open
(	const char	*name
)
{
	die("Can't open file `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_opendir
(	const char	*name
)
{
	die("Can't open directory `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_read
(	const char	*name
)
{
	die("Can't read from file `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_write
(	const char	*name
)
{
	die("Can't write to file `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_seek
(	const char	*name
)
{
	die("Can't seek in file `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_stat
(	const char	*name
)
{
	die("Can't get info about file `%s': %s\n", name, strerror(errno));
}

/*--------------------------------------------------------------------------*/

void
error_ioctl
(	const char	*name,
	const char	*ioctl
)
{
	die("Operation `%s' failed for file `%s': %s\n",
		ioctl, name, strerror(errno));
}

/*--------------------------------------------------------------------------*/
/* Write a File
 */

static
void
write_file
(	const char	*name,
	const void	*data,
	int			size
)
{	int			fh;

	if (f_test)
	{
		message("Test mode - file `%s' not written.\n", name);
		return;
	}

	if (f_verbose)
	{
		message("Writing file `%s'.\n", name);
	}

	if ((fh = open(name, O_CREAT|O_RDWR|O_TRUNC, S_IRUSR|S_IWUSR)) == -1)
	{
		error_open(name);
	}

	if (write(fh, data, size) != size)
	{
		error_write(name);
	}

	close(fh);
}

/*--------------------------------------------------------------------------*/
/* Read a Boot Block from a Device or File
 */

static
void
read_boot_block
(	const char	*name
)
{	int			fh;

	if (f_verbose)
	{
		message("Reading boot block from `%s'.\n", name);
	}

	if ((fh = open(name, O_RDONLY)) == -1)
	{
		error_open(name);
	}

	if (read(fh, boot_block_buffer, boot_block_size) != boot_block_size)
	{
		error_read(name);
	}

	close(fh);
}

/*--------------------------------------------------------------------------*/
/* Write a Boot Block to a Device
 */

static
void
write_boot_block
(	const char		*name
)
{	int				fh;
	int				i;
	int				j;

	if (f_test)
	{
		unsigned char	*cp = (unsigned char *)boot_block_buffer;

		message("Test mode - boot block for device %s - not written.\n",
			name);

		if (f_install)
			message("Boot loader file not yet written, boot block inaccurate.\n");

		for (i = 0; i < boot_block_size; i += 16, cp += 16)
		{
			int max = boot_block_size - i;
			if (max > 16) max = 16;
			message("%03X:", i);
			for (j = 0; j < 16; j++)
			{
				if ((j & 1) == 0)
					message(" ");
				if (j < max)
					message("%02X", cp[j]);
				else
					message("  ");
			}
			message("  ");
			for (j = 0; j < max; j++)
				message("%c", (cp[j] < ' ' || cp[j] > '~') ? '.' : cp[j]);
			message("\n");
		}
	}
	else
	{
		if (f_verbose)
		{
			message("Writing boot block to device `%s'.\n", name);
		}
	
		if ((fh = open(name, O_RDWR)) == -1)
		{
			error_open(name);
		}
	
		if (write(fh, boot_block_buffer, boot_block_size) != boot_block_size)
		{
			error_write(name);
		}
	
		close(fh);
	}
}

/*--------------------------------------------------------------------------*/
/* Backup the Boot Block if it wasn't backed up before
 */

static
void
backup_boot_block
(	void
)
{	struct stat info;

	if (stat(backup_file, &info) == -1 || f_backup)
	{
		write_file(backup_file, boot_block_buffer, boot_block_size);
		if (f_verbose)
		{
	    	message("Backup copy of boot sector in `%s'.\n", backup_file);
	    }
	}
	else if (f_verbose)
	{
	    message("`%s' exists - no backup copy made.\n", backup_file);
	}
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal ignoring case
 */

int
equal_strings
(	const char	*s1,
	const char	*s2
)
{
	return s1 && s2 && (strcasecmp(s1, s2) == 0);
}

/*--------------------------------------------------------------------------*/
/* Check whether two strings exist and are equal including case
 */

int
case_equal_strings
(	const char	*s1,
	const char	*s2
)
{
	return s1 && s2 && (strcmp(s1, s2) == 0);
}

/*--------------------------------------------------------------------------*/
/*
 */

static
void
mem_file_create
(	MEMFILE		*mf
)
{
	mf->maxsize = 0;
	mf->cursize = 0;
	mf->data    = NULL;
}

/*--------------------------------------------------------------------------*/

static
void
mem_file_write
(	MEMFILE		*mf,
	const void	*data,
	int			count
)
{
	/* have we enough memory allocated */
	if (mf->maxsize < (mf->cursize + count))
	{
		/* add enough for new data and 1K extra */
		mf->maxsize += (count + 1024);

		/* get more memory */
		if ((mf->data = realloc(mf->data, mf->maxsize)) == NULL)
		{
			error_nomemory();
		}
	}

	/* append the data */
	memcpy(&mf->data[mf->cursize], data, count);

	/* count it */
	mf->cursize += count;
}

/*--------------------------------------------------------------------------*/
/* Write a Tag Record to the Map Data
 */

static
void
write_tag_data
(	MEMFILE		*mf,
	u_long		tagid,
	const void	*data,
	u_long		size
)
{   TAGRECORD	tr;

    tr.tagid = tagid;
    tr.size  = size;

    mem_file_write(mf, &tr, sizeof(tr));
    if (size)
    {
    	mem_file_write(mf, data, size);
    }
}

/*--------------------------------------------------------------------------*/

static
void
write_tag_string
(	MEMFILE		*mf,
	u_long		tagid,
	const char	*s
)
{	TAGRECORD	tr;
	u_long		size;
	char		pad[3];

	size	 = strlen(s) + 1;
	tr.tagid = tagid;
	tr.size  = (size + 3) & ~3;

	mem_file_write(mf, &tr, sizeof(tr));
	mem_file_write(mf, s, size);

	/* get amount to pad */
	if ((size = tr.size - size) != 0)
	{
		pad[0] = pad[1] = pad[2] = 0;
		mem_file_write(mf, pad, size);
	}
}

/*--------------------------------------------------------------------------*/
/* Create a Complete File Name
 */

static
char *
create_file_name
(	const char	*name
)
{	int			size;
	char		*s;

	size = strlen(name) + 1;

	if (root_path)
	{
		size += strlen(root_path);
	}

	if ((s = malloc(size)) == NULL)
	{
		error_nomemory();
	}

	if (root_path)
	{
		strcpy(s, root_path);
		strcat(s, name);
	}
	else
	{
		strcpy(s, name);
	}

	return s;
}

/*--------------------------------------------------------------------------*/

static
int
check_file_access
(	const char		*path
)
{	int				ret;
	char			*fullpath;

	fullpath = create_file_name(path);
	ret      = access(fullpath, R_OK) == 0;
	free(fullpath);

	return ret;
}

/*--------------------------------------------------------------------------*/
/* Return FALSE if 'path' exists.
 * Return TRUE  if 'path' does not exist but boot record is optional.
 * otherwise die().
 */

static
int
die_unless_optional
(	const BOOTRECORD	*record,
	const char			*path,
	const char			*description
)
{	int					optional;

	if (check_file_access(path))
	{
		/* exists */
		return FALSE;
	}

	optional = record->optional
			 ? *record->optional
			 : FALSE;

	if (!optional)
	{
		/* does not exist, but not optional */
		die("%s - %s file '%s' is not available.\n",
					record->label, description, path);
	}
	
	if (f_verbose)
	{
		message("   %-32.32s -> ** Skipped **\n", record->label);
	}

	/* dereference files we are not going to use */
	remove_file_def(with_default(record, kernel));
	remove_file_def(record->ramdisk             );
	remove_file_def(record->symtab              );

	/* does not exist, but optional */
	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Create the Map Data
 */

static
void
create_map_data
(	void
)
{	const BOOTRECORD	*record;
	const BOOTRECORD	*defrecord = NULL;
	FILEDEF				*file;
	const char			*kernel;
	const char			*password;
	const char			*cmdline;
	const char			*append;
	const char			*console;
	const char			*root;
	const u_long		*read_only;
	const u_long		*restricted;
	int					records = 0;

#define WRITE_TAG_STRING(f, tag, arg) \
	do	{if (arg) write_tag_string(f, tag, arg);} while(0)

#define WRITE_TAG_DATA(f, tag, arg, size) \
	do	{if (arg) write_tag_data(f, tag, arg, size);} while(0)

	mem_file_create(&map_data);

	/* file identification */
	write_tag_data(&map_data, TAG_LILO, NULL, 0);

	/* boot options */
	write_tag_data  (&map_data, TAG_HEADER,     NULL,                        0);
	WRITE_TAG_STRING(&map_data, TAG_DEFAULT,    config.options.boot_default   );
	WRITE_TAG_STRING(&map_data, TAG_MESSAGE,    config.options.boot_message   );
	WRITE_TAG_STRING(&map_data, TAG_MASTER_PASSWORD,
												config.options.boot_masterpswd);
	WRITE_TAG_DATA  (&map_data, TAG_PROMPT,     config.options.boot_prompt,  4);
	WRITE_TAG_DATA  (&map_data, TAG_DELAY,      config.options.boot_delay,   4);
	WRITE_TAG_DATA  (&map_data, TAG_TIMEOUT,    config.options.boot_timeout, 4);
	WRITE_TAG_DATA  (&map_data, TAG_DEBUG,      config.options.boot_debug,   4);
	write_tag_data  (&map_data, TAG_HEADER_END, NULL,                        0);

	/* boot records */
	if (f_verbose)
	{
		message("[Boot Records]\n");
	}
	for (record = config.records; record; record = record->next)
	{
		password	= with_default(record, password  );
		restricted	= with_default(record, restricted);
		cmdline		= with_default(record, cmdline   );
		append		= with_default(record, append    );
		kernel		= with_default(record, kernel    );
		console		= with_default(record, console   );
		root		= with_default(record, root      );
		read_only	= with_default(record, read_only );

		/* don't write a zero lenth password */
		if (password && (*password == '\0'))
		{
			password = NULL;
		}

		/* don't write restricted flag if FALSE */
		if (restricted && (*restricted == FALSE))
		{
			restricted = NULL;
		}

		/* must specify a kernel */
		if ((kernel == NULL) || (*kernel == '\0'))
		{
			die("No kernel image specified for `%s'\n", record->label);
		}

		/* does kernel image exist */
		if (die_unless_optional(record, kernel, "kernel image"))
		{
			continue;
		}
	
		if (record->ramdisk && *record->ramdisk)
		{
			if (die_unless_optional(record, record->ramdisk, "ramdisk image"))
			{
				continue;
			}
		}

		if (record->symtab && *record->symtab)
		{
			if (die_unless_optional(record, record->symtab, "kernel symbol"))
			{
				continue;
			}
		}

		/* find default boot record as we go */
		if (equal_strings(config.options.boot_default, record->label)
		||	equal_strings(config.options.boot_default, record->alias))
		{
			defrecord = record;
		}

		if (f_verbose)
		{
			message("  %c%-32.32s -> %s\n",
				(defrecord == record) ? '*' : ' ',
				record->label,
				kernel);
			if (record->ramdisk && *record->ramdisk)
				message("%36s-> %s\n", "", record->ramdisk);
			if (record->symtab && *record->symtab)
				message("%36s-> %s\n", "", record->symtab);
		}
		else
		{
			message("  %c%s\n",
				(defrecord == record) ? '*' : ' ',
				record->label);
		}

		write_tag_string(&map_data, TAG_BOOT_RECORD,     record->label        );
		WRITE_TAG_STRING(&map_data, TAG_ALIAS,           record->alias        );
		WRITE_TAG_STRING(&map_data, TAG_KERNEL,          kernel               );
		WRITE_TAG_STRING(&map_data, TAG_PASSWORD,        password             );
		WRITE_TAG_DATA  (&map_data, TAG_RESTRICTED,      restricted,         4);
		WRITE_TAG_STRING(&map_data, TAG_ARGUMENTS,       cmdline              );
		WRITE_TAG_STRING(&map_data, TAG_RAMDISK,         record->ramdisk      );
		WRITE_TAG_STRING(&map_data, TAG_SYMTAB,          record->symtab       );
		WRITE_TAG_DATA  (&map_data, TAG_MEMSIZE,         record->memsize,    4);
		WRITE_TAG_DATA  (&map_data, TAG_CALLMONITOR,     record->callmonitor,4);
		WRITE_TAG_STRING(&map_data, TAG_APPEND,          append               );
		WRITE_TAG_STRING(&map_data, TAG_ROOT,            root                 );
		WRITE_TAG_STRING(&map_data, TAG_CONSOLE,         console              );
		WRITE_TAG_DATA  (&map_data, TAG_READ_ONLY,       read_only,          4);
		write_tag_data  (&map_data, TAG_BOOT_RECORD_END, NULL,               0);
		records++;
	}

	if (records == 0)
	{
		die("There are no bootable configurations.\n");
	}

	if (!defrecord)
	{
		die("Default boot record `%s' does not exist.\n",
			config.options.boot_default);
	}

	if (f_verbose)
	{
		message("[File Definitions]\n");
	}
	for (file = config.files; file; file = file->next)
	{
		char	*path;

		/* skip file if it is not in any active configuration */
		if (file->refcount == 0)
		{
			continue;
		}

		path = create_file_name(file->path);

		if ((file->map = create_file_map(path)) == NULL)
		{
			if (case_equal_strings(defrecord->kernel, file->path))
			{
				die("Kernel image `%s' for default boot record `%s' does not "
						"exist.\n", path, defrecord->label);
			}
			else
			{
				message("Warning: file `%s' doesn't exist.\n", path);
			}
		}
		else
		{
			if (f_verbose)
			{
				message("  %s (%ld bytes in %ld fragments)\n",
						path, MAP_FILESIZE(file->map), MAP_NUMFRAGS(file->map));
			}

			write_tag_string(&map_data, TAG_FILE_DEF, file->path);
			write_tag_data  (&map_data, TAG_FILEMAP,  file->map,
												MAP_MAPSIZE(file->map));
			write_tag_data  (&map_data, TAG_FILE_DEF_END, NULL, 0);
		}

		free(path);
	}

	/* end of file */
	write_tag_data(&map_data, TAG_EOF, NULL, 0);

	if (!f_verbose)
	{
		message("\n");
	}
}

/*--------------------------------------------------------------------------*/
/* Write the Loader
 */

static
void
write_loader
(	void
)
{	int						fh;
	int						i;
	const u_long			pattern[]	= { LILO_ID, LILO_MAPDATAID };
	u_long					size;

	i = (loader_code_size - sizeof(pattern)) + 1;
	while (i--)
	{
		if (memcmp(&loader_code[i], pattern, sizeof(pattern)) == 0)
		{
			/* get offset immediately following pattern */
			size = i + sizeof(pattern);
			goto found_it;
		}
	}

	die("Couldn't find magic key in loader template.\n");

found_it:

	if (f_verbose)
	{
		message(f_test ? "Test mode - loader file '%s' - not written.\n"
					  : "Writing loader to file `%s'.\n",
				 loader_file);
	}

	if (f_test)
	{
		return;
	}

	/* create loader file */
	if ((fh = open(loader_file, O_CREAT|O_RDWR|O_TRUNC, S_IRUSR|S_IWUSR)) == -1)
	{
		error_open(loader_file);
	}

	/* write loader code */
	if (write(fh, loader_code, size) != size)
	{
		error_write(loader_file);
	}

	/* write map data size */
	size = map_data.cursize;
	if (write(fh, &size, sizeof(size)) != sizeof(size))
	{
		error_write(loader_file);
	}

	/* append map data */
	if (write(fh, map_data.data, size) != size)
	{
		error_write(loader_file);
	}

	close(fh);
}

/*--------------------------------------------------------------------------*/
/* Determine hardware type
 */

static
void
get_m68k_model
(	void
)
{	FILE	*f;
	char	line[128];
	char	*model = NULL;

	if (m68k_model == -1)
	{
		/* open /proc/hardware */
		if ((f = fopen( "/proc/hardware", "r" )) == NULL)
		{
			die("Can't open /proc/hardware: %s\nmachine type unknown!\n",
						strerror(errno));
		}
	
		/* read until we see a line that starts with "Model:" */
		while (fgets(line, sizeof(line) - 1, f))
		{
			if (strncmp(line, "Model:", 6) == 0)
			{
				/* skip whitespace after ':' */
				model = skip_white(line + 6);
	
				/* terminate at newline */
				model[strlen(model) - 1] = '\0';
	
				break;
			}
		}
		fclose(f);
	
		if (model == NULL || !*model)
		{
			die("Unable to determine m68k machine type!\n");
		}
	
		if (strncmp(model, "BVME", 4) == 0)
		{
			m68k_model = MODEL_BVME;
		}
		else if (strncmp(model, "Motorola", 8) == 0)
		{
			m68k_model = MODEL_MVME;
		}

		else die("Unsupported machine type: %s\n", model);
	}

	if (f_verbose)
	{
		switch (m68k_model)
		{
			case MODEL_BVME: model = "BVM BVME4000/6000"; break;
			case MODEL_MVME: model = "Motorola MVME SBC"; break;
		}
		message("VME Architecture: %s\n", model);
	}
}

/*--------------------------------------------------------------------------*/
/* Print the Usage Template and Exit
 */

static
void
usage
(	void
)
{
	die(
"%s\n"
"Usage: %s [options]\n"
"Valid options are:\n"
"    -h, --help                Display this usage information\n"
"    -v, --verbose             Enable verbose mode\n"
"    -q, --quiet               Operate silently\n"
"    -a, --arch id             Force use of specific boot loader,\n"
"                              where id is: bvme or mvme\n"
"                               bvme - BVME4000/6000\n"
"                               mvme - Motorola MVME\n"
"    -f, --force               Force installation over an unknown boot block\n"
"    -B, --force-backup        Force a boot block backup on installation\n"
"    -b, --backup-file file    Write boot block backup to file\n"
"    -u, --uninstall           Uninstall the boot block\n"
"    -c, --config-file file    Read configuration settings from file\n"
"    -l, --loader-file file    Boot loader name (default /boot/boot.loader)\n"
"    -w, --root prefix         Use another root directory\n"
"    -r, --restore-from file   Restore boot block from file\n"
"    -s, --save-to file        Save boot block to file\n"
"    -d, --device device       Override the block special device\n"
"    -t, --test                Test (read-only) mode\n"
"    -V, --version             Display version information\n",
	LiloVersion, program_name);
}

/*--------------------------------------------------------------------------*/

static
void
parse_opts
(	int			argc,
	char		*argv[]
)
{	static const char			s_opts[] = "BVa:b:c:d:fhl:qr:s:tuvw:"; 
	static const struct option	l_opts[] = {
		{"force-backup",		no_argument,			NULL,	'B'},
		{"version",				no_argument,			NULL,	'V'},
		{"arch",				required_argument,		NULL,	'a'},
		{"backup-file",			required_argument,		NULL,	'b'},
		{"config-file",			required_argument,		NULL,	'c'},
		{"device",				required_argument,		NULL,	'd'},
		{"force",				no_argument,			NULL,	'f'},
		{"help",				no_argument,			NULL,	'h'},
		{"loader-file",			required_argument,		NULL,	'l'},
		{"quiet",				no_argument,			NULL,	'q'},
		{"restore-from",		required_argument,		NULL,	'r'},
		{"save-to",				required_argument,		NULL,	's'},
		{"test",				no_argument,			NULL,	't'},
		{"uninstall",			no_argument,			NULL,	'u'},
		{"verbose",				no_argument,			NULL,	'v'},
		{"root",				required_argument,		NULL,	'w'},
		{NULL}
	};
	int							c;
	int							opt_index;
	const char					*device    = NULL;

    program_name = argv[0];

	while (1)
	{
		opt_index = -1;
		c         = getopt_long(argc, argv, s_opts, l_opts, &opt_index);

		if (c == EOF)
		{
			break;
		}

		switch (c)
		{
			default :
			{
				die("Try %s --help\n", program_name);
			}
	
			case 'B':
			{
				f_backup = TRUE;
				break;
			}

			case 'V':
			{
			    die(LiloVersion);
				break;
			}

			case 'a':
			{
				if (m68k_model != -1) goto duplicate_option;

				if      (strcasecmp(optarg, "bvme") == 0)
					m68k_model = MODEL_BVME;
				else if (strcasecmp(optarg, "mvme") == 0)
					m68k_model = MODEL_MVME;
				else
					die("Unrecognised architecture specified '%s'\n", optarg);
				break;
			}

			case 'b':
			{
				if (backup_file) goto duplicate_option;
				backup_file = optarg;
				break;
			}

			case 'c':
			{
				if (config_file) goto duplicate_option;
				config_file = optarg;
				break;
			}

			case 'd':
			{
				if (device) goto duplicate_option;
				device = optarg;
				break;
			}

			case 'f':
			{
			    f_force = TRUE;
				break;
			}

			case 'h':
			{
			    usage();
				break;
			}

			case 'l':
			{
				if (loader_file) goto duplicate_option;
				loader_file = optarg;
				break;
			}

			case 'r':
			{
				if (restore_boot_block) goto duplicate_option;
				restore_boot_block = optarg;
				break;
			}

			case 's':
			{
				if (save_boot_block) goto duplicate_option;
				save_boot_block = optarg;
				break;
			}

			case 't':
			{
			    f_test = TRUE;
				break;
			}

			case 'u':
			{
			    f_uninstall = TRUE;
				break;
			}

			case 'v':
			{
			    f_verbose = TRUE;
				break;
			}

			case 'q':
			{
			    f_quiet = TRUE;
				break;
			}

			case 'w':
			{
				if (root_path) goto duplicate_option;
				root_path = optarg;
				break;
			}	
		}
	}

	if (optind < argc)
	{
		die("One or more spurious arguments specified\n"); 
	}

	/* if not uninstalling, restoring or saving */
    if (!f_uninstall && !restore_boot_block && !save_boot_block)
	{
		/* we must be installing */
   	    f_install = TRUE;
    }

    /* don't allow backup options if we're not installing */
    if (!f_install && (f_backup || backup_file))
	{
		die("You can't specify --backup or --force-backup,\n"
			"with --uninstall, --restore-from or --save-to\n");
	}

	/* don't allow restore and uninstall */
    if (restore_boot_block && f_uninstall)
	{
		die("You can't specify both --restore-from and --uninstall\n");
	}

    if (f_verbose)
    {
		message(LiloVersion);
	}

	/* if no configuration file specified */
	if (config_file == NULL)
	{
		/* use the default */
		config_file = create_file_name(LILO_CONFIGFILE);
	}

	/* create pathlist for writing boot loader to */
	loader_file = create_file_name(loader_file ? loader_file : LILO_LOADERFILE);

	if (save_boot_block)
	{
		save_boot_block = create_file_name(save_boot_block);
	}

    if (restore_boot_block)
    {
		restore_boot_block = create_file_name(restore_boot_block);
	}

    read_config_file();

    /* override configuration file device */
    if (device)
	{
        if (f_verbose)
        {
            message("Using specified device `%s'.\n", device);
        }

        config.boot_device_name = device;
    }
    else if (config.boot_device_name == NULL)
   	{
   		die("No device given on command line or config file\n");
    }

	return;

duplicate_option:
	if (opt_index == -1)
		for (opt_index = 0; l_opts[opt_index].val != c; opt_index++)
			;

	die("You should only specify the --%s, -%c option once\n",
		l_opts[opt_index].name, c);	
}

/*--------------------------------------------------------------------------*/
/* Main Routine
 */

int
main
(	int			argc,
	char		*argv[]
)
{	struct stat	info;

	/* parse command line */
	parse_opts(argc, argv);

	/* determine hardware type if not specified on command line */
	get_m68k_model();

	/* allocate boot block buffer */
	if ((boot_block_buffer = malloc(boot_block_size)) == NULL)
	{
		error_nomemory();
	}

    if (stat(config.boot_device_name, &info) == -1)
    {
		error_stat(config.boot_device_name);
	}

    if (!S_ISBLK(info.st_mode))
    {
		die("%s is not a block special device\n", config.boot_device_name);
	}

	if (is_scsi_partition(info.st_rdev))
	{
		die("%s is a SCSI disk partition\n", config.boot_device_name);
	}

    boot_device = info.st_rdev;

    if (backup_file == NULL)
    {
   		backup_file = create_backup_filename();
   	}

    read_boot_block(config.boot_device_name);

    /* allow save of invalid boot block without force */
    if (!save_boot_block)
    {
	    if (!f_force && !valid_boot_block(boot_block_buffer))
	    {
		    message("%s has a non %s boot sector.\n",
					config.boot_device_name, monitor_name);
	    	die("Use --force option to continue.\n");
		}
    }

    if (save_boot_block)
	{
		write_file(save_boot_block, boot_block_buffer, boot_block_size);
		message("Boot block saved to file `%s'.\n", save_boot_block);
    }

    else if (restore_boot_block)
	{
		read_boot_block(restore_boot_block);
		message("Boot block read from file `%s'.\n", restore_boot_block);
		write_boot_block(config.boot_device_name);
    }

    else if (f_uninstall)
	{
		read_boot_block(backup_file);
		message("Original boot block read from file `%s'.\n", backup_file);
		write_boot_block(config.boot_device_name);
    }

    else if (f_install)
	{
		backup_boot_block();
		create_map_data  ();
		write_loader     ();

		if ((loader_map = create_file_map(loader_file)) == NULL)
		{
			if (!f_test)
			{
				error_open(loader_file);
			}
		}

		if (f_verbose)
		{
			puts("Creating boot block.");
		}

		if (!valid_boot_block(boot_block_buffer))
		{
			memset(boot_block_buffer, 0, boot_block_size);
		}
		init_boot_block(boot_block_buffer);

		write_boot_block(config.boot_device_name);
    }

    exit(0);
}

/*-----------------------------< end of file >------------------------------*/
