/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#define LILO_ID					(0x4C494C4F)	/* `LILO'					*/
#define LILO_MAPDATAID			(0x4D415044)	/* `MAPD'					*/

#define TRUE					(1)
#define FALSE					(0)

#define arraysize(x)			(sizeof(x) / sizeof(*(x)))

/*--------------------------------------------------------------------------*/
/* File Map Entry Definition
 */
#define SECTOR_SIZE		512

typedef struct {
	unsigned long	offset;				/* Device sector offset				*/
	unsigned long	length;				/* Length in sectors				*/

} FILEMAP;

#define MAP_FILESIZE(map)	(map[0].offset)
#define MAP_NUMFRAGS(map)	(map[0].length)
#define MAP_FIRSTFRAG(map)	(&map[1])
#define MAP_MAPSIZE(map)	((MAP_NUMFRAGS(map) + 1) * sizeof(FILEMAP))

/*--------------------------------------------------------------------------*/
/*		Tag Record
 *
 *		The Map File consists of a concatenation of Tag Records
 *
 *			Tag							Size	Data
 *			---							----	----
 *
 *		File Identification:
 *
 *			TAG_LILO					0		-
 *
 *		Boot Options:
 *
 *			TAG_HEADER					0		-
 *			TAG_DEFAULT							char []
 *			TAG_MESSAGE							char []
 *			TAG_MASTER_PASSWORD					char []
 *			TAG_PROMPT					4		unsigned long
 *			TAG_DELAY					4		unsigned long
 *			TAG_DEBUG					4		unsigned long
 *			TAG_HEADER_END				0		-
 *
 *		Boot Records:
 *
 *			TAG_BOOT_RECORD						char []
 *			TAG_ALIAS							char []
 *			TAG_KERNEL							char []
 *			TAG_ARGUMENTS						char []
 *			TAG_APPEND							char []
 *			TAG_PASSWORD						char []
 *			TAG_RAMDISK							char []
 *			TAG_SYMTAB							char []
 *			TAG_MEMSIZE					4		unsigned long
 *			TAG_CONSOLE							char []
 *			TAG_ROOT							char []
 *			TAG_READ_ONLY				4		unsigned long
 *			TAG_BOOT_RECORD_END			0		-
 *
 *		File Records:
 *
 *			TAG_FILE_DEF						char []
 *			TAG_FILEMAP							unsigned long []
 *			TAG_FILE_DEF_END			0		-
 *
 *		End of File:
 *
 *			TAG_EOF						0		-
 *
 */

typedef struct
{
	unsigned long	tagid;				/* Tag Identifier					*/
	unsigned long	size;				/* Must be a multiple of 4 bytes	*/
	unsigned long	data[0];			/* Size bytes of Tag Data			*/
} TAGRECORD;

/*--------------------------------------------------------------------------*/
/* Tag Identifiers
 */

#define TAG_LILO				LILO_ID /* File Identification				*/
#define TAG_EOF					(0)		/* End of File						*/

#define TAG_HEADER				(1000)	/* Start of Header					*/
#define TAG_HEADER_END			(1001)	/* End of Header					*/
#define TAG_DEFAULT				(1002)	/* Default Boot Image				*/
#define TAG_PROMPT				(1003)	/* Force Boot Prompt Flag			*/
#define TAG_DELAY				(1004)	/* Delay Value						*/
#define TAG_TIMEOUT				(1005)	/* Timeout Value					*/
#define TAG_DEBUG				(1006)	/* Debug Flag						*/
#define TAG_MESSAGE				(1007)	/* Boot message file name			*/
#define TAG_MASTER_PASSWORD		(1008)	/* Password to enable master mode	*/

#define TAG_BOOT_RECORD			(2000)	/* Start of Boot Record				*/
#define TAG_BOOT_RECORD_END		(2001)	/* End of Boot Record 				*/
#define TAG_ALIAS				(2002)	/* Alias Label Name					*/
#define TAG_KERNEL				(2003)	/* Kernel Image						*/
#define TAG_ARGUMENTS			(2004)	/* Boot Arguments					*/
#define TAG_PASSWORD			(2005)	/* Password							*/
#define TAG_RESTRICTED			(2006)	/* Need Password For CmdLine Args	*/
#define TAG_RAMDISK				(2007)	/* Ramdisk Image					*/
#define	TAG_MEMSIZE				(2008)	/* DRAM RAM size in bytes			*/
#define TAG_SYMTAB				(2009)	/* Symbol table file (MAP)			*/
#define TAG_CALLMONITOR			(2010)	/* Call ROM monitor before boot		*/
#define TAG_APPEND				(2011)	/* Additional Boot Arguments		*/
#define TAG_CONSOLE				(2012)	/* Console device name				*/
#define TAG_ROOT				(2013)	/* Root file system device name		*/
#define TAG_READ_ONLY			(2014)	/* read-only/read-write root flag	*/

#define TAG_FILE_DEF			(3000)	/* Start of File Definition			*/
#define TAG_FILE_DEF_END		(3001)	/* End of File Definition			*/
#define TAG_FILEMAP				(3002)	/* File Map							*/

/*--------------------------------------------------------------------------*/
/*	Global Boot Options
 */

typedef struct {
	const char			*boot_default;	/* Default Boot Image				*/
	const unsigned long	*boot_prompt;	/* Force Boot Prompt Flag			*/
	const unsigned long	*boot_delay;	/* In Seconds, Zero no delay		*/
	const unsigned long	*boot_timeout;	/* In Seconds, Zero no timout		*/
	const unsigned long	*boot_debug;	/* Debug Flag						*/
	const char			*boot_message;	/* Boot message file name			*/
	const char			*boot_masterpswd;	/* Master mode password			*/

} BOOTOPTIONS;

/*--------------------------------------------------------------------------*/
/* Boot Record Definitions
 */

typedef struct bootrecord {
	struct bootrecord	*next;			/* Next Boot Record					*/
	const char			*label;			/* Boot Record Label				*/
	const char			*alias;			/* Alias Label Name					*/
	const char			*kernel;		/* Kernel Image						*/
	const char			*cmdline;		/* Default Boot Arguments			*/
	const char			*append;		/* Extra Boot Arguments				*/
	const char			*password;		/* Password							*/
	const char			*ramdisk;		/* Ramdisk							*/
	const char			*symtab;		/* Symbol Table						*/
	const char			*console;		/* Console device name				*/
	const char			*root;			/* Root file system device name		*/
	const unsigned long	*read_only;		/* Root is read-only or read-write	*/
	const unsigned long	*memsize;		/* DRAM RAM size in bytes			*/
	const unsigned long	*restricted;	/* Need password for boot cmdline	*/
	const unsigned long	*callmonitor;	/* Enter debugger prior to boot		*/
	/* The following fields are only used by the vmelilo.conf configuration
	 * parser and are not used by the kernel loader.
	 */
#ifdef CONFIG_PARSER
	const unsigned long *optional;		/* Boot record is optional			*/
#endif
} BOOTRECORD;

/*--------------------------------------------------------------------------*/
/* File Definition
 */

typedef struct filedef {
	struct filedef	*next;				/* Next File Definition				*/
	const char		*path;				/* File Path						*/
	const FILEMAP	*map;				/* File Map							*/
#ifdef CONFIG_PARSER
	int				refcount;			/* Reference count					*/
#endif
} FILEDEF;

/*-----------------------------< end of file >------------------------------*/
