// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: CommunicationManagerFactory.cpp
// 
//---------------------------------------------------------------------------
#include "DefaultPhysicalCommunicationLayer.h"
#include "CommunicationManagerFactory.h"
#include "DefaultCommunicationManager.h"
#include "MsgAggregatingCommunicationManager.h"
#include "TimeWarpSimulationManager.h"
#include "SimulationConfiguration.h"
#include <clutils/ConfigurationScope.h>
#include <clutils/Debug.h>
#include "warped-config.h"

#ifdef USE_ECLMPL
#include "eclmpl/eclmpl.h"
#endif

CommunicationManagerFactory::CommunicationManagerFactory(){}

CommunicationManagerFactory::~CommunicationManagerFactory(){
}

// Note: configure() performs two configurations steps:
// (a) it first configures the physical communication library
// (b) then it configures the communication manager (passing in the
//     physical communication library as a parameter)
Configurable *
CommunicationManagerFactory::allocate( SimulationConfiguration &configuration,
				       Configurable *parent ) const {
  ASSERT( parent != 0 );
  TimeWarpSimulationManager *mySimulationManager = 
    dynamic_cast<TimeWarpSimulationManager *>( parent );

  PhysicalCommunicationLayer *myPhysicalCommunicationLayer = NULL;
  Configurable *retval = 0;
  
  // first configure what physical communication layer we are going to use ...
  if( configuration.physicalLayerIs( "MPI" ) ||
      configuration.physicalLayerIs( "UDPSelect" ) ||
      configuration.physicalLayerIs( "TCPSelect" ) ){
    myPhysicalCommunicationLayer = allocatePhysicalCommunicationLayer( configuration.getPhysicalLayerType() );
    clutils::debug << "(" << mySimulationManager->getSimulationManagerID() << ") ";
    if( myPhysicalCommunicationLayer == 0 ){
      mySimulationManager->shutdown( "Could not allocate physical layer corresponding to " +
				     configuration.getPhysicalLayerType() +
				     ", perhaps not configured at compile time?" );
    }
    else{
      clutils::debug << "configured the " << configuration.getPhysicalLayerType()
		     << " Physical Communication Layer" << endl;
    }
  }
  else{
    myPhysicalCommunicationLayer = new DefaultPhysicalCommunicationLayer();
    clutils::debug << "configured the default physical communication layer" << endl;
  }

  myPhysicalCommunicationLayer->physicalInit( configuration );
  if( configuration.communicationManagerIs( "DEFAULT" ) ){
    retval = new DefaultCommunicationManager( myPhysicalCommunicationLayer,
					      mySimulationManager );
    clutils::debug << "configured the default communication manager" << endl;
  }
  else if ( configuration.communicationManagerIs( "MessageAggregating" ) ){
    retval = new MsgAggregatingCommunicationManager( myPhysicalCommunicationLayer,
						     mySimulationManager );
    clutils::debug << "configured a message aggregating communication manager" << endl;
  }
  else {
    mySimulationManager->shutdown( "Unknown CommunicationManager type \"" +
				   configuration.getCommunicationManagerType() );
  }

  return retval;
}

const CommunicationManagerFactory *
CommunicationManagerFactory::instance(){
  static const CommunicationManagerFactory *singleton = new CommunicationManagerFactory();

  return singleton;
}


PhysicalCommunicationLayer *
CommunicationManagerFactory::
allocatePhysicalCommunicationLayer(const string &physicalLayer){
  PhysicalCommunicationLayer *retval = 0;
#ifdef USE_ECLMPL
#ifdef USE_MPI
  if( stringCaseCompare( physicalLayer, "MPI" ) ) {
    retval = new MPIPhysicalCommunicationLayer();
  }
  else 
#endif
  if( stringCaseCompare( physicalLayer, "UDPSELECT" ) ) {
    retval = new UDPSelectPhysicalCommunicationLayer();
  }
  else if( stringCaseCompare( physicalLayer, "TCPSELECT" ) ) {
    retval = new TCPSelectPhysicalCommunicationLayer();
  }
#endif
  return retval;
}
