/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "separatorcombobox.h"

#include <QItemDelegate>
#include <QStandardItemModel>
#include <QStandardItem>
#include <QListView>
#include <QPainter>
#include <QKeyEvent>

using namespace WebIssues;

namespace WebIssues
{

class SeparatorItemDelegate : public QAbstractItemDelegate
{
public:
    SeparatorItemDelegate( QObject* parent, QAbstractItemDelegate* delegate ) : QAbstractItemDelegate( parent ),
        m_delegate( delegate )
    {
    }

    ~SeparatorItemDelegate()
    {
    }

public: // overrides
    void paint( QPainter* painter, const QStyleOptionViewItem& option, const QModelIndex& index ) const
    {
        if ( option.state & QStyle::State_Enabled ) {
            m_delegate->paint( painter, option, index );
        } else {
            QStyleOptionViewItem noFocus = option;
            noFocus.state &= ~QStyle::State_HasFocus;

            m_delegate->paint( painter, noFocus, index );

            int y = ( option.rect.top() + option.rect.bottom() ) / 2;

            painter->setPen(  option.palette.color( QPalette::Active, QPalette::Dark ) );
            painter->drawLine( option.rect.left(), y, option.rect.right(), y );
        }
    }

    QSize sizeHint( const QStyleOptionViewItem& option, const QModelIndex& index ) const
    {
        QSize size = m_delegate->sizeHint( option, index );

        QStandardItemModel* itemModel = (QStandardItemModel*)index.model();

        if ( !itemModel->item( index.row() )->isEnabled() )
            size.setHeight( 5 );

        return size;
    }

private:
    QAbstractItemDelegate* m_delegate;
};

}

SeparatorComboBox::SeparatorComboBox( QWidget* parent ) : QComboBox( parent )
{
    QAbstractItemDelegate* delegate = view()->itemDelegate();
    view()->setItemDelegate( NULL );

    setItemDelegate( new SeparatorItemDelegate( this, delegate ) );

    view()->installEventFilter( this );
    view()->viewport()->installEventFilter( this );
}

SeparatorComboBox::~SeparatorComboBox()
{
}

void SeparatorComboBox::addSeparator()
{
    QStandardItem* item = new QStandardItem( " " );
    item->setFlags( item->flags() & ~( Qt::ItemIsEnabled | Qt::ItemIsSelectable ) );
    item->setSizeHint( QSize( 10, 5 ) );

    QStandardItemModel* itemModel = (QStandardItemModel*)model();
    itemModel->appendRow( item );
}

void SeparatorComboBox::showPopup()
{
    QComboBox::showPopup();

    QStandardItemModel* itemModel = (QStandardItemModel*)model();

    QListView* listView = (QListView*)view();
    QWidget* container = listView->parentWidget();

    QStyleOption option;
    option.initFrom( this );

    int totalHeight = style()->pixelMetric( QStyle::PM_DefaultFrameWidth, &option, this ) * 2;

    int currentHeight = container->height();

    for ( int i = 0; i < itemModel->rowCount(); i++ ) {
        totalHeight += listView->sizeHintForRow( i );
        if ( totalHeight > currentHeight + 5 )
            break;
    }

    if ( totalHeight <= currentHeight + 5 ) {
        QRect rect = container->geometry();
        rect.setHeight( totalHeight );
        container->setGeometry( rect );
    }
}

bool SeparatorComboBox::eventFilter( QObject* watched, QEvent* e )
{
    if ( ( watched == view() || watched == view()->viewport() ) && e->type() == QEvent::KeyPress ) {
        QKeyEvent* ke = (QKeyEvent*)e;
        if ( ke->key() == Qt::Key_Up || ke->key() == Qt::Key_Down ) {
            QStandardItemModel* itemModel = (QStandardItemModel*)model();

            int dir = ( ke->key() == Qt::Key_Up ) ? -1 : 1;
            int row = view()->currentIndex().row() + dir;

            while ( row >= 0 && row < itemModel->rowCount() ) {
                if ( itemModel->item( row )->isSelectable() ) {
                    view()->setCurrentIndex( itemModel->index( row, 0 ) );
                    break;
                }
                row += dir;
            }

            return true;
        }
    }

    return QComboBox::eventFilter( watched, e );
}
