/*
 * Copyright (C) 2016 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <CoreText/CoreText.h>
#include <JavaScriptCore/InitializeThreading.h>
#include <WebCore/FontCache.h>
#include <WebCore/FontCascadeDescription.h>
#include <WebCore/FontPlatformData.h>
#include <wtf/MainThread.h>
#include <wtf/RunLoop.h>
#include <wtf/Vector.h>

using namespace WebCore;

namespace TestWebKitAPI {

#if PLATFORM(MAC) && __MAC_OS_X_VERSION_MIN_REQUIRED < 101300

class FontCacheTest : public testing::Test {
public:
    virtual void SetUp()
    {
        WTF::initializeMainThread();
        JSC::initializeThreading();
        RunLoop::initializeMainRunLoop();
    }
};

static std::unique_ptr<FontPlatformData> createPlatformFont(const char* familyName, float weight, float italic, float stretch, float size)
{
    AtomicString familyNameString(familyName);
    FontCascadeDescription description;
    description.setOneFamily(familyNameString);
    description.setWeight(FontSelectionValue(weight));
    description.setItalic(FontSelectionValue(italic));
    description.setStretch(FontSelectionValue(stretch));
    description.setComputedSize(size);

    auto& fontCache = FontCache::singleton();
    return fontCache.createFontPlatformDataForTesting(description, familyNameString);
}

static bool compareFonts(const char* familyName, float weight, float italic, float stretch, CFStringRef expectedPostScriptName)
{
    float size = 80;
    auto platformFont = createPlatformFont(familyName, weight, italic, stretch, size);
    auto expectedFont = adoptCF(CTFontCreateWithName(expectedPostScriptName, size, nullptr));
    return platformFont && expectedFont && platformFont->font() && CFEqual(platformFont->font(), expectedFont.get());
}

TEST_F(FontCacheTest, FontLookupFromFamilyName)
{
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 200, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("Apple SD Gothic Neo", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("Arial", 100, 0 , 100, CFSTR("ArialMT")));
    EXPECT_TRUE(compareFonts("Arial", 100, 20, 100, CFSTR("Arial-ItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 200, 0 , 100, CFSTR("ArialMT")));
    EXPECT_TRUE(compareFonts("Arial", 200, 20, 100, CFSTR("Arial-ItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 300, 0 , 100, CFSTR("ArialMT")));
    EXPECT_TRUE(compareFonts("Arial", 300, 20, 100, CFSTR("Arial-ItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 400, 0 , 100, CFSTR("ArialMT")));
    EXPECT_TRUE(compareFonts("Arial", 400, 20, 100, CFSTR("Arial-ItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 500, 0 , 100, CFSTR("ArialMT")));
    EXPECT_TRUE(compareFonts("Arial", 500, 20, 100, CFSTR("Arial-ItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 600, 0 , 100, CFSTR("Arial-BoldMT")));
    EXPECT_TRUE(compareFonts("Arial", 600, 20, 100, CFSTR("Arial-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 700, 0 , 100, CFSTR("Arial-BoldMT")));
    EXPECT_TRUE(compareFonts("Arial", 700, 20, 100, CFSTR("Arial-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 800, 0 , 100, CFSTR("Arial-BoldMT")));
    EXPECT_TRUE(compareFonts("Arial", 800, 20, 100, CFSTR("Arial-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Arial", 900, 0 , 100, CFSTR("Arial-BoldMT")));
    EXPECT_TRUE(compareFonts("Arial", 900, 20, 100, CFSTR("Arial-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Avenir Next", 100, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("Avenir Next", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 200, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("Avenir Next", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 300, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("Avenir Next", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 400, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("Avenir Next", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 500, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("Avenir Next", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("Avenir Next", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("Avenir Next", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir Next", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("Avenir Next", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir Next", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("Avenir", 100, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 200, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 300, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 400, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 500, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Baskerville", 100, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 100, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 200, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 200, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 300, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 300, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 400, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 400, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 500, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 500, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 600, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 700, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 800, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 900, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Didot", 100, 0 , 100, CFSTR("Didot")));
    EXPECT_TRUE(compareFonts("Didot", 100, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 200, 0 , 100, CFSTR("Didot")));
    EXPECT_TRUE(compareFonts("Didot", 200, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 300, 0 , 100, CFSTR("Didot")));
    EXPECT_TRUE(compareFonts("Didot", 300, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 400, 0 , 100, CFSTR("Didot")));
    EXPECT_TRUE(compareFonts("Didot", 400, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 500, 0 , 100, CFSTR("Didot")));
    EXPECT_TRUE(compareFonts("Didot", 500, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 600, 0 , 100, CFSTR("Didot-Bold")));
    EXPECT_TRUE(compareFonts("Didot", 600, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 700, 0 , 100, CFSTR("Didot-Bold")));
    EXPECT_TRUE(compareFonts("Didot", 700, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 800, 0 , 100, CFSTR("Didot-Bold")));
    EXPECT_TRUE(compareFonts("Didot", 800, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Didot", 900, 0 , 100, CFSTR("Didot-Bold")));
    EXPECT_TRUE(compareFonts("Didot", 900, 20, 100, CFSTR("Didot-Italic")));
    EXPECT_TRUE(compareFonts("Futura", 100, 0 , 100, CFSTR("Futura-Medium")));
    EXPECT_TRUE(compareFonts("Futura", 100, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 200, 0 , 100, CFSTR("Futura-Medium")));
    EXPECT_TRUE(compareFonts("Futura", 200, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 300, 0 , 100, CFSTR("Futura-Medium")));
    EXPECT_TRUE(compareFonts("Futura", 300, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 400, 0 , 100, CFSTR("Futura-Medium")));
    EXPECT_TRUE(compareFonts("Futura", 400, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 500, 0 , 100, CFSTR("Futura-Medium")));
    EXPECT_TRUE(compareFonts("Futura", 500, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 600, 0 , 100, CFSTR("Futura-Bold")));
    EXPECT_TRUE(compareFonts("Futura", 600, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 700, 0 , 100, CFSTR("Futura-Bold")));
    EXPECT_TRUE(compareFonts("Futura", 700, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 800, 0 , 100, CFSTR("Futura-Bold")));
    EXPECT_TRUE(compareFonts("Futura", 800, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Futura", 900, 0 , 100, CFSTR("Futura-Bold")));
    EXPECT_TRUE(compareFonts("Futura", 900, 20, 100, CFSTR("Futura-MediumItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 100, 0 , 100, CFSTR("HelveticaNeue-UltraLight")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 100, 20, 100, CFSTR("HelveticaNeue-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 200, 0 , 100, CFSTR("HelveticaNeue-Thin")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 200, 20, 100, CFSTR("HelveticaNeue-ThinItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 300, 0 , 100, CFSTR("HelveticaNeue-Light")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 300, 20, 100, CFSTR("HelveticaNeue-LightItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 400, 0 , 100, CFSTR("HelveticaNeue")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 400, 20, 100, CFSTR("HelveticaNeue-Italic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 500, 0 , 100, CFSTR("HelveticaNeue-Medium")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 500, 20, 100, CFSTR("HelveticaNeue-MediumItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 600, 0 , 100, CFSTR("HelveticaNeue-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 600, 20, 100, CFSTR("HelveticaNeue-BoldItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 0 , 100, CFSTR("HelveticaNeue-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 20, 100, CFSTR("HelveticaNeue-BoldItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 0 , 100, CFSTR("HelveticaNeue-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 20, 100, CFSTR("HelveticaNeue-BoldItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 0 , 100, CFSTR("HelveticaNeue-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 20, 100, CFSTR("HelveticaNeue-BoldItalic")));
    EXPECT_TRUE(compareFonts("Helvetica", 100, 0 , 100, CFSTR("Helvetica-Light")));
    EXPECT_TRUE(compareFonts("Helvetica", 100, 20, 100, CFSTR("Helvetica-LightOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 200, 0 , 100, CFSTR("Helvetica-Light")));
    EXPECT_TRUE(compareFonts("Helvetica", 200, 20, 100, CFSTR("Helvetica-LightOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 300, 0 , 100, CFSTR("Helvetica-Light")));
    EXPECT_TRUE(compareFonts("Helvetica", 300, 20, 100, CFSTR("Helvetica-LightOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 400, 0 , 100, CFSTR("Helvetica")));
    EXPECT_TRUE(compareFonts("Helvetica", 400, 20, 100, CFSTR("Helvetica-Oblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 500, 0 , 100, CFSTR("Helvetica")));
    EXPECT_TRUE(compareFonts("Helvetica", 500, 20, 100, CFSTR("Helvetica-Oblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 600, 0 , 100, CFSTR("Helvetica-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica", 600, 20, 100, CFSTR("Helvetica-BoldOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 700, 0 , 100, CFSTR("Helvetica-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica", 700, 20, 100, CFSTR("Helvetica-BoldOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 800, 0 , 100, CFSTR("Helvetica-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica", 800, 20, 100, CFSTR("Helvetica-BoldOblique")));
    EXPECT_TRUE(compareFonts("Helvetica", 900, 0 , 100, CFSTR("Helvetica-Bold")));
    EXPECT_TRUE(compareFonts("Helvetica", 900, 20, 100, CFSTR("Helvetica-BoldOblique")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 100, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 100, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 200, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 200, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 300, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 300, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 400, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 400, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 500, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 500, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 600, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 600, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 700, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 700, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 800, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 800, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 900, 0 , 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Kaku Gothic StdN", 900, 20, 100, CFSTR("HiraKakuStdN-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 100, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 100, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 200, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 200, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 300, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 300, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 400, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 400, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 500, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 500, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 600, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 600, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 700, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 700, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 800, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 800, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 900, 0 , 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Maru Gothic ProN", 900, 20, 100, CFSTR("HiraMaruProN-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 100, 0 , 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 100, 20, 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 200, 0 , 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 200, 20, 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 300, 0 , 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 300, 20, 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 400, 0 , 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 400, 20, 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 500, 0 , 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 500, 20, 100, CFSTR("HiraMinProN-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 600, 0 , 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 600, 20, 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 700, 0 , 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 700, 20, 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 800, 0 , 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 800, 20, 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 900, 0 , 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Mincho ProN", 900, 20, 100, CFSTR("HiraMinProN-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 100, 0 , 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 100, 20, 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 200, 0 , 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 200, 20, 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 300, 0 , 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 300, 20, 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 400, 0 , 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 400, 20, 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 500, 0 , 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 500, 20, 100, CFSTR("HiraginoSansGB-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 600, 0 , 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 600, 20, 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 700, 0 , 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 700, 20, 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 800, 0 , 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 800, 20, 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 900, 0 , 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans GB", 900, 20, 100, CFSTR("HiraginoSansGB-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 100, 0 , 100, CFSTR("HiraginoSans-W0")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 100, 20, 100, CFSTR("HiraginoSans-W0")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 200, 0 , 100, CFSTR("HiraginoSans-W1")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 200, 20, 100, CFSTR("HiraginoSans-W1")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 300, 0 , 100, CFSTR("HiraginoSans-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 300, 20, 100, CFSTR("HiraginoSans-W3")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 400, 0 , 100, CFSTR("HiraginoSans-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 400, 20, 100, CFSTR("HiraginoSans-W4")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 500, 0 , 100, CFSTR("HiraginoSans-W5")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 500, 20, 100, CFSTR("HiraginoSans-W5")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 600, 0 , 100, CFSTR("HiraginoSans-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 600, 20, 100, CFSTR("HiraginoSans-W6")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 700, 0 , 100, CFSTR("HiraginoSans-W7")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 700, 20, 100, CFSTR("HiraginoSans-W7")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 800, 0 , 100, CFSTR("HiraginoSans-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 800, 20, 100, CFSTR("HiraginoSans-W8")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 900, 0 , 100, CFSTR("HiraginoSans-W9")));
    EXPECT_TRUE(compareFonts("Hiragino Sans", 900, 20, 100, CFSTR("HiraginoSans-W9")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 100, 0 , 100, CFSTR("HoeflerText-Regular")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 100, 20, 100, CFSTR("HoeflerText-Italic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 200, 0 , 100, CFSTR("HoeflerText-Regular")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 200, 20, 100, CFSTR("HoeflerText-Italic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 300, 0 , 100, CFSTR("HoeflerText-Regular")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 300, 20, 100, CFSTR("HoeflerText-Italic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 400, 0 , 100, CFSTR("HoeflerText-Regular")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 400, 20, 100, CFSTR("HoeflerText-Italic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 500, 0 , 100, CFSTR("HoeflerText-Regular")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 500, 20, 100, CFSTR("HoeflerText-Italic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 600, 0 , 100, CFSTR("HoeflerText-Black")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 600, 20, 100, CFSTR("HoeflerText-BlackItalic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 700, 0 , 100, CFSTR("HoeflerText-Black")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 700, 20, 100, CFSTR("HoeflerText-BlackItalic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 800, 0 , 100, CFSTR("HoeflerText-Black")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 800, 20, 100, CFSTR("HoeflerText-BlackItalic")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 900, 0 , 100, CFSTR("HoeflerText-Black")));
    EXPECT_TRUE(compareFonts("Hoefler Text", 900, 20, 100, CFSTR("HoeflerText-BlackItalic")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 100, 0 , 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 100, 20, 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 200, 0 , 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 200, 20, 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 300, 0 , 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 300, 20, 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 400, 0 , 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 400, 20, 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 500, 0 , 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 500, 20, 100, CFSTR("LucidaGrande")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 600, 0 , 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 600, 20, 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 700, 0 , 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 700, 20, 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 800, 0 , 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 800, 20, 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 900, 0 , 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Lucida Grande", 900, 20, 100, CFSTR("LucidaGrande-Bold")));
    EXPECT_TRUE(compareFonts("Optima", 100, 0 , 100, CFSTR("Optima-Regular")));
    EXPECT_TRUE(compareFonts("Optima", 100, 20, 100, CFSTR("Optima-Italic")));
    EXPECT_TRUE(compareFonts("Optima", 200, 0 , 100, CFSTR("Optima-Regular")));
    EXPECT_TRUE(compareFonts("Optima", 200, 20, 100, CFSTR("Optima-Italic")));
    EXPECT_TRUE(compareFonts("Optima", 300, 0 , 100, CFSTR("Optima-Regular")));
    EXPECT_TRUE(compareFonts("Optima", 300, 20, 100, CFSTR("Optima-Italic")));
    EXPECT_TRUE(compareFonts("Optima", 400, 0 , 100, CFSTR("Optima-Regular")));
    EXPECT_TRUE(compareFonts("Optima", 400, 20, 100, CFSTR("Optima-Italic")));
    EXPECT_TRUE(compareFonts("Optima", 500, 0 , 100, CFSTR("Optima-Regular")));
    EXPECT_TRUE(compareFonts("Optima", 500, 20, 100, CFSTR("Optima-Italic")));
    EXPECT_TRUE(compareFonts("Optima", 600, 0 , 100, CFSTR("Optima-Bold")));
    EXPECT_TRUE(compareFonts("Optima", 600, 20, 100, CFSTR("Optima-BoldItalic")));
    EXPECT_TRUE(compareFonts("Optima", 700, 0 , 100, CFSTR("Optima-Bold")));
    EXPECT_TRUE(compareFonts("Optima", 700, 20, 100, CFSTR("Optima-BoldItalic")));
    EXPECT_TRUE(compareFonts("Optima", 800, 0 , 100, CFSTR("Optima-ExtraBlack")));
    EXPECT_TRUE(compareFonts("Optima", 800, 20, 100, CFSTR("Optima-BoldItalic")));
    EXPECT_TRUE(compareFonts("Optima", 900, 0 , 100, CFSTR("Optima-ExtraBlack")));
    EXPECT_TRUE(compareFonts("Optima", 900, 20, 100, CFSTR("Optima-BoldItalic")));
    EXPECT_TRUE(compareFonts("PingFang HK", 100, 0 , 100, CFSTR("PingFangHK-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang HK", 100, 20, 100, CFSTR("PingFangHK-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang HK", 200, 0 , 100, CFSTR("PingFangHK-Thin")));
    EXPECT_TRUE(compareFonts("PingFang HK", 200, 20, 100, CFSTR("PingFangHK-Thin")));
    EXPECT_TRUE(compareFonts("PingFang HK", 300, 0 , 100, CFSTR("PingFangHK-Light")));
    EXPECT_TRUE(compareFonts("PingFang HK", 300, 20, 100, CFSTR("PingFangHK-Light")));
    EXPECT_TRUE(compareFonts("PingFang HK", 400, 0 , 100, CFSTR("PingFangHK-Regular")));
    EXPECT_TRUE(compareFonts("PingFang HK", 400, 20, 100, CFSTR("PingFangHK-Regular")));
    EXPECT_TRUE(compareFonts("PingFang HK", 500, 0 , 100, CFSTR("PingFangHK-Medium")));
    EXPECT_TRUE(compareFonts("PingFang HK", 500, 20, 100, CFSTR("PingFangHK-Medium")));
    EXPECT_TRUE(compareFonts("PingFang HK", 600, 0 , 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 600, 20, 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 700, 0 , 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 700, 20, 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 800, 0 , 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 800, 20, 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 900, 0 , 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang HK", 900, 20, 100, CFSTR("PingFangHK-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 100, 0 , 100, CFSTR("PingFangSC-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang SC", 100, 20, 100, CFSTR("PingFangSC-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang SC", 200, 0 , 100, CFSTR("PingFangSC-Thin")));
    EXPECT_TRUE(compareFonts("PingFang SC", 200, 20, 100, CFSTR("PingFangSC-Thin")));
    EXPECT_TRUE(compareFonts("PingFang SC", 300, 0 , 100, CFSTR("PingFangSC-Light")));
    EXPECT_TRUE(compareFonts("PingFang SC", 300, 20, 100, CFSTR("PingFangSC-Light")));
    EXPECT_TRUE(compareFonts("PingFang SC", 400, 0 , 100, CFSTR("PingFangSC-Regular")));
    EXPECT_TRUE(compareFonts("PingFang SC", 400, 20, 100, CFSTR("PingFangSC-Regular")));
    EXPECT_TRUE(compareFonts("PingFang SC", 500, 0 , 100, CFSTR("PingFangSC-Medium")));
    EXPECT_TRUE(compareFonts("PingFang SC", 500, 20, 100, CFSTR("PingFangSC-Medium")));
    EXPECT_TRUE(compareFonts("PingFang SC", 600, 0 , 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 600, 20, 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 700, 0 , 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 700, 20, 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 800, 0 , 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 800, 20, 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 900, 0 , 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang SC", 900, 20, 100, CFSTR("PingFangSC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 100, 0 , 100, CFSTR("PingFangTC-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang TC", 100, 20, 100, CFSTR("PingFangTC-Ultralight")));
    EXPECT_TRUE(compareFonts("PingFang TC", 200, 0 , 100, CFSTR("PingFangTC-Thin")));
    EXPECT_TRUE(compareFonts("PingFang TC", 200, 20, 100, CFSTR("PingFangTC-Thin")));
    EXPECT_TRUE(compareFonts("PingFang TC", 300, 0 , 100, CFSTR("PingFangTC-Light")));
    EXPECT_TRUE(compareFonts("PingFang TC", 300, 20, 100, CFSTR("PingFangTC-Light")));
    EXPECT_TRUE(compareFonts("PingFang TC", 400, 0 , 100, CFSTR("PingFangTC-Regular")));
    EXPECT_TRUE(compareFonts("PingFang TC", 400, 20, 100, CFSTR("PingFangTC-Regular")));
    EXPECT_TRUE(compareFonts("PingFang TC", 500, 0 , 100, CFSTR("PingFangTC-Medium")));
    EXPECT_TRUE(compareFonts("PingFang TC", 500, 20, 100, CFSTR("PingFangTC-Medium")));
    EXPECT_TRUE(compareFonts("PingFang TC", 600, 0 , 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 600, 20, 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 700, 0 , 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 700, 20, 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 800, 0 , 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 800, 20, 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 900, 0 , 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("PingFang TC", 900, 20, 100, CFSTR("PingFangTC-Semibold")));
    EXPECT_TRUE(compareFonts("Times New Roman", 100, 0 , 100, CFSTR("TimesNewRomanPSMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 100, 20, 100, CFSTR("TimesNewRomanPS-ItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 200, 0 , 100, CFSTR("TimesNewRomanPSMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 200, 20, 100, CFSTR("TimesNewRomanPS-ItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 300, 0 , 100, CFSTR("TimesNewRomanPSMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 300, 20, 100, CFSTR("TimesNewRomanPS-ItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 400, 0 , 100, CFSTR("TimesNewRomanPSMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 400, 20, 100, CFSTR("TimesNewRomanPS-ItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 500, 0 , 100, CFSTR("TimesNewRomanPSMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 500, 20, 100, CFSTR("TimesNewRomanPS-ItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 600, 0 , 100, CFSTR("TimesNewRomanPS-BoldMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 600, 20, 100, CFSTR("TimesNewRomanPS-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 700, 0 , 100, CFSTR("TimesNewRomanPS-BoldMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 700, 20, 100, CFSTR("TimesNewRomanPS-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 800, 0 , 100, CFSTR("TimesNewRomanPS-BoldMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 800, 20, 100, CFSTR("TimesNewRomanPS-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 900, 0 , 100, CFSTR("TimesNewRomanPS-BoldMT")));
    EXPECT_TRUE(compareFonts("Times New Roman", 900, 20, 100, CFSTR("TimesNewRomanPS-BoldItalicMT")));
    EXPECT_TRUE(compareFonts("Times", 100, 0 , 100, CFSTR("Times-Roman")));
    EXPECT_TRUE(compareFonts("Times", 100, 20, 100, CFSTR("Times-Italic")));
    EXPECT_TRUE(compareFonts("Times", 200, 0 , 100, CFSTR("Times-Roman")));
    EXPECT_TRUE(compareFonts("Times", 200, 20, 100, CFSTR("Times-Italic")));
    EXPECT_TRUE(compareFonts("Times", 300, 0 , 100, CFSTR("Times-Roman")));
    EXPECT_TRUE(compareFonts("Times", 300, 20, 100, CFSTR("Times-Italic")));
    EXPECT_TRUE(compareFonts("Times", 400, 0 , 100, CFSTR("Times-Roman")));
    EXPECT_TRUE(compareFonts("Times", 400, 20, 100, CFSTR("Times-Italic")));
    EXPECT_TRUE(compareFonts("Times", 500, 0 , 100, CFSTR("Times-Roman")));
    EXPECT_TRUE(compareFonts("Times", 500, 20, 100, CFSTR("Times-Italic")));
    EXPECT_TRUE(compareFonts("Times", 600, 0 , 100, CFSTR("Times-Bold")));
    EXPECT_TRUE(compareFonts("Times", 600, 20, 100, CFSTR("Times-BoldItalic")));
    EXPECT_TRUE(compareFonts("Times", 700, 0 , 100, CFSTR("Times-Bold")));
    EXPECT_TRUE(compareFonts("Times", 700, 20, 100, CFSTR("Times-BoldItalic")));
    EXPECT_TRUE(compareFonts("Times", 800, 0 , 100, CFSTR("Times-Bold")));
    EXPECT_TRUE(compareFonts("Times", 800, 20, 100, CFSTR("Times-BoldItalic")));
    EXPECT_TRUE(compareFonts("Times", 900, 0 , 100, CFSTR("Times-Bold")));
    EXPECT_TRUE(compareFonts("Times", 900, 20, 100, CFSTR("Times-BoldItalic")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 0 , 87.5, CFSTR("HelveticaNeue-CondensedBold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 0 , 87.5, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 0 , 87.5, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 0 , 75, CFSTR("HelveticaNeue-CondensedBold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 0 , 75, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 0 , 75, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 20, 87.5, CFSTR("HelveticaNeue-CondensedBold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 20, 87.5, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 20, 87.5, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 700, 20, 75, CFSTR("HelveticaNeue-CondensedBold")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 800, 20, 75, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Helvetica Neue", 900, 20, 75, CFSTR("HelveticaNeue-CondensedBlack")));
    EXPECT_TRUE(compareFonts("Futura", 500, 0 , 87.5, CFSTR("Futura-CondensedMedium")));
    EXPECT_TRUE(compareFonts("Futura", 500, 20, 87.5, CFSTR("Futura-CondensedMedium")));
    EXPECT_TRUE(compareFonts("Futura", 600, 0 , 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 600, 20, 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 700, 0 , 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 700, 20, 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 800, 0 , 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 800, 20, 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 900, 0 , 87.5, CFSTR("Futura-CondensedExtraBold")));
    EXPECT_TRUE(compareFonts("Futura", 900, 20, 87.5, CFSTR("Futura-CondensedExtraBold")));
}

TEST_F(FontCacheTest, FontLookupFromPostScriptName)
{
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 600, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 800, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Bold", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 100, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 200, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 300, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 400, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 500, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 600, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 700, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 900, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-ExtraBold", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 600, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 800, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Heavy", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Light", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Medium", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Regular", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 100, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 200, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 300, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 400, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 500, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 700, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 800, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 900, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-SemiBold", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 100, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 200, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 300, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 400, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 500, 0 , 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-Thin", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 100, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 100, 20, 100, CFSTR("AppleSDGothicNeo-Thin")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 200, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 200, 20, 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 300, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 300, 20, 100, CFSTR("AppleSDGothicNeo-Light")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 400, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 400, 20, 100, CFSTR("AppleSDGothicNeo-Regular")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 500, 0 , 100, CFSTR("AppleSDGothicNeo-UltraLight")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 500, 20, 100, CFSTR("AppleSDGothicNeo-Medium")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 600, 0 , 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 600, 20, 100, CFSTR("AppleSDGothicNeo-SemiBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 700, 0 , 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 700, 20, 100, CFSTR("AppleSDGothicNeo-Bold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 800, 0 , 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 800, 20, 100, CFSTR("AppleSDGothicNeo-ExtraBold")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 900, 0 , 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("AppleSDGothicNeo-UltraLight", 900, 20, 100, CFSTR("AppleSDGothicNeo-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 100, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 200, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 300, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 400, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 500, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 900, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Black", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 100, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 100, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 200, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 200, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 300, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 300, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 400, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 400, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 500, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 500, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 600, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 700, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 800, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 900, 0 , 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BlackOblique", 900, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 100, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 200, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 300, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 400, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 500, 0 , 100, CFSTR("Avenir-Book")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Book", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 100, 0 , 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 100, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 200, 0 , 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 200, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 300, 0 , 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 300, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 400, 0 , 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 500, 0 , 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 500, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-BookOblique", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 100, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 200, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 300, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 400, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 500, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 600, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 700, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 800, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Heavy", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 100, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 100, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 200, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 200, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 300, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 300, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 400, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 400, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 500, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 500, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 600, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 600, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 700, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 700, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 800, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 800, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 900, 0 , 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-HeavyOblique", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 100, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 200, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 300, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 400, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 500, 0 , 100, CFSTR("Avenir-Light")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Light", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 100, 0 , 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 200, 0 , 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 300, 0 , 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 400, 0 , 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 400, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 500, 0 , 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 500, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-LightOblique", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 100, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 200, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 300, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 400, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 500, 0 , 100, CFSTR("Avenir-Medium")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Medium", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 100, 0 , 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 100, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 200, 0 , 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 200, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 300, 0 , 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 300, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 400, 0 , 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 400, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 500, 0 , 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-MediumOblique", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 100, 0 , 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 100, 20, 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 200, 0 , 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 200, 20, 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 300, 0 , 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 300, 20, 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 400, 0 , 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 400, 20, 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 500, 0 , 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 500, 20, 100, CFSTR("Avenir-Oblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Oblique", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 100, 0 , 100, CFSTR("Avenir-Roman")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 100, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 200, 0 , 100, CFSTR("Avenir-Roman")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 200, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 300, 0 , 100, CFSTR("Avenir-Roman")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 300, 20, 100, CFSTR("Avenir-LightOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 400, 0 , 100, CFSTR("Avenir-Roman")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 400, 20, 100, CFSTR("Avenir-BookOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 500, 0 , 100, CFSTR("Avenir-Roman")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 500, 20, 100, CFSTR("Avenir-MediumOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 600, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 600, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 700, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 700, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 800, 0 , 100, CFSTR("Avenir-Black")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 800, 20, 100, CFSTR("Avenir-BlackOblique")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 900, 0 , 100, CFSTR("Avenir-Heavy")));
    EXPECT_TRUE(compareFonts("Avenir-Roman", 900, 20, 100, CFSTR("Avenir-HeavyOblique")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 100, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 200, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 300, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 400, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 500, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 600, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 800, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 900, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Bold", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 100, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 100, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 200, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 200, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 300, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 300, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 400, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 400, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 500, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 500, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 600, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 600, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 700, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 800, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 800, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 900, 0 , 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-BoldItalic", 900, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 100, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 200, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 300, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 400, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 500, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 700, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 800, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 900, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBold", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 100, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 100, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 200, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 200, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 300, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 300, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 400, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 400, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 500, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 500, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 600, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 700, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 700, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 800, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 800, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 900, 0 , 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-DemiBoldItalic", 900, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 100, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 200, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 300, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 400, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 500, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 600, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 700, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Heavy", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 100, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 100, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 200, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 200, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 300, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 300, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 400, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 400, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 500, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 500, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 600, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 600, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 700, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 700, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 800, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 900, 0 , 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-HeavyItalic", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 100, 0 , 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 100, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 200, 0 , 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 200, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 300, 0 , 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 300, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 400, 0 , 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 500, 0 , 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 500, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Italic", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 100, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 200, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 300, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 400, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 500, 0 , 100, CFSTR("AvenirNext-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Medium", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 100, 0 , 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 100, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 200, 0 , 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 200, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 300, 0 , 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 300, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 400, 0 , 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 400, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 500, 0 , 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-MediumItalic", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 100, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 200, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 300, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 400, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 500, 0 , 100, CFSTR("AvenirNext-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-Regular", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 100, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 200, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 300, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 400, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 400, 20, 100, CFSTR("AvenirNext-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 500, 0 , 100, CFSTR("AvenirNext-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 500, 20, 100, CFSTR("AvenirNext-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLight", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 100, 0 , 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 100, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 200, 0 , 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 200, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 300, 0 , 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 300, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 400, 0 , 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 400, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 500, 0 , 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 500, 20, 100, CFSTR("AvenirNext-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 600, 0 , 100, CFSTR("AvenirNext-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 600, 20, 100, CFSTR("AvenirNext-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 700, 0 , 100, CFSTR("AvenirNext-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 700, 20, 100, CFSTR("AvenirNext-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 800, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 800, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 900, 0 , 100, CFSTR("AvenirNext-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNext-UltraLightItalic", 900, 20, 100, CFSTR("AvenirNext-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 100, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 200, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 300, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 400, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 500, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 600, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 800, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 900, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Bold", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 100, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 100, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 200, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 200, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 300, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 300, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 400, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 400, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 500, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 500, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 600, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 600, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 700, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 800, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 800, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 900, 0 , 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-BoldItalic", 900, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 100, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 200, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 300, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 400, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 500, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 700, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 800, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 900, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBold", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 100, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 100, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 200, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 200, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 300, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 300, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 400, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 400, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 500, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 500, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 700, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 700, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 800, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 800, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 900, 0 , 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-DemiBoldItalic", 900, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 100, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 200, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 300, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 400, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 500, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 600, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 700, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Heavy", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 100, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 100, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 200, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 200, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 300, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 300, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 400, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 400, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 500, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 500, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 600, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 600, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 700, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 700, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 800, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 900, 0 , 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-HeavyItalic", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 100, 0 , 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 100, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 200, 0 , 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 200, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 300, 0 , 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 300, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 400, 0 , 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 500, 0 , 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 500, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Italic", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 100, 0 , 100, CFSTR("AvenirNextCondensed-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 200, 0 , 100, CFSTR("AvenirNextCondensed-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 300, 0 , 100, CFSTR("AvenirNextCondensed-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 400, 0 , 100, CFSTR("AvenirNextCondensed-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 500, 0 , 100, CFSTR("AvenirNextCondensed-Medium")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Medium", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 100, 0 , 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 100, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 200, 0 , 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 200, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 300, 0 , 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 300, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 400, 0 , 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 400, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 500, 0 , 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-MediumItalic", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 100, 0 , 100, CFSTR("AvenirNextCondensed-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 200, 0 , 100, CFSTR("AvenirNextCondensed-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 300, 0 , 100, CFSTR("AvenirNextCondensed-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 400, 0 , 100, CFSTR("AvenirNextCondensed-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 500, 0 , 100, CFSTR("AvenirNextCondensed-Regular")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-Regular", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 100, 0 , 100, CFSTR("AvenirNextCondensed-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 200, 0 , 100, CFSTR("AvenirNextCondensed-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 300, 0 , 100, CFSTR("AvenirNextCondensed-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 400, 0 , 100, CFSTR("AvenirNextCondensed-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 400, 20, 100, CFSTR("AvenirNextCondensed-Italic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 500, 0 , 100, CFSTR("AvenirNextCondensed-UltraLight")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 500, 20, 100, CFSTR("AvenirNextCondensed-MediumItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLight", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 100, 0 , 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 100, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 200, 0 , 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 200, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 300, 0 , 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 300, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 400, 0 , 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 400, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 500, 0 , 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 500, 20, 100, CFSTR("AvenirNextCondensed-UltraLightItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 600, 0 , 100, CFSTR("AvenirNextCondensed-DemiBold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 600, 20, 100, CFSTR("AvenirNextCondensed-DemiBoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 700, 0 , 100, CFSTR("AvenirNextCondensed-Bold")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 700, 20, 100, CFSTR("AvenirNextCondensed-BoldItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 800, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 800, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 900, 0 , 100, CFSTR("AvenirNextCondensed-Heavy")));
    EXPECT_TRUE(compareFonts("AvenirNextCondensed-UltraLightItalic", 900, 20, 100, CFSTR("AvenirNextCondensed-HeavyItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 100, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 100, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 200, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 200, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 300, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 300, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 400, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 400, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 500, 0 , 100, CFSTR("Baskerville")));
    EXPECT_TRUE(compareFonts("Baskerville", 500, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville", 600, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 700, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 800, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville", 900, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 100, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 100, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 200, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 200, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 300, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 300, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 400, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 400, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 500, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 500, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 600, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 700, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 800, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 900, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Bold", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 100, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 100, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 200, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 200, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 300, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 300, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 400, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 400, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 500, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 500, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 600, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 600, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 700, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 800, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 900, 0 , 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-BoldItalic", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 100, 0 , 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 100, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 200, 0 , 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 200, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 300, 0 , 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 300, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 400, 0 , 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 400, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 500, 0 , 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 500, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 600, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 700, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 800, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 900, 0 , 100, CFSTR("Baskerville-Bold")));
    EXPECT_TRUE(compareFonts("Baskerville-Italic", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 100, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 100, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 200, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 200, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 300, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 300, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 400, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 400, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 500, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 500, 20, 100, CFSTR("Baskerville-Italic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 600, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 700, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 700, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 800, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 800, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 900, 0 , 100, CFSTR("Baskerville-SemiBold")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBold", 900, 20, 100, CFSTR("Baskerville-BoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 100, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 100, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 200, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 200, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 300, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 300, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 400, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 400, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 500, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 500, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 600, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 600, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 700, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 700, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 800, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 800, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 900, 0 , 100, CFSTR("Baskerville-SemiBoldItalic")));
    EXPECT_TRUE(compareFonts("Baskerville-SemiBoldItalic", 900, 20, 100, CFSTR("Baskerville-SemiBoldItalic")));
}

#endif

}
