/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

#ifndef _WF_PROTOCOL_H
#define _WF_PROTOCOL_H

#include <iostream>
#include <list>
#include <string>

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/types.h> /* <netinet/in.h> needs this */
#endif
#include <netinet/in.h> /* for IPPROTO_* */
#include <inttypes.h>
#include "wftcpflags.h"

/**
 * protocol
 */
class wf_protocol {
 public:
  wf_protocol();
  wf_protocol(int16_t protocol);
  wf_protocol(const string& name);
  virtual wf_protocol* VClone() const;
  virtual ~wf_protocol() {}
  // wf_protocol(const wf_protocol&) {} implicit copy constructor
  
  int16_t get() const;
  bool set(int16_t protocol);
  bool set(const string& name);

  string tostr() const;
  ostream& print(ostream& os) const;

  friend ostream& operator<<(ostream& os, const wf_protocol* protocol) {
    return (protocol != NULL) ? protocol->print(os) : os << "(undefined)";
  }
  friend ostream& operator<<(ostream& os, const wf_protocol& protocol) {
    return protocol.print(os);
  }
  friend bool operator==(const wf_protocol& protocol1,
			 const wf_protocol& protocol2);

 protected:
  int16_t _num;
};

class wf_protocol_tcp : public wf_protocol {
 public:
  wf_protocol_tcp();
  virtual wf_protocol_tcp* VClone() const;
  virtual ~wf_protocol_tcp() {}
  // wf_protocol_tcp(const wf_protocol_tcp&) {} implicit copy constructor

  friend ostream& operator<<(ostream& os, const wf_protocol_tcp* protocol) {
    return (protocol != NULL) ? protocol->print(os) : os << "(undefined)";
  }
  friend ostream& operator<<(ostream& os, const wf_protocol_tcp& protocol);

  friend bool operator==(const wf_protocol_tcp& protocol1,
			 const wf_protocol_tcp& protocol2);

 public:
  wf_tcpflags flags, flagsmask;
  bool neg;
};

class wf_protocol_icmp : public wf_protocol {
 public:
  wf_protocol_icmp();
  wf_protocol_icmp(uint8_t type);
  wf_protocol_icmp(uint8_t type, int8_t code);
  wf_protocol_icmp(const string& name);
  virtual wf_protocol_icmp* VClone() const;
  virtual ~wf_protocol_icmp() {}
  // wf_protocol_icmp(const wf_protocol_icmp&) {} implicit copy constructor

  bool set(const string& name);
  bool set(uint8_t type);
  bool set(uint8_t type, int8_t code);

  int8_t type() const { return _type; };
  int8_t code() const { return _code; };

  string type_tostr() const;

  friend ostream& operator<<(ostream& os, const wf_protocol_icmp* protocol) {
    return (protocol != NULL) ? protocol->print(os) : os << "(undefined)";
  }
  friend ostream& operator<<(ostream& os, const wf_protocol_icmp& protocol) {
    return protocol.print(os);
  }

  friend bool operator==(const wf_protocol_icmp& protocol1,
			 const wf_protocol_icmp& protocol2);

 private:
  int8_t _type; /* -1 if none */
  int8_t _code; /* -1 if none */
 public:
  bool neg;
};

inline wf_protocol*
wf_protocol::VClone() const {
  return new wf_protocol(*this);
}

inline wf_protocol_tcp*
wf_protocol_tcp::VClone() const {
  return new wf_protocol_tcp(*this);
}

inline wf_protocol_icmp*
wf_protocol_icmp::VClone() const {
  return new wf_protocol_icmp(*this);
}

ostream& wf_protocol_icmp_types_print(ostream& os);

string wf_protocol_icmp_type_tostr(uint8_t type);
string wf_protocol_icmp_type_tostr(uint8_t type, int8_t code);
bool wf_protocol_icmp_totype(const string& name, int* type, int* code);
bool wf_protocol_icmp_type_hascode(uint8_t type);

#endif
