/*
  libwftk - a C++ widget library based on SDL (Simple Direct Layer)
  Copyright (C) 2003 Ron Steinke

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.
  
  You should have received a copy of the GNU Library General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef WINDOWED_ROOT_H
#define WINDOWED_ROOT_H

#include "rootwindow.h"
#include "surface.h"
#include "screensurface.h"

#include <sigc++/object_slot.h>

namespace wftk {

/// This is a 'shaped' root window. The target area for
/// the smoke is not contained in the window's shape,
/// so it never gets drawn to except by our custom drawing
/// function. Of course, it will never get overlayed by
/// dialogs either, so this is kind of a hack.
class WindowedRoot : public RootWindow
{
 public:
  // reference window_ so it never gets delete called on it
  WindowedRoot(int w, int h) : RootWindow(w, h), window_(*this) {window_.ref();}

  /// the target surface
  Surface& target() {return window_.target();}
  /// a widget to control the packing of the target surface
  ScreenArea* widget() {return &window_;}

  /// update the window
  void update() {screen()->update(window_.rootRect());}

  /// draw in the RootWindow background
  void clear()
  {
    screen()->mutex.grab();
    draw(*screen(), Point(0, 0), window_.rootRect());
    screen()->mutex.release();
  }

 private:
  void setWindow(const Rect& r)
  {
    setShape(Region(getRect()) - r, Region());
    clear();
  }

  friend class Window : public ScreenArea
  {
   public:
    explicit Window(WindowedRoot& root) : root_(root) {}

    Surface& target() {return target_;}

    Rect rootRect() {return root_.getRect().intersect(screenRect());}

   protected:
    ///
    virtual void handleResize(Uint16 w, Uint16 h)
    {
      ScreenArea::handleResize(w, h);

      Rect global = rootRect();

      root_.setWindow(global);

      if(global.isEmpty())
        target_ = Surface();
      else
        target_.setAsChild(*root_.screen(), global);
    }
    ///
    virtual void setPackingInfo()
    {
      packing_info_.x.filler = packing_info_.y.filler = PackingInfo::Expander::FILL;
    }

   private:
    ///
    WindowedRoot& root_;
    ///
    Surface target_;
  };

  Window window_;
};

} // namespace

#endif // WINDOWED_ROOT_H
