/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _SCREEN_AREA_H
#define _SCREEN_AREA_H

#include <list>
#include <string>
#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <wftk/rect.h>
#include <wftk/region.h>
#include <wftk/refobj.h>
#include <wftk/mouse.h>

namespace wftk {

class Surface;

/** ScreenArea Baseclass.
 *
 * For information on writing your own widgets, set README.widgets in the source
 **/
class ScreenArea : virtual public SigC::Object, public RefCountObj
{
 public:
  /// Contstructor
  ScreenArea();
  /// Destructor
  virtual ~ScreenArea();

  /// Emitted in destructor
  SigC::Signal0<void> deleted;
  /// emitted when ScreenArea is resized
  SigC::Signal2<void,Uint16,Uint16> resized;

  ///show this widget and its children.
  /**
   * Returns true for a successful show(), false if the widget
   * was already shown.
   **/
  bool show();
  ///hide this widget and its children.
  /**
   * Returns true for a successful hide(), false if the widget
   * was already hidden.
   **/
  bool hide();

  /// raise this widget to the front
  void raise();
  /// lower this widget to the back
  void lower();

  /// Resize this widget
  void resize(const Rect&);
  /// Resize this widget
  void resize(Uint16 w, Uint16 h) {resize(Rect(rect_.x, rect_.y, w, h));}

  /// Returns true if this widget is hidden
  bool isHidden() const {return hidden_;}

  /// Set the parent of the widget
  /**
   * Changing the parent from null to non-null increases the widget's
   * refcount, and changing from non-null to null decreases it.
   * Effectively, the parent holds a reference to the child.
   **/
  void setParent(ScreenArea*);
  /// Get this widget's parent
  ScreenArea* parent() {return parent_;}
  /// Get this widget's parent
  const ScreenArea* parent() const {return parent_;}

  /// true if this widget contains the argument
  bool contains(const ScreenArea&) const;

  /// Returns the (ultimate) child widget that contains the point
  /**
   * If no child widget contains the point, either the
   * current widget (if it contains the point) or 0 (if not)
   * is returned.
   **/
  ScreenArea* getContainer(const Point&);

  /// packing information to pass to containers
  struct PackingInfo {
    /// The per-axis packing info
    struct Expander {
      /// the object's prefered size
      Uint16 pref;
      /// the object's minimum size
      Uint16 min;
      /// true if an object can expand past its prefered size
      bool expand;
      /// the 'filler weight' of the object
      /**
       * Excess space in a container is assigned to children based on
       * their 'filler weight', with the children with the highest weight
       * splitting the space evenly. If the highest weight is zero,
       * the space is split among the children weighted by their
       * prefered sizes. If 'expand' is false, this field is ignored.
       * Containers inherit the highest filler weight from among
       * their children, reduced by five.
       **/
      unsigned char filler;

      /// default values
      Expander() : pref(0), expand(true), filler(0) {
        // win32 uses min() as a preprocessor macro, have to init it here
	min = 0;
      }

      // container functions

      /// expand a container dimension by adding an object on the end
      void extend(const Expander&);

      /// make sure a object fits in this dimension's packing info
      void contain(const Expander&);

      enum {
        /// nice default value for filler (set to 100)
        FILL = 100,
        /// amount containers should subtract from a child's filler value (set to 5)
        SUBTRACT = 5
      };
    } x, y;
    /// For use by containers
    class Weights {
     public:
      Weights() : expand_(0), shrink_(0), high_fill_(0), expand_frac_(0) {}

      /// add a child widget's expansion capabilities to the accumulated weights
      void extend(const Expander&);

      /// set the amount to expand in padding(), based on size difference in parent
      void setExpand(Uint16 pref_size, Uint16 real_size);
      /// get the amount of padding for a child widget
      double padding(const Expander&) const;

     private:
      /// holds the number of children at maximum filler weight, or the sum of the prefered sizes if there is no filler
      Uint16 expand_;
      /// holds the sum of (pref - min) size for all children
      Uint16 shrink_;
      /// holds the largest value of 'filler' of any child
      unsigned char high_fill_;
      /// how much padding() to give
      double expand_frac_;
    };
  };
  /// returns this widget's packing information
  const PackingInfo& getPackingInfo() const {return packing_info_;}

  /// return geometry
  const Rect& getRect() const {return rect_;}
  /// return geometry for shaped widgets
  const Region& getShape() const {return shape_;}
  /// return opaque subarea for partially transparent widgets
  const Region& getCoverage() const {return covered_;}

  /// The absolute position of the widget on the screen
  Rect screenRect() const
	{return parent_ ? parent_->globalCoord(getRect()) : getRect();}

  /// the width of the widget
  Uint16 width() const {return rect_.w;}
  /// the height of the widget
  Uint16 height() const {return rect_.h;}

  /// convert widget coordinates to root window coordinates
  Rect globalCoord (const Rect& local) const;
  /// convert root window coordinates to widget coordinates
  Rect localCoord (const Rect& global) const;

  /// true if the mouse is in the widget
  bool hasMouse() const;

  // Event handlers for this widget's behavior,
  // non-widget-specific event handlers should connect to
  // the signals in Mouse or Focus, respectively

  /// mouse motion event handler
  virtual bool mouseEvent(const Point& pos, const Point& rel, Mouse::Button mask)
	{return false;}
  /// mouse button event handler
  virtual bool buttonEvent(Mouse::Button, bool pressed, const Point&)
	{return false;}
  /// mouse gain event handler
  virtual void gainedMouse() {}
  /// mouse loss event handler
  virtual void lostMouse() {}

  /// std::type_info::name(), for debugging
  std::string name() const;
 
 protected:

  /// set the parent of all children to 0
  /**
   * Sometimes (e.g., RootWindow) it is necessary to
   * remove all the children of a derived class instance
   * before doing other things in the derived class
   * destructor. We therefore export this functionality.
   **/
  void removeChildren();

  /// update this widget's packing in its parent
  void packingUpdate();

  /// Set the shape of the widget
  /**
   * The 'shape' argument specifies which pixels are part of the
   * widget, and is clipped to the widget's rectangle. The 'coverage'
   * argument specifies the amount of area the widget covers opaquely,
   * and is clipped to the shape. By default, shape is set equal to
   * the widget's rectangle for both Widget and ScreenArea. The default
   * for coverage is the widget's rectangle for Widget and an empty
   * region for ScreenArea (since it doesn't draw a background).
   * For derived non-rectangular or non-opaque widgets, this function
   * will generally need to be called in both the widget's constructor
   * and in the virtual function handleResize().
   **/
  void setShape(const Region& shape, const Region& coverage);
  /// Set the area the widget covers opaquely
  void setCoverage(const Region& coverage) {setShape(shape_, coverage);}

  /// get the exposed portion of the widget
  const Region& getExposed() const {return exposed_;}

  /// check if part of the widget needs to be redrawn
  bool dirty() const {return !hidden_ && !dirtyFull_.empty();}

  /// Invalidate part of the widget to force a redraw
  void invalidate(const Region&);
  /// Invalidate the whole widget to force a redraw
  void invalidate() {invalidate(Rect(0, 0, rect_.w, rect_.h));}

  /// just blit to the parent's surface, using the dirty region
  /**
   * This should only be called as part of the drawing engine.
   * It's protected instead of private so RootWindow can call it.
   **/
  void blit(Surface& target, const Point& offset);

  /// The widget's drawing implementation.
  /**
   * The area drawn to must lie within the given region.
   * This is protected so derived classes can call their
   * parents' draw() in their own implementations.
   **/
  virtual void draw(Surface&, const Point& offset, const Region&) {}

  /// The widget's drawing implementation, for things drawn after the child widgets.
  /**
   * The area drawn to must lie within the given region.
   * This is protected so derived classes can call their
   * parents' drawAfter() in their own implementations.
   **/
  virtual void drawAfter(Surface&, const Point& offset, const Region&) {}

  /// Calculate new shape for shaped widgets, call to setShape() after a resize
  virtual void handleResize(Uint16 w, Uint16 h)
	{setShape(Rect(0, 0, w, h), Region());}

  /// recalculate the object's prefered size
  virtual void setPackingInfo()
	{packing_info_.x = packing_info_.y = PackingInfo::Expander();}

  /// chains PackingInfo updates (due to resize, etc.) to the containing widget
  virtual void packingUpdateParent() {if(parent_) parent_->packingUpdate();}

  /// information on the widget's prefered size
  PackingInfo packing_info_;

 private:
  // Unimplemented functions
  ScreenArea(const ScreenArea&);
  ScreenArea& operator=(const ScreenArea&);

  /**
   * Tell the widget to redraw the area in the given region in
   * the next update. The widget updates the ExposeRegion
   * to reflect the area that it will be drawing. Partially
   * transparent regions should be considered still exposed,
   * since the lower layers will still need to be drawn.
   **/
  void expose(Region& full, Region& remaining);

  ///
  void invalidateRecurse(Region&);
  ///
  void doExpose(const Region&);

  /// the widget's bounding rectangle, relative to its parent
  Rect rect_;
  /// The amount of the widget that is actually visible.
  /**
   * Used when the widget is updated to reduce the amount
   * of the root window that gets redrawn.
   * (exposed_ - shape_).empty() should always be true.
   **/
  Region exposed_;
  /// The amount of the widget that needs to be redrawn.
  /**
   * (dirty_ - dirtyFull_).empty() should always be true
   **/
  Region dirty_;
  /// The combined ammount of the widget and its children that needs to be redrawn.
  /**
   * (dirtyFull_ - exposed_).empty() should always be true
   **/
  Region dirtyFull_;
  /// The amount of rect_ that is considered 'part' of the widget
  /**
   * (shape_ - Rect(0, 0, rect_.w, rect_.h)).empty() should always be true
   **/
  Region shape_;
  /// The amount of rect_ that the widget covers opaquely
  /**
   * (covered_ - shape_).empty() should always be true
   **/
  Region covered_;

  /**the parent. 
     if NULL this widget is  toplevel (root window)
  */
  ScreenArea *parent_;

  ///
  bool hidden_;
 
  typedef std::list<ScreenArea*> Children;
  ///child list.
  Children childs_;

  ///
  Children::iterator find(ScreenArea*);
  /// Get the area obscured by this widget, including the area of all children
  Region getFullObscure();

};

} // namespace wftk

#endif // _SCREEN_AREA_H
