/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "video.h"

#ifdef HAVE_CONFIG_H
#include "../config.h"
#endif

#include "mixer.h"
#include "rootwindow.h"
#include "screensurface.h"
#include "debug.h"

#ifdef HAVE_MIXER
#include <SDL_mixer.h>
#endif

#ifdef HAVE_SMPEG
#include <smpeg/smpeg.h>
#endif

namespace wftk {

Video::SurfaceMap Video::map_;
Mutex Video::map_mutex_;

#ifdef HAVE_SMPEG

Video::Video(const Point& screen_loc, ScreenSurface* screen) :
  use_mask_(false),
  loc_(screen_loc),
  screen_(screen),
  mpeg(NULL)
{
  Mixer::instance(); // insure init audio 
  if(!screen_) {
    assert(RootWindow::instance());
    screen_ = RootWindow::instance()->screen();
    assert(screen_); // insure we have a root window
  }
}

Video::~Video()
{
  stop();

  if(mpeg)
    SMPEG_delete((SMPEG*)mpeg);

  if(sdlSurface_) {
    mutex_.grab();
    map_mutex_.grab();
    map_.erase(sdlSurface_);
    map_mutex_.release();
    mutex_.release();
  }
}

void Video::setMask(const Region& r)
{
  mutex_.grab();
  mask_ = r;
  mask_.offset(loc_); // mask_ relative to screen surface
  use_mask_ = true;
  mutex_.release();

  Debug out(Debug::INVALIDATE);
  if(out) {
    Rect clip = r.getClipbox();
    out << "Setting video mask, mask lies inside " << clip
      << ", removed region lies inside " << (clip - r).getClipbox() << Debug::endl;
  }
}

void Video::clearMask()
{
  mutex_.grab();
  use_mask_ = false;
  mutex_.release();
}

void Video::move(const Point& p)
{
  mutex_.grab();
  loc_ = p;
  if(use_mask_)
    mask_.offset(p);
  mutex_.release();
}

bool
Video::load(const std::string& filename, double scale)
{
  Debug out(Debug::SOUND);

  out << "Starting Video::load()" << Debug::endl;

  if(mpeg)
    SMPEG_delete((SMPEG*)mpeg);

  out << "Trying to load Video using SMPEG" << Debug::endl;

  SMPEG_Info info;
  mpeg = SMPEG_new(filename.c_str(), &info, 0);   
  if ( SMPEG_error((SMPEG*)mpeg) ) 
    {
      out <<"Video: " << SMPEG_error((SMPEG*)mpeg) << Debug::endl;
      SMPEG_delete((SMPEG*)mpeg);
      mpeg = NULL;
      return false;
    }

  if(scale <= 0) // the user probably wants the default size
    scale = 1;

  Uint16 width = (Uint16) (info.width * scale);
  Uint16 height = (Uint16) (info.height * scale);
  if(width == 0)
    width = 1;
  if(height == 0)
    height = 1;

  SMPEG_enableaudio((SMPEG*)mpeg, 0);
  SMPEG_enablevideo((SMPEG*)mpeg, 1);

#ifdef HAVE_MIXER
  SDL_AudioSpec audiofmt;
  Uint16 format;
  int freq, channels;

  /* Tell SMPEG what the audio format is */
  Mix_QuerySpec(&freq, &format, &channels);
  audiofmt.format = format;
  audiofmt.freq = freq;
  audiofmt.channels = channels;
  SMPEG_actualSpec((SMPEG*)mpeg, &audiofmt);
#endif

  mutex_.grab();

  map_mutex_.grab();

  if(sdlSurface_)
    map_.erase(sdlSurface_);

  Debug::channel(Debug::PACKING) << "Video surface size set to (" << width
	<< "," << height << ")" << Debug::endl;

  Surface::setSurface(width, height, screen_->pixelformat());
  assert(sdlSurface_);
  map_[sdlSurface_] = this;

  map_mutex_.release();

  SMPEG_move((SMPEG*) mpeg, 0, 0);
  SMPEG_setdisplay((SMPEG*) mpeg, sdlSurface_, mutex_, screenUpdate);

  mutex_.release();

  out << "Loaded video using SMPEG" << Debug::endl;

  out << "Finished Video::load()" << Debug::endl;

  return true;
}

int
Video::setVolume(int vol)
{
  if(mpeg)
    SMPEG_setvolume((SMPEG*)mpeg, vol);

  return Mixer::instance()->setMusicVolume(vol);
}

bool
Video::play(int loop) const
{  
  if(!mpeg)
    return false;

  /* Hook in the MPEG music mixer */
#ifdef HAVE_MIXER
  if(Mixer::instance()->audioAvailable()) {
    SMPEG_enableaudio((SMPEG*)mpeg, 1);
    Mix_HookMusic(SMPEG_playAudioSDL, (SMPEG*)mpeg);
  }
#endif
  SMPEG_play((SMPEG*)mpeg);

  return true;
}

bool
Video::playing() const
{  
  return mpeg && (SMPEG_status((SMPEG*)mpeg) == SMPEG_PLAYING);
}

void
Video::stop() const
{
  if(!mpeg)
    return;

  SMPEG_stop((SMPEG*)mpeg);

#ifdef HAVE_MIXER
  // unhook SMPEG from the mixer
  if(Mixer::instance()->audioAvailable()) {
    Mix_HookMusic(NULL, NULL);
    SMPEG_enableaudio((SMPEG*)mpeg, 0);
  }
#endif
}

void
Video::pause()
{
  if(mpeg && !paused_) {
    SMPEG_pause((SMPEG*)mpeg);
    paused_ = true;
  }
}

void
Video::resume()
{  
  if(mpeg && paused_) {
    SMPEG_pause((SMPEG*)mpeg);
    paused_ = false;
  }
}

void
Video::rewind() const
{  
  if(mpeg)
    SMPEG_rewind((SMPEG*) mpeg);
}

void
Video::screenUpdate(SDL_Surface* surf, int x, int y, unsigned w, unsigned h)
{
  // mutex should already be grabbed at this point,
  // since SMPEG is accessing the target surface

  Debug out(Debug::DRAWING);

  out << "Updating SMPEG video on screen" << Debug::endl;

  map_mutex_.grab();

  SurfaceMap::iterator I = map_.find(surf);
  Video *video = I != map_.end() ? I->second : 0;

  map_mutex_.release();

  if(video) {
    assert(surf == video->sdlSurface_);
    video->update(Rect(x, y, w, h));
  }
}

void
Video::update(const Rect& r)
{
  // mutex_ is already grabbed at this point, since we're being called
  // from screenUpdate()

  // calculate dirty region

  Region mask = Rect(r.x + loc_.x, r.y + loc_.y, r.w, r.h);
  if(use_mask_) // this is almost always true
    mask &= mask_;

  if(mask.empty()) {
    Debug::out << "No exposed video to update" << Debug::endl;
    return;
  }
  else
    Debug::out << "Exposed video lies inside" << mask.getClipbox() << Debug::endl;

  // screen update

  screen_->mutex.grab();

  blit(*screen_, loc_, mask);
  screen_->update(mask);

  screen_->mutex.release();
}

#else // HAVE_SMPEG
//-------------------------------------------------------------
// dummy implementation when not linking against libsmpeg

Video::Video(const Point&, ScreenSurface*)
{
}

Video::~Video()
{
}

void Video::setMask(const Region&)
{
}

void Video::clearMask()
{
}

void Video::move(const Point&)
{
}

bool
Video::load(const std::string&, double scale )
{
  return false;
}

void
Video::pause()
{  
}

void
Video::resume()
{  
}

void
Video::rewind() const
{  
}

bool
Video::play(int ) const
{  
  return false;
}

int
Video::setVolume(int)
{
  return 0;
}

bool
Video::playing() const
{  
  return false;
}

void
Video::stop() const
{

}

#endif // HAVE_SMPEG

} // namespace

