/* worker.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: worker.cc,v 1.49 2002/03/22 22:38:38 ralf Exp $ */

#include "worker.h"
#include "lister.h"
#include "locale.h"
#include "normalmode.h"
#include "wconfig.h"
#include "filereq.h"
#include "grouphash.h"
#include "wdefines.h"
#include "showimagemode.h"
#include "informationmode.h"
#include <string>

Requester *Worker::req_static=NULL;

Worker::Worker(int targc,char **targv)
{
  aguix=new AGUIX();
  aguix->initX(targc,targv,"Worker");
  mainwin=NULL;
  aboutb=NULL;
  configureb=NULL;
  clockbar=NULL;
  statebar=NULL;
  pathbs=NULL;
  buttons=NULL;
  lister[0]=NULL;
  lister[1]=NULL;
  oldrows=0;
  oldcolumns=0;
  w=600;
  h=400;
  pbanknr=0;
  bbanknr=0;
  runningmode=WORKER_NORMAL;
  this->argc=targc;
  this->argv=targv;
  
  dm=new DNDMsg(this);
  lasttimep=0;
  
  if(req_static==NULL) {
    // only if not initialized
    req_static=new Requester(aguix);
  }
  freesp = new PartSpace();
}

Worker::~Worker()
{
  if(req_static!=NULL) {
    delete req_static;
    req_static=NULL;
  }

  delete aguix;
  delete dm;
  delete freesp;
}

void Worker::run()
{
  int i,j;
  wconfig->loadsize(&w,&h);
  aguix->setFont(wconfig->getFont(0));
  buildMainWin(false);
  lister[0]=new Lister(this);
  lister[1]=new Lister(this);
  initListerState();
  AGMessage *msg;
  bool handleSub;
  int rows,columns;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  int pathact=-1;
  bool found;
  bool restart=false;
  bool configstarted=false;
  int waitcount=0;
  while(runningmode!=WORKER_QUIT) {
    msg=aguix->GetMessage(NULL);
    handleSub=true;
    found=false;
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          quit(1);
          handleSub=false;
          break;
        case AG_SIZECHANGED:
          if(msg->size.window==mainwin->getWindow()) {
            w=msg->size.neww;
            h=msg->size.newh;
            resizeMainWin(w,h);
          } else handleSub=false;
          break;
        case AG_BUTTONPRESSED:
          if(msg->button.state==2) {
            for(i=0;i<rows;i++) {
              if(pathbs[i]==msg->button.button) {
                for(j=0;j<i;j++) pathbs[j]->setState(2);
                for(j=i+1;j<rows;j++) pathbs[j]->setState(2);
                pathact=i;
                handleSub=false;
                break;
              }
            }
          }
          break;
        case AG_BUTTONRELEASED:
          if(msg->button.button==aboutb) {
            about();
            handleSub=false;
          } else if(msg->button.button==configureb) {
            restart=configure();
            handleSub=false;
            configstarted=true;
          } else if(msg->button.button==clockbar) {
            if( msg->button.state == 1 ) {
              shuffleButton( 1 );
            } else if ( msg->button.state == 2 ) {
              shuffleButton( -1 );
            }
            handleSub=false;
          } else {
            for(i=0;i<rows;i++) {
              if(pathbs[i]==msg->button.button) {
                if(msg->button.state==2) {
                  shufflePath();
                } else {
                  // Path setzen
                  setPath(i+pbanknr*rows);
                }
                found=true;
                break;
              }
            }
            if(found==false) {
              for(i=0;i<rows;i++) {
                for(j=0;j<columns;j++) {
                  if(msg->button.button==buttons[i*columns+j]) {
                    activateButton(i,j,msg->button.state);
                    found=true;
                  }
                }
              }
            } else handleSub=false;
          }
          break;
        case AG_MOUSERELEASED:
          if(pathact>=0) {
            for(j=0;j<pathact;j++) pathbs[j]->setState(0);
            for(j=pathact+1;j<rows;j++) pathbs[j]->setState(0);
            pathact=-1;
            handleSub=false;
          }
          break;
        case AG_DND_START:
          // Start of DND; only safe the message
          dm->setStart(&(msg->dnd));
          break;
        case AG_DND_END:
          // End of DND; safe the message and find the mode
          dm->setEnd(&(msg->dnd));
          if(dm->getSourceLister()!=NULL) dm->getSourceLister()->startdnd(dm);
          break;
        case AG_KEYPRESSED:
//        case AG_KEYRELEASED:
          activateShortkey(msg);
          break;
        default:
          break;
      }
      if(handleSub==true) {
        lister[0]->messageHandler(msg);
        lister[1]->messageHandler(msg);
      }
      aguix->ReplyMessage(msg);
    } else {
      updateTime();
      if(waitcount>3) {
        lister[0]->cyclicfunc(CYCLICFUNC_MODE_NORMAL);
        lister[1]->cyclicfunc(CYCLICFUNC_MODE_NORMAL);
      }
      waitcount++;
      waittime(20);
    }
    if(configstarted==true) {
      if(restart==false) {
        lister[0]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
        lister[1]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      }
      configstarted=false;
    }
    if(restart==true) {
      //TODO
      aguix->setFont(wconfig->getFont(0));
      buildMainWin(true);
      rows=wconfig->getRows();
      columns=wconfig->getColumns();
      lister[0]->reconfig();
      lister[1]->reconfig();
      lister[0]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      lister[1]->cyclicfunc(CYCLICFUNC_MODE_RESTART);
      restart=false;
      lasttimep=0;
    }
  }
  saveListerState();
  wconfig->savesize(w,h);
  delete lister[0];
  delete lister[1];
  mainwin->close();
  delete mainwin;
}

AGUIX *Worker::getAGUIX()
{
  return aguix;
}

int main(int argc,char **argv)
{
  bool showversion,showhelp;
  int i;
  char *setlang;

  showversion=false;
  showhelp=false;
  for(i=1;i<argc;i++) {
    if((strcmp(argv[i],"-V")==0)||(strcmp(argv[i],"--version")==0)) {
      showversion=true;
    } else if((strcmp(argv[i],"-help")==0)||(strcmp(argv[i],"--help")==0)) {
      showhelp=true;
    }
  }
  if(showversion==true) {
    printf("Worker Version: %d.%d.%d%s\n",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    printf("  Author: Ralf Hoffmann <ralf.hoffmann@epost.de>\n");
    printf("  Homepage: http://www.boomerangsworld.de/worker\n\n");
    exit(0);
  }
  if(showhelp==true) {
    printf("Worker Filemanager\n");
    printf("  Version: %d.%d.%d%s\n",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    printf("  Author: Ralf Hoffmann <ralf.hoffmann@epost.de>\n");
    printf("  Homepage: http://www.boomerangsworld.de/worker\n\n");
    printf("  Options: -V, --version\tPrint the program version and exit\n");
    printf("           -help, --help\tPrint this help and exit\n\n");
    printf("           <dir>\t\tShows the dir on the left side\n");
    printf("           <dir1> <dir2>\tShows dir1 on the left side, dir2 on the right side\n\n");
    exit(0);
  }
#ifdef USE_MEM_SYSTEM
  _memsysteminit();
#endif
  ugdb=new GroupHash();
  Worker *worker=new Worker(argc,argv);
  if(worker->getAGUIX()->checkX()==True) {
    wconfig=new WConfig(worker);
    wconfig->applyColorList(wconfig->getColors());
    
    // now check for existence of ".worker" and if not try to create it and copy an example config
    setlang = NULL;
    worker->checkfirststart( &setlang );
    
    wconfig->load();
    
    if ( setlang != NULL ) {
      wconfig->setLang( setlang );
    }

    wconfig->applyColorList(wconfig->getColors());
    wconfig->applyLanguage();

    if ( setlang != NULL ) {
      wconfig->save();
      _freesafe( setlang );
    }
    
    worker->run();
    delete wconfig;
  } else {
    printf("Worker: Can't connect to X!\n");
  }
  delete worker;
  delete ugdb;
#ifdef USE_MEM_SYSTEM
  catalog.freeLanguage();
  _memsystemcheck();
#endif
  return 0;
}

int Worker::getMaxModeNr()
{
  // leider hardcoded, da aber die schlechte Methode in dieser Klasse bleibt, nicht so
  // schlimm
  return 3;
}

ListerMode *Worker::getMode4ID(int nr,Lister *parent)
{
  ListerMode *lm=NULL;
  switch(nr) {
    case 1:
      lm=new ShowImageMode(parent);
      break;
    case 2:
      lm=new InformationMode(parent);
      break;
    default:
      lm=new NormalMode(parent);
      break;
  }
  return lm;
}

int Worker::getID4Mode(ListerMode *lm)
{
  int type=-1;
  if(lm->isType("NormalMode")==true) type=0;
  else if(lm->isType("ShowImageMode")==true) type=1;
  else if(lm->isType("InformationMode")==true) type=2;
  return type;
}

AWindow *Worker::getMainWin()
{
  return mainwin;
}

void Worker::buildMainWin(bool removeold)
{
  // nur temporr
  int rows,columns;
  int tx,ty,tw,th;
  unsigned int i;
  int i2;
  AGUIXFont *font;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  if(removeold==true) {
    if((oldrows!=0)&&(oldcolumns!=0)) {
      for(i=0;i<oldrows;i++) {
        for(unsigned int j=0;j<oldcolumns;j++) {
          mainwin->remove(buttons[i*oldcolumns+j]);
        }
      }
      delete buttons;
      for(i=0;i<oldrows;i++) {
        mainwin->remove(pathbs[i]);
      }
      delete pathbs;
    }
  } else {
    mainwin=new AWindow(aguix);
    mainwin->create(NULL,10,10,w,h,0,"Worker");
    mainwin->setMinSize(300,200);

    aboutb=new Button(aguix,0,0,10,"A",1,0,1);
    aboutb->getSize(&tw,&th);
    aboutb->resize(th,th);
    mainwin->add(aboutb);
    tx=th;
    configureb=new Button(aguix,tx,0,th,"C",1,0,2);
    mainwin->add(configureb);
    tx+=th;
    statebar=new SolidButton(aguix,tx,0,w-2*th,"",1,0,0);
    mainwin->add(statebar);
    clockbar=new Button(aguix,0,h-th,w,"","",1,1,0,0,3);
    clockbar->setShowDual(false);
    mainwin->add(clockbar);
  }
  font=aguix->getFont(wconfig->getFont(1));
//  th=aboutb->getHeight();
  th=aguix->getCharHeight()+4;
  aboutb->resize(th,th);
  tx=th;
  configureb->resize(th,th);
  configureb->move(tx,0);
  tx+=th;
  statebar->resize(w-2*th,th);
  statebar->move(tx,0);
  clockbar->resize(w,th);
  clockbar->move(0,h-th);
  if(font==NULL) {
    ty=h-rows*th-th;
    tw=w/(columns+1);
    buttons=new Button*[columns*rows];
    for(i2=0;i2<rows;i2++) {
      for(int j=0;j<columns;j++) {
        buttons[i2*columns+j]=new Button(aguix,w-(columns-j)*tw,ty+i2*th,tw,"",1,0,0);
        mainwin->add(buttons[i2*columns+j]);
      }
    }
    tw=w-columns*tw;
    pathbs=new Button*[rows];
    for(i2=0;i2<rows;i2++) {
      pathbs[i2]=new Button(aguix,0,ty+i2*th,tw,"","",1,1,0,0,0);
      mainwin->add(pathbs[i2]);
    }
  } else {
    ty=h-th;
    th=font->getCharHeight()+4;
    ty-=rows*th;
    tw=w/(columns+1);
    buttons=new Button*[columns*rows];
    for(i2=0;i2<rows;i2++) {
      for(int j=0;j<columns;j++) {
        buttons[i2*columns+j]=new Button(aguix,w-(columns-j)*tw,ty+i2*th,tw,th,"",1,0,0);
        buttons[i2*columns+j]->setFont(wconfig->getFont(1));
        mainwin->add(buttons[i2*columns+j]);
      }
    }
    tw=w-columns*tw;
    pathbs=new Button*[rows];
    for(i2=0;i2<rows;i2++) {
      pathbs[i2]=new Button(aguix,0,ty+i2*th,tw,"","",1,1,0,0,0);
      pathbs[i2]->resize(tw,th);
      pathbs[i2]->setFont(wconfig->getFont(1));
      mainwin->add(pathbs[i2]);
    }
  }
  for(i2=0;i2<rows;i2++) {
    pathbs[i2]->setShowDual(false);
  }
  if(removeold!=true) {
    mainwin->show();
  } else {
    //TODO: Lister anpassen
    //Sollte so reichen
    AGMessage *msg=new AGMessage;
    msg->type=AG_SIZECHANGED;
    msg->size.window=mainwin->getWindow();
    msg->size.neww=w;
    msg->size.newh=h;
    lister[0]->messageHandler(msg);
    lister[1]->messageHandler(msg);
    delete msg;
  }
  oldrows=rows;
  oldcolumns=columns;
  showPathBank();
  showButtonBank();
  aboutb->setFG(0,wconfig->getStatusbar(0));
  aboutb->setBG(0,wconfig->getStatusbar(1));
  configureb->setFG(0,wconfig->getStatusbar(0));
  configureb->setBG(0,wconfig->getStatusbar(1));
  statebar->setFG(wconfig->getStatusbar(0));
  statebar->setBG(wconfig->getStatusbar(1));
  clockbar->setFG(0,wconfig->getClockbar(0));
  clockbar->setBG(0,wconfig->getClockbar(1));
  clockbar->setFG(1,wconfig->getClockbar(0));
  clockbar->setBG(1,wconfig->getClockbar(1));
}

void Worker::resizeMainWin(int nw,int nh)
{
  int rows,columns;
  AGUIXFont *font;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  int tx,ty,tw,th,i;
  aboutb->getSize(&tw,&th);
  tx=th;
  tx+=th;
  statebar->resize(w-2*th,th);
  clockbar->resize(w,th);
  clockbar->move(0,h-th);
  ty=h-th;
  font=aguix->getFont(wconfig->getFont(1));
  th=font->getCharHeight()+4;
  ty-=rows*th;
  tw=w/(columns+1);
  for(i=0;i<rows;i++) {
    for(int j=0;j<columns;j++) {
      buttons[i*columns+j]->move(w-(columns-j)*tw,ty+i*th);
      buttons[i*columns+j]->resize(tw,th);
    }
  }
  tw=w-columns*tw;
  for(i=0;i<rows;i++) {
    pathbs[i]->move(0,ty+i*th);
    pathbs[i]->resize(tw,th);
  }
}

void Worker::getGeometry(Lister *l,int *lx,int *ly,int *lw,int *lh)
{
  // temp.
  int rows,columns;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  int tw,th;
  int ttx,tty,ttw,tth;
  aboutb->getSize(&tw,&th);
  tty=th;
  if(lister[0]==NULL) lister[0]=l;
  else if(lister[1]==NULL) lister[1]=l;
  if(lister[1]==l) ttx=w/2; else ttx=0;
  if(lister[1]==l) ttw=w-w/2; else ttw=w/2;
  tth=pathbs[0]->getY()-tty;
  *lx=ttx;
  *ly=tty;
  *lw=ttw;
  *lh=tth;
}

void Worker::setStatebarText(const char *str)
{
  if(statebar!=NULL) statebar->setText(str);
}

void Worker::shufflePath()
{
  pbanknr++;
  showPathBank();
}

void Worker::shuffleButton( int dir )
{
  int rows, columns, maxbank;

  rows = wconfig->getRows();
  columns = wconfig->getColumns();
  List *buttonlist = wconfig->getButtons();
  maxbank = buttonlist->size() / ( rows * columns * 2 );

  bbanknr += dir;
  if ( bbanknr < 0 ) bbanknr = maxbank - 1;
  else if ( bbanknr >= maxbank ) bbanknr = 0;
  showButtonBank();
}

void Worker::showPathBank()
{
  int rows,id,maxbank;
  List *paths=wconfig->getPaths();
  rows=wconfig->getRows();
  maxbank=paths->size()/rows;
  if(pbanknr>=maxbank) pbanknr=0;
  id=paths->initEnum();
  WCPath *p1=(WCPath*)paths->getElementAt(id,pbanknr*rows);
  for(int i=0;i<rows;i++) {
    if( p1->getCheck() == true ) {
      pathbs[i]->setText(0,p1->getName());
      pathbs[i]->setFG(0,p1->getFG());
      pathbs[i]->setBG(0,p1->getBG());
      pathbs[i]->setText(1,p1->getName());
      pathbs[i]->setFG(1,p1->getFG());
      pathbs[i]->setBG(1,p1->getBG());
      pathbs[i]->activate(0);
    } else {
      pathbs[i]->setText(0,"");
      pathbs[i]->setFG(0,0);
      pathbs[i]->setBG(0,0);
      pathbs[i]->setText(1,"");
      pathbs[i]->setFG(1,0);
      pathbs[i]->setBG(1,0);
      pathbs[i]->deactivate(0);
    }
    p1=(WCPath*)paths->getNextElement(id);
  }
  paths->closeEnum(id);
}

void Worker::showButtonBank()
{
  int rows,columns,id,maxbank;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  List *buttonlist=wconfig->getButtons();
  maxbank=buttonlist->size()/(rows*columns*2);
  if(bbanknr>=maxbank) bbanknr=0;
  id=buttonlist->initEnum();
  WCButton *b1=(WCButton*)buttonlist->getElementAt(id,bbanknr*rows*columns*2);
  for(int i=0;i<rows;i++) {
    for(int j=0;j<columns;j++) {
      if( b1->getCheck() == true ) {
        buttons[i*columns+j]->setText(0,b1->getText());
        buttons[i*columns+j]->setFG(0,b1->getFG());
        buttons[i*columns+j]->setBG(0,b1->getBG());
        // only activate if this button has some commands
        if ( b1->getComs()->size() > 0 )
          buttons[i*columns+j]->activate(0);
        else
          buttons[i*columns+j]->deactivate(0);
      } else {
        buttons[i*columns+j]->setText(0,"");
        buttons[i*columns+j]->setFG(0,0);
        buttons[i*columns+j]->setBG(0,0);
        buttons[i*columns+j]->deactivate(0);
      }
      b1=(WCButton*)buttonlist->getNextElement(id);
      if(b1->getCheck()==true) {
        buttons[i*columns+j]->setDualState(true);
        buttons[i*columns+j]->setShowDual(true);
        buttons[i*columns+j]->setText(1,b1->getText());
        buttons[i*columns+j]->setFG(1,b1->getFG());
        buttons[i*columns+j]->setBG(1,b1->getBG());
        buttons[i*columns+j]->activate(1);
      } else {
        buttons[i*columns+j]->setDualState(false);
        buttons[i*columns+j]->setShowDual(false);
        buttons[i*columns+j]->deactivate(1);
      }
      b1=(WCButton*)buttonlist->getNextElement(id);
    }
  }
  buttonlist->closeEnum(id);
}

void Worker::about()
{
  Requester *req=new Requester(aguix);
  char *tstr;
  int line=0;
  tstr=(char*)_allocsafe(40960);
  sprintf(tstr,"Worker %d.%d.%d%s|(C) Ralf Hoffmann",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
  line=2;
#ifdef HAVE_LIBZ
  strcat(tstr,"||This program transparently decompress GZIPed files");
  line+=2;
#endif
#ifdef WANT_THREADS
  strcat(tstr,"|This program is compiled with thread-support");
  line+=2;
#endif
#ifdef USE_MEM_SYSTEM
  if(line==2) strcat(tstr,"|");
  strcat(tstr,"|This program is compiled with SLOW memorysystem"); 
  line++;
#endif
#ifdef DEVELOPER
  if(line==2) strcat(tstr,"|");
  strcat(tstr,"|This program is compiled as developer version|  If you aren't a developer, please notice the author about it!"); 
  line+=2;
#endif
#ifdef DEBUG
  if(line==2) strcat(tstr,"|");
  strcat(tstr,"|This program is compiled as debugging version|  If you aren't a developer, please notice the author about it!"); 
  line+=2;
#endif
  strcat(tstr,"||Homepage: http://www.boomerangsworld.de/worker|EMail: Ralf Hoffmann <ralf.hoffmann@epost.de>"); 
  strcat(tstr,"||Released under the GNU General Public License:||"); 
  strcat(tstr,"This program is free software; you can redistribute it and/or modify|");
  strcat(tstr,"it under the terms of the GNU General Public License as published by|");
  strcat(tstr,"the Free Software Foundation; either version 2 of the License, or|");
  strcat(tstr,"(at your option) any later version.||");
  strcat(tstr,"This program is distributed in the hope that it will be useful,|");
  strcat(tstr,"but WITHOUT ANY WARRANTY; without even the implied warranty of|");
  strcat(tstr,"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the|");
  strcat(tstr,"GNU General Public License for more details.");
  req->request(catalog.getLocale(208),tstr,catalog.getLocale(11));
  _freesafe(tstr);
  delete req;
}

bool Worker::configure()
{
  return wconfig->configure();
}

Lister *Worker::getOtherLister(Lister *l1)
{
  if(l1==lister[0]) return lister[1];
  if(l1==lister[1]) return lister[0];
  return NULL;
}

int Worker::getSide(Lister *l1)
{
  if(l1==lister[0]) return 0;
  if(l1==lister[1]) return 1;
  return -1;
}

void Worker::setPath(int nr)
{
  List *paths=wconfig->getPaths();
  WCPath *p1=(WCPath*)paths->getElementAt(nr);
  if(p1!=NULL) {
    if(p1->getCheck()==true) {
      Lister *l1=lister[0];
      if(lister[1]->isActive()==true) l1=lister[1];
      l1->switch2Mode(0);
      NormalMode *nm=(NormalMode*)l1->getActiveMode();
      if(nm!=NULL) {
        nm->enterDir(p1->getPath());
      } else {
        printf("Unexpected error in Worker::setPath 2\n");
      }
    }
  } else {
    printf("Unexpected error in Worker::setPath 1\n");
  }
}

int Worker::interpret(List* coms,ActionMessage *msg)
{
  FunctionProto *fp;
  int id=coms->initEnum();
  fp=(FunctionProto*)coms->getFirstElement(id);
  while(fp!=NULL) {
    fp->run(msg);
    fp=(FunctionProto*)coms->getNextElement(id);
  }
  coms->closeEnum(id);
  return 0;
}

void Worker::activateShortkey(AGMessage *msg)
{
  int keystate,id,i;
  List *l1;
  WCHotkey *hk1;
  WCButton *b1;
  WCPath *p1;
  WCShortkey *sk1;
  ActionMessage amsg;
  bool found;
  if((msg->type==AG_KEYPRESSED)||(msg->type==AG_KEYRELEASED)) {
    keystate=KEYSTATEMASK(msg->key.keystate);
    found=false;
    l1=wconfig->getHotkeys();
    id=l1->initEnum();
    hk1=(WCHotkey*)l1->getFirstElement(id);
    while(hk1!=NULL) {
      sk1=hk1->getKey();
      if(sk1->isShortkey(msg->key.key,keystate)==true) {
        amsg.flags=hk1->getComsGF();
        amsg.mode=amsg.AM_MODE_NORMAL;
        interpret(hk1->getComs(),&amsg);
        found=true;
        break;
      }
      hk1=(WCHotkey*)l1->getNextElement(id);
    }
    l1->closeEnum(id);
    if(found==false) {
      l1=wconfig->getButtons();
      id=l1->initEnum();
      b1=(WCButton*)l1->getFirstElement(id);
      while(b1!=NULL) {
        sk1=b1->getKey();
        if(sk1->isShortkey(msg->key.key,keystate)==true) {
          amsg.flags=b1->getComsGF();
          amsg.mode=amsg.AM_MODE_NORMAL;
          interpret(b1->getComs(),&amsg);
          found=true;
          break;
        }
        b1=(WCButton*)l1->getNextElement(id);
      }
      l1->closeEnum(id);
      if(found==false) {
        l1=wconfig->getPaths();
        id=l1->initEnum();
        p1=(WCPath*)l1->getFirstElement(id);
        i=0;
        while(p1!=NULL) {
          sk1=p1->getKey();
          if(sk1->isShortkey(msg->key.key,keystate)==true) {
            setPath(i);
            found=true;
            break;
          }
          p1=(WCPath*)l1->getNextElement(id);
          i++;
        }
        l1->closeEnum(id);
      }
    }
  }
}

Lister *Worker::getLister(int side)
{
  if((side<0)||(side>1)) return NULL;
  return lister[side];
}

Lister *Worker::getActiveLister()
{
  if(lister[0]->isActive()==true) return lister[0];
  if(lister[1]->isActive()==true) return lister[1];
  return NULL;
}

void Worker::activateButton(int i,int j,int m)
{
  List *l1;
  WCButton *b1;
  int id,pos;
  int rows,columns;
  ActionMessage amsg;
  rows=wconfig->getRows();
  columns=wconfig->getColumns();
  l1=wconfig->getButtons();
  id=l1->initEnum();
  pos=2*(bbanknr*rows*columns+i*columns+j)+m-1;
  b1=(WCButton*)l1->getElementAt(id,pos);
  if(b1!=NULL) {
    if(b1->getCheck()==true) {
      amsg.flags=b1->getComsGF();
      amsg.mode=amsg.AM_MODE_NORMAL;
      interpret(b1->getComs(),&amsg);
    }
  }
  l1->closeEnum(id);
}

int Worker::saveListerState()
{
  // Datei oeffnen ($HOME/.worker/listerstates)
  // int-Wert mit Seite (0/1)
  // Vielleicht: Die ListerModes nicht durchnumeriert laden sondern
  //             die ID speichern und dann dem entsprechenden Mode
  //             das laden ueberlassen, nach Seitennummer die Anzahl der
  //             der abgespeicherten Modis
  // Also der Lister-lade-funktion den FH geben und die macht dann den Rest

  const char *home=wconfig->getHome();
  int len=strlen(home);
  if(len<1) return false;
  char *str=(char*)_allocsafe((len+1+12+2)*sizeof(char));
  strcpy(str,home);
  if(len>1) if(home[len-1]!='/') strcat(str,"/");
  strcat(str,"listerstates");
  Datei *fh=new Datei();
  if(fh->open(str,"w")==0) {
    fh->putInt(0);
    lister[0]->saveState(fh);
    fh->putInt(1);
    lister[1]->saveState(fh);
    fh->close();
  }
  _freesafe(str);
  return 0;
}

int Worker::initListerState()
{
  int pos,s;
  char *startd[2],*tstr,*tstr2;
  const char *home=wconfig->getHome();
  int len=strlen(home);
  if(len<1) return false;
  char *str=(char*)_allocsafe((len+1+12+2)*sizeof(char));
  strcpy(str,home);
  if(len>1) if(home[len-1]!='/') strcat(str,"/");
  strcat(str,"listerstates");
  Datei *fh=new Datei();
  int rside;
  char *actdir;
  int size=4096;

  if(fh->open(str,"r")==0) {
    for(int i=0;i<2;i++) {
      rside=fh->getInt();
      if((rside>=0)&&(rside<2)) {
        lister[rside]->loadState(fh);
      }
    }
    fh->close();
  }
  _freesafe(str);
  
  // TODO: Vielleicht nur diesen check, wenn Datei nicht gefunden oder beide auf -1
  int lmode=-1,rmode=-1;
  if(lister[0]->getActiveMode()!=NULL) {
    lmode=getID4Mode(lister[0]->getActiveMode());
  }
  if(lister[1]->getActiveMode()!=NULL) {
    rmode=getID4Mode(lister[1]->getActiveMode());
  }
  if((lmode!=0)&&(rmode!=0)) {
    lister[0]->switch2Mode(0);
    lister[1]->switch2Mode(0);
  }
  //now check for startdirs
  startd[0]=NULL;
  startd[1]=NULL;
  pos=1;
  for(s=0;s<2;s++) {
    for(;pos<argc;pos++) {
      if(strncmp(argv[pos],"-",1)!=0) {
        if(strncmp(argv[pos],"--",2)!=0) {
          startd[s]=dupstring(argv[pos]);
          break;
        }
      }
    }
    pos++;
  }

  for(;;) {
    actdir=(char*)_allocsafe(size);
    if(getcwd(actdir,size)!=NULL) break;
    else if(errno!=ERANGE) {
      strcpy(actdir,"/");  // use the root if getcwd doesn't report the actual
      break;
    }
    _freesafe(actdir);
    size*=2;
  }

  for(s=0;s<2;s++) {
    if(startd[s]==NULL) {
      tstr=wconfig->getStartDir(s);
      if(tstr!=NULL) startd[s]=dupstring(tstr);
    }
    if(startd[s]!=NULL) {
      if(strlen(startd[s])>0) {
        lister[s]->switch2Mode(0);
        NormalMode *nm=(NormalMode*)lister[s]->getActiveMode();
        if(nm!=NULL) {
          if(startd[s][0]!='/') {
            tstr=catstring(actdir,"/");
            tstr2=catstring(tstr,startd[s]);
            _freesafe(tstr);
          } else {
            tstr2=dupstring(startd[s]);
          }
          tstr=HandlePathExt(tstr2);
          _freesafe(tstr2);
          nm->enterDir(tstr);
          _freesafe(tstr);
        }
      }
      _freesafe(startd[s]);
    }
  }
  _freesafe(actdir);
  
  /* if no side is active make 0 active */
  if(getActiveLister()==NULL) lister[0]->makeActive();
  return 0;
}

int Worker::quit(int mode)
{
  int returnvalue=0;
  if(mode==1) {
    runningmode=WORKER_QUIT;
    returnvalue=1;
  } else {
    Requester *req=new Requester(aguix);
    char *str;
    str=(char*)_allocsafe(strlen(catalog.getLocale(11))+1+strlen(catalog.getLocale(8))+1);
    sprintf(str,"%s|%s",catalog.getLocale(11),catalog.getLocale(8));
    int res=req->request(catalog.getLocale(114),catalog.getLocale(115),str);
    _freesafe(str);
    if(res==0) {
      runningmode=WORKER_QUIT;
      returnvalue=1;
    }
    delete req;
  }
  return returnvalue;
}

void Worker::updateTime(void)
{
  struct tm *timeptr;
#ifdef HAVE_LINUX_KERNEL_H
  struct sysinfo info;
  unsigned long freeram,freeswap;
#endif
  time_t timep;
  char *str,*buffer,*tmpfile,*exestr;
  const char *clbc;
  int dt=wconfig->getClockbarUpdatetime();
  int maxchars,rc,i;
  WConfig::clockbar_mode_t cbm=wconfig->getClockbarMode();
  FILE *fp;
  
  time(&timep);
  if((lasttimep==0)||(((lasttimep+dt)<=timep)&&(cbm!=WConfig::CLOCKBAR_MODE_VERSION))) {
    if(cbm==WConfig::CLOCKBAR_MODE_VERSION) {
      buffer=(char*)_allocsafe(strlen("Worker %d.%d.%d%s - (C) Ralf Hoffmann")+strlen(WORKER_COMMENT)+3*sizeof(int));
      sprintf(buffer,"Worker %d.%d.%d%s - (C) Ralf Hoffmann",WORKER_MAJOR,WORKER_MINOR,WORKER_PATCH,WORKER_COMMENT);
    } else if(cbm==WConfig::CLOCKBAR_MODE_EXTERN) {
      clbc=wconfig->getClockbarCommand();
      if(strlen(clbc)>0) {
        // this is the count the max chars fit in the clockbar
        maxchars=clockbar->getWidth();
        maxchars/=aguix->getCharWidth();

        // start the program and store the output in a temp file
        // then read no more then maxchars bytes and display it
        tmpfile=Datei::createTMPName();
        exestr=(char*)_allocsafe(strlen(clbc)+strlen(" >%s")+strlen(tmpfile)+1);
        sprintf(exestr,"%s >%s",clbc,tmpfile);
        system(exestr);
        _freesafe(exestr);

        fp=fopen(tmpfile,"r");
        if(fp!=NULL) {
          buffer=(char*)_allocsafe(maxchars+1);
          rc=fread(buffer,1,maxchars,fp);
          for(i=0;(i<maxchars);i++) {
            if(buffer[i]=='\0') break;
            if(buffer[i]=='\n') {
              buffer[i]='\0';
              break;
            }
            if(i>=rc) {
              buffer[i]='\0';
              break;            
            }
          }
          fclose(fp);
        } else buffer=dupstring("");
        
        remove(tmpfile);
        _freesafe(tmpfile);
      } else buffer=dupstring("");
    } else { // _TIME or _TIMERAM
      timeptr=localtime(&timep);
      str=asctime(timeptr);
      str[strlen(str)-1]=0;
#ifdef HAVE_LINUX_KERNEL_H
      if(cbm==WConfig::CLOCKBAR_MODE_TIMERAM) {
        sysinfo(&info);
        freeram=info.freeram;
        freeswap=info.freeswap;
        
        buffer=(char*)_allocsafe(strlen(str)+
                                 strlen("  --  %s: %ld KB  -  %s: %ld KB")+
                                 strlen(catalog.getLocale(105))+strlen(catalog.getLocale(106))+
                                 2*sizeof(unsigned long)+1);
        
        strcpy(buffer,str);
        sprintf(buffer+strlen(str),"  --  %s: %ld KB  -  %s: %ld KB",catalog.getLocale(105),freeram/1024,
                                                       catalog.getLocale(106),
                                                       freeswap/1024);
      } else {
        buffer=(char*)_allocsafe(strlen(str)+1);
        strcpy(buffer,str);
      }
#else
      buffer=(char*)_allocsafe(strlen(str)+1);
      strcpy(buffer,str);
#endif
    }
    clockbar->setText(0,buffer);
    clockbar->setText(1,buffer);
    _freesafe(buffer);
    lasttimep=timep;
  }
}

void Worker::checkfirststart( char **setlang )
{
  char *tstr,*homestr,*filestr;
  bool copyconfig=false;
  Requester *req=new Requester(aguix);
  char *textstr,*buttonstr;
  int erg;

  homestr=getenv("HOME");
  if(homestr==NULL) {
    textstr="There is no HOME-variable. Please set it to your home and retry.";
    buttonstr="Ok";
    req->request("Worker warning",textstr,buttonstr);
  } else {
    filestr=catstring(homestr,"/.worker/config");
    if(Datei::fileExists(filestr)==false) {
      textstr="It seems that Worker is started the first time!|I will try to install an example configuration in $HOME/.worker|but I will not overwrite any existing file!||If you choose \"Skip\", Worker will use a simple builtin config\
|but then not all functions are accessable (without reconfiguration)!";
      buttonstr="Ok|Skip";
      erg=req->request("Worker message",textstr,buttonstr);
      if(erg==0) copyconfig=true;
    }
    if(copyconfig==true) {
      List *langs = new List();
      DIR *dir2;
      std::string catdir, buttonstr2, str2, str3;
      struct dirent *namelist;
      int id;

      langs->addElement( dupstring( "english" ) );
#ifdef GLOBALDATADIR
      catdir = GLOBALDATADIR;
      catdir += "/catalogs";
      dir2 = opendir( catdir.c_str() );
      if ( dir2 != NULL ) {
        while ( ( namelist = readdir( dir2 ) ) != NULL ) {
          if ( strlen( namelist->d_name ) > 8 ) {
            if ( strcmp( ".catalog", ( namelist->d_name ) + strlen( namelist->d_name ) - 8 ) == 0 ) {
              // Dies ist ein Katalog
              tstr = namelist->d_name;
              tstr[ strlen( tstr ) - 8 ] = 0;
              langs->addElement( dupstring( tstr ) );
            }
          }
        }
        closedir(dir2);
      }
#endif
      // now all available languages are in the list
      // build a requester
      
      id = langs->initEnum();
      tstr = (char*)langs->getFirstElement( id );
      buttonstr2 = "";
      while ( tstr != NULL ) {
        if ( buttonstr2.length() > 0 )
          buttonstr2 += "|";
        buttonstr2 += tstr;
        tstr = (char*)langs->getNextElement( id );
      }

      textstr="Now choose your preferred language (this can be changed later):";
      erg = req->request( "Worker request", textstr, buttonstr2.c_str() );
      
      tstr = (char*)langs->getElementAt( id, erg );
      if ( tstr == NULL ) {
        tstr = (char*)langs->getFirstElement( id );
      }

      catdir = homestr;
      catdir += "/.worker";
      if(Datei::fileExists( catdir.c_str() )==false) {
        str2 = "mkdir ";
        str2 += catdir;

        system( str2.c_str() );
      }

      catdir = homestr;
      catdir += "/.worker/catalogs";
      if(Datei::fileExists( catdir.c_str() )==false) {
        str2 = "mkdir ";
        str2 += catdir;

        system( str2.c_str() );
      }

      str2 = GLOBALDATADIR;
      str2 += "/config-";
      str2 += tstr;
      
      if ( Datei::fileExists( str2.c_str() ) == false ) {
        // choosed language has no default config so take the english
        // and set the language later
        str2 = GLOBALDATADIR;
        str2 += "/config-english";
        if ( setlang != NULL ) {
          // if not english remember to set the language later
          if ( langs->getIndex( tstr ) > 0 )
            *setlang = dupstring( tstr );
        }
      }
      
      if(Datei::fileExists( str2.c_str() )==true) {
        str3 = "cp " + str2 + " " + filestr;        
        system( str3.c_str() );
        if(Datei::fileExists(filestr)==false) {
          textstr="The configfile couldn't be copyied!";
          buttonstr="Ok";
          req->request("Worker error",textstr,buttonstr);
        } else {
          textstr="Installation finished.";
          buttonstr="Ok";
          req->request("Worker message",textstr,buttonstr);
        }
      } else {
        str3 = "There is no example configuration at " + str2 + "!|Perhaps worker is not completely installed!";
        buttonstr="Ok";
        req->request("Worker error", str3.c_str() ,buttonstr);
      }
      tstr = (char*)langs->getFirstElement( id );
      while ( tstr != NULL ) {
        _freesafe( tstr );
        langs->removeFirstElement();
        tstr = (char*)langs->getFirstElement( id );
      }
      langs->closeEnum( id );
      delete langs;
    }
    _freesafe(filestr);
  }
  
  delete req;
}

void Worker::setTitle(const char *add_infos)
{
  if(add_infos==NULL) {
    mainwin->setTitle("Worker");
  } else {
    char *tstr=catstring("Worker - ",add_infos);
    mainwin->setTitle(tstr);
    _freesafe(tstr);
  }
}

Requester *Worker::getRequester()
{
  return req_static;
}

void Worker::activateShortkeyFromList()
{
  // 1.alle mglichen Shortkeys finden und in eine Liste packen
  // 2.Diese Liste sortieren
  //   Hotkey<Button<Pfad   (Hotkey zuerst, weil man die am schnellsten vergisst)
  //   Unter gleichem Typ einfach nach Name case-ins sortieren
  // 3.Aus dieser Liste wird nun das LV erstellt:
  //   <Typ>:<Name>   <Shortkey>
  //   Dabei <Typ>:<Name> maximieren, damit Shortkey ganz rechts steht
  //   Achtung: Vielleicht fr das LV den Font des linken LVs nehmen, falls der User doch mal
  //            einen nichtpro. Font einstellt, dann klappt das auch hier
  // 4.lv Horizontal maximieren
  // 5.lv->takeFocus(), damit man das ganze auch einfach mit der Tastatur benutzen kann
  // 6.der User whlt nun aus
  // 7.Bei Abbruch nichts machen
  // 8.Bei Okay/Return/Doppelklick dann diesen Shortkey aktivieren

  List *sks;
  struct shortkeylisttype *sk1;
  int id;
  List *l1;
  WCHotkey *hk1;
  WCButton *b1;
  WCPath *p1;
  WCShortkey *k1;
  ActionMessage amsg;
  AWindow *win;
  Text *ttext;
  FieldListView *lv;
  int tx,ty,w,tw,l,t1,t2,h;
  char *name1, *name2, *tstr1;
  AGMessage *msg;
  int ende=0;
  Button *okb,*cb;
  bool firstkey;
  int lastrow, row;
  struct timeval lastclick;

  sks=new List();
  l1=wconfig->getHotkeys();
  id=l1->initEnum();
  hk1=(WCHotkey*)l1->getFirstElement(id);
  while(hk1!=NULL) {
    if(hk1->getKey()->getShortkey()!=0) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=hk1;
      sk1->b=NULL;
      sk1->p=NULL;
      sks->addElement(sk1);
    }
    hk1=(WCHotkey*)l1->getNextElement(id);
  }
  l1->closeEnum(id);

  l1=wconfig->getButtons();
  id=l1->initEnum();
  b1=(WCButton*)l1->getFirstElement(id);
  while(b1!=NULL) {
    if(b1->getKey()->getShortkey()!=0) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=NULL;
      sk1->b=b1;
      sk1->p=NULL;
      sks->addElement(sk1);
    }
    b1=(WCButton*)l1->getNextElement(id);
  }
  l1->closeEnum(id);

  l1=wconfig->getPaths();
  id=l1->initEnum();
  p1=(WCPath*)l1->getFirstElement(id);
  while(p1!=NULL) {
    if(p1->getKey()->getShortkey()!=0) {
      sk1=(struct shortkeylisttype*)_allocsafe(sizeof(struct shortkeylisttype));
      sk1->h=NULL;
      sk1->b=NULL;
      sk1->p=p1;
      sks->addElement(sk1);
    }
    p1=(WCPath*)l1->getNextElement(id);
  }
  l1->closeEnum(id);
  
  // sorting
  sks->sort(Worker::shortkeysort);
  
  // create window
  win=new AWindow(aguix);
  w=h=20;
  win->create(NULL,10,10,w,h,0,catalog.getLocale(387));
  
  tx=ty=5;
  ttext=(Text*)win->add(new Text(aguix,tx,ty,catalog.getLocale(388),1));
  tw=ttext->getWidth();
  if(tw>w) w=tw;
  ty+=ttext->getHeight()+5;
  lv = (FieldListView*)win->add( new FieldListView( aguix,
                                                    tx,
                                                    ty,
                                                    50,
                                                    400,
                                                    0 ) );
  lv->setNrOfFields( 5 );
  lv->setFieldWidth( 1, 1 );
  lv->setFieldWidth( 3, 1 );
  
  // add the shortkeys
  id=sks->initEnum();
  sk1=(struct shortkeylisttype*)sks->getFirstElement(id);
  while(sk1!=NULL) {
    k1=NULL;
    name2 = NULL;
    name1 = NULL;
    if(sk1->b!=NULL) {
      k1=sk1->b->getKey();
      name1=catalog.getLocale(389);
      name2=sk1->b->getText();
    } else if(sk1->h!=NULL) {
      k1=sk1->h->getKey();
      name1=catalog.getLocale(391);
      name2=sk1->h->getName();
    } else if(sk1->p!=NULL) {
      k1=sk1->p->getKey();
      name1=catalog.getLocale(390);
      name2=sk1->p->getName();
    }
    if((k1!=NULL)&&(name2!=NULL)) {
      tstr1=aguix->getNameOfKey(k1->getShortkey(),k1->getMod());
      row = lv->addRow();
      lv->setText( row, 0, name1 );
      lv->setText( row, 2, name2 );
      lv->setText( row, 4, tstr1 );
      lv->setPreColors( row, FieldListView::PRECOLOR_ONLYACTIVE );
      _freesafe(tstr1);
    }

    sk1=(struct shortkeylisttype*)sks->getNextElement(id);
  }
  sks->closeEnum(id);
  // lv ready
  
  lv->setHBarState(2);
  lv->setVBarState(2);
  lv->maximizeX();
  tw=lv->getWidth();
  if(tw>w) w=tw;
  ty+=lv->getHeight()+5;
  
  w+=10;

  t1=(strlen(catalog.getLocale(392))+2);
  t1*=aguix->getCharWidth();
  t2=(strlen(catalog.getLocale(8))+2);
  t2*=aguix->getCharWidth();
  tw=5+t1+5+t2+5;
  if(tw>w) w=tw;
  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   ty,
                                   t1,
                                   catalog.getLocale(392),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  w-5-t2,
                                  ty,
                                  t2,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  ty+=okb->getHeight()+5;
  
  win->resize(w,ty);
  win->setMinSize(w,ty);
  win->show();

  lv->takeFocus();

  while((msg=aguix->GetMessage(NULL))!=NULL) aguix->ReplyMessage(msg);
  firstkey=true;
  lastrow = -1;
  while(ende==0) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      if(msg->type==AG_CLOSEWINDOW) {
        if(msg->closewindow.window==win->getWindow()) ende=-1;
      } else if(msg->type==AG_BUTTONCLICKED) {
        if(msg->button.button==okb) ende=1;
        else if(msg->button.button==cb) ende=-1;
      } else if(msg->type==AG_SIZECHANGED) {
        if(msg->size.window==win->getWindow()) {
          w=msg->size.neww;
          h=msg->size.newh;
          okb->move(okb->getX(),h-5-okb->getHeight());
          cb->move(w-5-cb->getWidth(),h-5-cb->getHeight());
          lv->resize(w-10,okb->getY()-lv->getY()-5);
        }
      } else if(msg->type==AG_KEYRELEASED) {
        if((win->isParent(msg->key.window,false)==true)&&(firstkey==false)) {
          if((msg->key.key==XK_Return)||
             (msg->key.key==XK_KP_Enter)) {
            ende=1;
          } else if(msg->key.key==XK_Escape) {
            ende=-1;
          }
        }
      } else if(msg->type==AG_KEYPRESSED) {
        firstkey=false;
      } else if(msg->type==AG_FIELDLV_ONESELECT) {
        if(msg->fieldlv.lv==lv) {
          if(lastrow == -1) {
            lastrow=msg->fieldlv.row;
            gettimeofday(&lastclick,NULL);
          } else {
            if(msg->fieldlv.row == lastrow ) {
              struct timeval acttime;
              gettimeofday(&acttime,NULL);
              if(aguix->isDoubleClick(&acttime,&lastclick)==true) {
                ende=1;
              } else {
                lastrow = msg->fieldlv.row;
                gettimeofday(&lastclick,NULL);
              }
            } else {
              lastrow=msg->fieldlv.row;
              gettimeofday(&lastclick,NULL);
            }
          }
        }
      }
      aguix->ReplyMessage(msg);
    }
  }
  l=-1;
  if(ende==1) {
    l = lv->getActiveRow();
  }
  win->close();
  delete win;

  if(l>=0) {
    sk1=(struct shortkeylisttype*)sks->getElementAt(l);
    if(sk1!=NULL) {
      amsg.mode=amsg.AM_MODE_NORMAL;
      if(sk1->h!=NULL) {
        amsg.flags=sk1->h->getComsGF();
        interpret(sk1->h->getComs(),&amsg);
      } else if(sk1->b!=NULL) {
        amsg.flags=sk1->b->getComsGF();
        interpret(sk1->b->getComs(),&amsg);
      } else if(sk1->p!=NULL) {
        setPath(wconfig->getPaths()->getIndex(sk1->p));
      }
    }
  }
  id=sks->initEnum();
  sk1=(struct shortkeylisttype*)sks->getFirstElement(id);
  while(sk1!=NULL) {
    _freesafe(sk1);
    sks->removeFirstElement();
    sk1=(struct shortkeylisttype*)sks->getFirstElement(id);    
  }
  sks->closeEnum(id);
  delete sks;
}

int Worker::shortkeysort(void*p1,void*p2)
{
  struct shortkeylisttype *sk1=(struct shortkeylisttype*)p1;
  struct shortkeylisttype *sk2=(struct shortkeylisttype*)p2;
  const char *str1,*str2;

  // Hotkey<Button<Pfad
  // Unter gleichem Typ einfach nach Name case-ins sortieren

  if((sk1->h==NULL)&&(sk1->b==NULL)&&(sk1->p==NULL)) return 0;
  if((sk2->h==NULL)&&(sk2->b==NULL)&&(sk2->p==NULL)) return 0;

  // Hotkey ist kleiner als Button & Pfad
  if((sk1->h!=NULL)&&(sk2->h==NULL)) return -1;
  // Button is kleiner Pfad, aber grer als Hotkey
  if(sk1->b!=NULL) {
    if(sk2->h!=NULL) return 1;
    if(sk2->p!=NULL) return -1;
  }
  // Pfad ist grer als Hotkey & Button
  if((sk1->p!=NULL)&&(sk2->p==NULL)) return 1;
  
  // Ab hier sind beide vom gleichen Typ
  str1=str2=NULL;
  if(sk1->h!=NULL) {
    str1=sk1->h->getName();
    str2=sk2->h->getName();
  } else if(sk1->b!=NULL) {
    str1=sk1->b->getText();
    str2=sk2->b->getText();
  } else if(sk1->p!=NULL) {
    str1=sk1->p->getName();
    str2=sk2->p->getName();  
  }
  if((str1!=NULL)&&(str2!=NULL)) {
    return strcasecmp(str1,str2);
  }
  return 0;
}

int Worker::PS_readSpace( const char*name )
{
  return freesp->readSpace( name );
}

long Worker::PS_getBlocksize()
{
  return freesp->getBlocksize();
}

long Worker::PS_getFreeSpace()
{
  return freesp->getFreeSpace();
}

long Worker::PS_getSpace()
{
  return freesp->getSpace();
}

long Worker::PS_getFreeSpaceH(char **unit_return)
{
  return freesp->getFreeSpaceH( unit_return );
}

long Worker::PS_getSpaceH(char **unit_return)
{
  return freesp->getSpaceH( unit_return );
}

