/*
 *  Copyright (C) 1997, 1998 Olivetti & Oracle Research Laboratory
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 *
 *  Seriously modified by Fredrik Hbinette <hubbe@hubbe.net>
 */

/*
 * x.c - functions to deal with X display.
 */

#include <sys/types.h>
#include <unistd.h>
#include <x2vnc.h>
#include <X11/X.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>


Display *dpy;
static Window topLevel;
static int topLevelWidth, topLevelHeight;

static Atom wmProtocols, wmDeleteWindow, wmState;
static Bool modifierPressed[256];

static Bool HandleTopLevelEvent(XEvent *ev);
static Bool HandleRootEvent(XEvent *ev);
static int displayWidth, displayHeight;
static int grabbed;
Cursor  grabCursor;

enum edge_enum edge = EDGE_EAST;
int edge_width=1;
int restingx=-1;
int restingy=-1;
int emulate_wheel=1;
int wheel_button_up=4;

char *client_selection_text=0;
size_t client_selection_text_length=0;


/*
 * This variable is true (1) if the mouse is on the same screen as the one
 * we're monitoring, or if there is only one screen on the X server.
 * - GRM
 */
static Bool mouseOnScreen;

/*
 * CreateXWindow.
 */

Bool CreateXWindow(void)
{
    XSetWindowAttributes attr;
    XEvent ev;
    char defaultGeometry[256];
    XSizeHints wmHints;
    XGCValues gcv;
    int i;

    Pixmap    nullPixmap;
    XColor    dummyColor;

    if (!(dpy = XOpenDisplay(displayname))) {
	fprintf(stderr,"%s: unable to open display %s\n",
		programName, XDisplayName(displayname));
	return False;
    }

    for (i = 0; i < 256; i++)
	modifierPressed[i] = False;

    /* Try to work out the geometry of the top-level window */

    displayWidth = WidthOfScreen(DefaultScreenOfDisplay(dpy));
    displayHeight = HeightOfScreen(DefaultScreenOfDisplay(dpy));

    if(restingy == -1)
    {
      restingy = si.framebufferHeight -2;
      restingx = si.framebufferWidth -2;
    }

    topLevelWidth=edge_width;
    topLevelHeight=edge_width;
    wmHints.x=0;
    wmHints.y=0;

    switch(edge)
    {
      case EDGE_EAST: wmHints.x=displayWidth-edge_width;
      case EDGE_WEST: topLevelHeight=displayHeight;
	break;

      case EDGE_SOUTH: wmHints.y=displayHeight-edge_width;
      case EDGE_NORTH: topLevelWidth=displayWidth;
	break;
    }

    wmHints.flags = PMaxSize | PMinSize |PPosition |PBaseSize;

    wmHints.min_width = topLevelWidth;
    wmHints.min_height = topLevelHeight;

    wmHints.max_width = topLevelWidth;
    wmHints.max_height = topLevelHeight;

    wmHints.base_width = topLevelWidth;
    wmHints.base_height = topLevelHeight;

    sprintf(defaultGeometry, "%dx%d+%d+%d",
	    topLevelWidth, topLevelHeight,
	    wmHints.x, wmHints.y);

    XWMGeometry(dpy, DefaultScreen(dpy), geometry, defaultGeometry, 0,
		&wmHints, &wmHints.x, &wmHints.y,
		&topLevelWidth, &topLevelHeight, &wmHints.win_gravity);

    /* Create the top-level window */

    attr.border_pixel = 0; /* needed to allow 8-bit cmap on 24-bit display -
			      otherwise we get a Match error! */
    attr.background_pixel = BlackPixelOfScreen(DefaultScreenOfDisplay(dpy));
    attr.event_mask = ( LeaveWindowMask|
			StructureNotifyMask|
			ButtonPressMask|
			ButtonReleaseMask|
			PointerMotionMask|
			KeyPressMask|
			KeyReleaseMask|
			EnterWindowMask|
			(resurface?VisibilityChangeMask:0) );
      
    attr.override_redirect=True;

    topLevel = XCreateWindow(dpy, DefaultRootWindow(dpy), wmHints.x, wmHints.y,
			     topLevelWidth, topLevelHeight, 0, CopyFromParent,
			     InputOutput, CopyFromParent,
			     (CWBorderPixel|
			      CWEventMask|
			      CWOverrideRedirect|
			      CWBackPixel),
			     &attr);

    wmHints.flags |= USPosition; /* try to force WM to place window */
    XSetWMNormalHints(dpy, topLevel, &wmHints);

    wmProtocols = XInternAtom(dpy, "WM_PROTOCOLS", False);
    wmDeleteWindow = XInternAtom(dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(dpy, topLevel, &wmDeleteWindow, 1);

    XStoreName(dpy, topLevel, desktopName);


    XMapRaised(dpy, topLevel);

    /*
     * For multi-screen setups, we need to know if the mouse is on the 
     * screen.
     * - GRM
     */
    if (ScreenCount(dpy) > 1) {
      Window root, child;
      int root_x, root_y;
      int win_x, win_y;
      unsigned int keys_buttons;

      XSelectInput(dpy, DefaultRootWindow(dpy), PropertyChangeMask |
                   EnterWindowMask | LeaveWindowMask);
        /* Cut buffer happens on screen 0 only. */
      if (DefaultRootWindow(dpy) != RootWindow(dpy, 0)) {
          XSelectInput(dpy, RootWindow(dpy, 0), PropertyChangeMask);
      }
      XQueryPointer(dpy, DefaultRootWindow(dpy), &root, &child,
                    &root_x, &root_y, &win_x, &win_y, &keys_buttons);
      mouseOnScreen = root == DefaultRootWindow(dpy);
    } else {
      XSelectInput(dpy, DefaultRootWindow(dpy), PropertyChangeMask);
      mouseOnScreen = 1;
    }

    nullPixmap = XCreatePixmap(dpy, DefaultRootWindow(dpy), 1, 1, 1);
    grabCursor = 
      XCreatePixmapCursor(dpy, nullPixmap, nullPixmap,
			  &dummyColor, &dummyColor, 0, 0);
    
    return True;
}



/*
 * ShutdownX.
 */

void
ShutdownX()
{
    XCloseDisplay(dpy);
}


/*
 * HandleXEvents.
 */

Bool HandleXEvents()
{
  XEvent ev;
  
  /* presumably XCheckMaskEvent is more efficient than XCheckIfEvent -GRM */
    while (XCheckIfEvent(dpy, &ev, AllXEventsPredicate, NULL))
/*  while (XCheckMaskEvent(dpy, ~0, &ev)) */
  {
    if (ev.xany.window == topLevel)
    {
      
      if (!HandleTopLevelEvent(&ev))
	return False;
      
    }
    else if (ev.xany.window == DefaultRootWindow(dpy) ||
	     ev.xany.window == RootWindow(dpy, 0)) {
      if (!HandleRootEvent(&ev))
	return False;
    }
    else if (ev.type == MappingNotify)
    {
      XRefreshKeyboardMapping(&ev.xmapping);
    }
  }

  return True;
}

#define EW (edge == EDGE_EAST || edge==EDGE_WEST)
#define NS (edge == EDGE_NORTH || edge==EDGE_SOUTH)
#define ES (edge == EDGE_EAST || edge==EDGE_SOUTH)
#define NS (edge == EDGE_NORTH || edge==EDGE_SOUTH)

static int enter_translate(int isedge, int width, int pos)
{
  if(!isedge) return pos;
  if(ES) return 0;
  return width-1;
}

static int leave_translate(int isedge, int width, int pos)
{
  if(!isedge) return pos;
  if(ES) return width-edge_width;
  return 0;
}


#define EDGE(X) ((X)?edge_width:0)

#define SCALEX(X) \
( ((X)-EDGE(edge==EDGE_WEST ))*(si.framebufferWidth-1 )/(displayWidth -EDGE(EW)) )

#define SCALEY(Y) \
( ((Y)-EDGE(edge==EDGE_NORTH))*(si.framebufferHeight-1)/(displayHeight-EDGE(NS)) )


static void grabit(XEvent *ev)
{
  Window selection_owner;

  XGrabPointer(dpy, topLevel, True,
	       PointerMotionMask | ButtonPressMask | ButtonReleaseMask,
	       GrabModeAsync, GrabModeAsync,
	       None, grabCursor, CurrentTime);
  XGrabKeyboard(dpy, topLevel, True, 
		GrabModeAsync, GrabModeAsync,
		CurrentTime);
  XFlush(dpy);
  
  XWarpPointer(dpy,None, DefaultRootWindow(dpy),0,0,0,0,
	       enter_translate(EW,displayWidth ,ev->xcrossing.x_root),
	       enter_translate(NS,displayHeight,ev->xcrossing.y_root));
  
  SendPointerEvent(SCALEX(ev->xcrossing.x_root), SCALEY(ev->xcrossing.y_root),
		   (ev->xcrossing.state & 0x1f00) >> 8);
  grabbed=1;
  mouseOnScreen = 1;

  selection_owner=XGetSelectionOwner(dpy, XA_PRIMARY);
/*   fprintf(stderr,"Selection owner: %lx\n",(long)selection_owner); */

  if(selection_owner != None && selection_owner != topLevel)
  {
    XConvertSelection(dpy,
		      XA_PRIMARY, XA_STRING, XA_CUT_BUFFER0,
		      topLevel, CurrentTime);
  }
}

/*
 * HandleTopLevelEvent.
 */

static void shortsleep(int usec)
{
  struct timeval timeout;
  timeout.tv_sec=0;
  timeout.tv_usec=usec;
  select(0,0,0,0,&timeout);
}


static Bool HandleTopLevelEvent(XEvent *ev)
{
  Bool grab;
  int i;
  int x, y;
  
  int buttonMask;
  KeySym ks;
  
  switch (ev->type)
  {
    case SelectionRequest:
    {
      XEvent reply;

      XSelectionRequestEvent *req=& ev->xselectionrequest;

      reply.xselection.type=SelectionNotify;
      reply.xselection.display=req->display;
      reply.xselection.selection=req->selection;
      reply.xselection.requestor=req->requestor;
      reply.xselection.target=req->target;
      reply.xselection.time=req->time;
      reply.xselection.property=None;
      
      if(client_selection_text)
      {
	if(req->target == XA_STRING)
	{
	  int ret;
	  if(req->property == None)
	    req->property = XA_CUT_BUFFER0;

#if 0
	  fprintf(stderr,"Setting property %d on window %x to: %s (len=%d)\n",req->property, req->requestor, client_selection_text,client_selection_text_length);
#endif

	  XChangeProperty(dpy,
			  req->requestor,
			  req->property,
			  XA_STRING,
			  8,
			  PropModeReplace,
			  client_selection_text,
			  client_selection_text_length);

	  reply.xselection.property=req->property;
	  
	}
      }
      XSendEvent(dpy, req->requestor, 0,0, &reply);
      XFlush (dpy);
    }
    return 1;

    case SelectionNotify:
    {
      Atom t;
      unsigned long len=0;
      unsigned long bytes_left=0;
      int format;
      unsigned char *data=0;
      int ret;

      ret=XGetWindowProperty(dpy,
			     topLevel,
			     XA_CUT_BUFFER0,
			     0,
			     0x100000,
			     1, /* delete */
			     XA_STRING,
			     &t,
			     &format,
			     &len,
			     &bytes_left,
			     &data);

      if(format == 8)
	SendClientCutText(data, len);
#ifdef DEBUG
      fprintf(stderr,"GOT selection info: ret=%d type=%d fmt=%d len=%d bytes_left=%d %p '%s'\n",
	ret, (int)t,format,len,bytes_left,data,data);
#endif
      if(data) XFree(data);
			 
    }
    return 1;
      
    case VisibilityNotify:
      /*
       * I avoid resurfacing when the window becomes fully obscured, because
       * that *probably* means that somebody is running xlock.
       * Please tell me if you have a problem with this.
       * - Hubbe
       */
      if (ev->xvisibility.state == VisibilityPartiallyObscured && resurface)
      {
	static long last_resurface=0;
	long t=time(0);

	if(t == last_resurface)
	  shortsleep(5000);

	last_resurface=t;
	  
	XRaiseWindow(dpy, topLevel);
      }
      return 1;

      /*
       * We don't need to worry about multi-screen here; that's handled
       * below, as a root event.
       * - GRM
       */
    case EnterNotify:
      if(!grabbed && ev->xcrossing.mode==NotifyNormal)
      {
	grabit(ev);
      }
      return 1;
      
    case MotionNotify:
      
      if(grabbed)
      {
	int i, d;
        int x, y;
        Window warpWindow;
        while (XCheckTypedWindowEvent(dpy, topLevel, MotionNotify, ev))
	  ;	/* discard all queued motion notify events */
	
	i=SendPointerEvent(SCALEX(ev->xmotion.x_root),
			   SCALEY(ev->xmotion.y_root),
			   (ev->xmotion.state & 0x1f00) >> 8);

	if(ev->xmotion.state & 0x1f00) return 1;

          /*
           * Things get complicated for multi-screen X servers,
           * particularly if the PC screen is "inserted" between two
           * X screens.
           * - GRM
           */
        /* First, check for normal edges (applies to single screen or
         * screen edge that doesn't switch screens)
         */
        if (ev->xmotion.same_screen)
	{
          x = ev->xmotion.x_root;
          y = ev->xmotion.y_root;
          warpWindow = DefaultRootWindow(dpy);
	  switch(edge)
	  {
	    case EDGE_NORTH: 
              d=y >= displayHeight-edge_width;
              y = edge_width;
              break;
	    case EDGE_SOUTH:
              d=y < edge_width;
              y = displayHeight - edge_width -1;
              break;
	    case EDGE_EAST:
              d=x < edge_width;
              x = displayWidth -  edge_width -1;
              break;
	    case EDGE_WEST:
              d=x >= displayWidth-edge_width;
              x = edge_width;
              break;
	  }
        } else {
            /*
             * Different screen. Depending on where the pointer ended up,
             * we warp to either our default screen or the new screen.
             * 
             * If the pointer is "near" the edge we're watching, then the
             * user moved the pointer off the _opposite_ side of the PC
             * screen, and the pointer should reappear on the original screen.
             * 
             * If not, then the pointer was moved off some other edge - and
             * we just release the pointer.
             * 
             * In either case, the pointer's location - relative to the
             * screen - is not moved.
             * 
             * - GRM
             */
          warpWindow = ev->xmotion.root;
          x = ev->xmotion.x_root;
          y = ev->xmotion.y_root;
	  switch(edge)
	  {
	    case EDGE_NORTH:
              d=ev->xmotion.y_root < displayHeight / 2;
              break;
	    case EDGE_SOUTH:
              d=ev->xmotion.y_root > displayHeight / 2;
              break;
	    case EDGE_EAST:
              d=ev->xmotion.x_root > displayWidth / 2;
              break;
	    case EDGE_WEST:
              d=ev->xmotion.x_root < displayWidth / 2;
              break;
	  }
          if (d) {
            warpWindow = DefaultRootWindow(dpy);
          }
          d = 1;
              
        }
          
	if(d)
	{
	  SendPointerEvent(restingx,restingy,0);
	  XWarpPointer(dpy,None, warpWindow, 0,0,0,0, x, y);
	  XUngrabKeyboard(dpy, CurrentTime);
	  XUngrabPointer(dpy, CurrentTime);
          mouseOnScreen = warpWindow == DefaultRootWindow(dpy);
	  XFlush(dpy);
	  
	  for (i = 255; i >= 0; i--)
	  {
	    if (modifierPressed[i]) {
	      if (!SendKeyEvent(XKeycodeToKeysym(dpy, i, 0), False))
		return False;
	      modifierPressed[i]=False;
	    }
	  }
	  
	  grabbed=0;
	}
	return i;
      }
      return 1;
      
    case ButtonPress:
    case ButtonRelease:
      if (ev->xbutton.button > 3 && emulate_wheel)
      {
	if (ev->xbutton.button == wheel_button_up)
	  ks = XK_Up;
	else
	  ks = XK_Down;

	if (ev->type == ButtonPress)
	  SendKeyEvent(ks, 1); /* keydown */
	else
	  SendKeyEvent(ks, 0); /* keyup */
	break;
      }

      if (ev->type == ButtonPress) {
	buttonMask = (((ev->xbutton.state & 0x1f00) >> 8) |
		      (1 << (ev->xbutton.button - 1)));
      } else {
	buttonMask = (((ev->xbutton.state & 0x1f00) >> 8) &
		      ~(1 << (ev->xbutton.button - 1)));
      }
      
      return SendPointerEvent(SCALEX(ev->xbutton.x_root),
			      SCALEY(ev->xbutton.y_root),
			      buttonMask);
	
    case KeyPress:
    case KeyRelease:
    {
      char keyname[256];
      keyname[0] = '\0';
      XLookupString(&ev->xkey, keyname, 256, &ks, NULL);
/*      fprintf(stderr,"Pressing %x (%c) name=%s  code=%d\n",ks,ks,keyname,ev->xkey.keycode); */

      if (IsModifierKey(ks)) {
	ks = XKeycodeToKeysym(dpy, ev->xkey.keycode, 0);
	modifierPressed[ev->xkey.keycode] = (ev->type == KeyPress);
      } else {
	/* This fixes the 'shift-tab' problem - Hubbe */
	switch(ks)
	{
#if XK_ISO_Left_Tab != 0x1000ff74
	  case 0x1000ff74: /* HP-UX */
#endif
	  case XK_ISO_Left_Tab: ks=XK_Tab; break;
	}
      }
/*      fprintf(stderr,"-------- %x (%c) name=%s\n",ks,ks,keyname); */

      return SendKeyEvent(ks, (ev->type == KeyPress));
    }
      
    case ClientMessage:
      if ((ev->xclient.message_type == wmProtocols) &&
	  (ev->xclient.data.l[0] == wmDeleteWindow))
	{
	    ShutdownX();
	    exit(0);
	}
	break;
    }

    return True;
}



/*
 * HandleRootEvent.
 */

static Bool HandleRootEvent(XEvent *ev)
{
  char *str;
  int len;
  
  Bool nowOnScreen;
  Bool grab;
  
  int x, y;
  
  switch (ev->type)
  {
    case EnterNotify:
    case LeaveNotify:
      /*
       * Ignore the event if it's due to leaving our window. This will
       * be after an ungrab.
       */
      if (ev->xcrossing.subwindow == topLevel &&
          !ev->xcrossing.same_screen) {
	break;
      }
      
      grab = 0;
      if(!grabbed)
      {
	nowOnScreen =  ev->xcrossing.same_screen;
	if (mouseOnScreen != nowOnScreen) {
	  /*
	   * Mouse has left, or entered, the screen. We must grab if
	   * the mouse is now near the edge we're watching.
	   * 
	   * If we do grab, the mouse coordinates are left alone.
	   * The test, however, depends on whether the mouse entered
	   * or left the screen.
	   * 
	   * - GRM
	   */
	  x = ev->xcrossing.x_root;
	  y = ev->xcrossing.x_root;
	  if (!nowOnScreen) {
	    x = enter_translate(EW,displayWidth,ev->xcrossing.x_root);
	    y = enter_translate(NS,displayHeight,ev->xcrossing.y_root);
	  }
	  switch(edge)
	  {
	    case EDGE_NORTH:
	      grab=y < displayHeight / 2;
	      break;
	    case EDGE_SOUTH:
	      grab=y > displayHeight / 2;
	      break;
	    case EDGE_EAST:
	      grab=x > displayWidth / 2;
	      break;
	    case EDGE_WEST:
	      grab=x < displayWidth / 2;
	      break;
	  }
	}
	mouseOnScreen = nowOnScreen;
      }
      
      /*
       * Do not grab if this is the result of an ungrab
       * or grab (caused by us, usually).
       * 
       * - GRM
       */
      if(grab && ev->xcrossing.mode == NotifyNormal)
      {
	grabit(ev);
      }
      break;
      
    case PropertyNotify:
      if (ev->xproperty.atom == XA_CUT_BUFFER0)
      {
	str = XFetchBytes(dpy, &len);
	if (str) {
#ifdef DEBUG
	  fprintf(stderr,"GOT CUT TEXT: %s\n",str);
#endif
	  if (!SendClientCutText(str, len))
	    return False;
	  XFree(str);
	}
      }
      break;
  }
  
  return True;
}

void handle_cut_text(char *str, size_t len)
{
  XWindowAttributes   attrs;
  
  if(client_selection_text)
    free((char *)client_selection_text);
  
  client_selection_text_length=len;
  client_selection_text = str;
  
  XGetWindowAttributes(dpy, RootWindow(dpy, 0), &attrs);
  XSelectInput(dpy, RootWindow(dpy, 0),
	       attrs.your_event_mask & ~PropertyChangeMask);
  XStoreBytes(dpy, str, len);
  XSetSelectionOwner(dpy, XA_PRIMARY, topLevel, CurrentTime);
  XSelectInput(dpy, RootWindow(dpy, 0), attrs.your_event_mask);
}



/*
 * AllXEventsPredicate is needed to make XCheckIfEvent return all events.
 */

Bool
AllXEventsPredicate(Display *dpy, XEvent *ev, char *arg)
{
  return True;
}



