# regexp.tcl
# $Id: regexp.tcl,v 1.2 1999/07/28 07:51:56 chris Exp $
# Regular expression handling...  in Tcl.  *hollow laugh*
#
# NOTE incompatibilities with standard ed:
#  Does not implement the \` , \' , \b , or \B escapes.  Implementing
#  these would be very hard.
#
#
# XED --- an X-enabled implementation of ed, the standard editor.
# Copyright (C) 1999  Chris Laas
# 
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
# 
# Chris Laas
# golem@mit.edu
# 22 Pratt Court
# Cohasset, MA 02025 USA


#################################################
# Command-line pattern extraction.

# parse-pattern extracts a regexp pattern from the line.
# linevar starts out as "/re/foo" and ends up as "foo";
# delimvar is set to "/";
# and the return value will be "re".
proc parse-pattern {linevar delimitervar} {
	upvar 1 $linevar line $delimitervar delimiter
	global last_regexp

	# Test if this is empty, in which case, use last regexp.
	if {$line == {}} {
		set delimiter {}
	} else {
		set delimiter [string index $line 0]
	}
	if {$line == {} || [string length $line] == 1
			|| [string index $line 0] == [string index $line 1]} {
		if {![info exists last_regexp]} { error "No previous pattern" }
		set line [string range $line 2 end]
		return $last_regexp
	}

	# Go ahead and parse out the regexp.
	set exp {}
	set delimiter [next-char]
	set exp {}
	set c [next-char]
	while {$c != $delimiter && $c != {}} {
		if {$c == "\["} {
			# Parse a character class.
			set c [next-char]
			if {$c == "^" || $c == "\]"} { set c [next-char] }
			while {$c != "\]"} {
				if {$c == {}} { error "Unbalanced brackets (\[\])" }
				if {$c == "\[" && [string match {[.:=]*} $line]} {
					# A subclass inclusion of the form [:FOO:].
					set d [next-char]
					set p [next-char]
					set c [next-char]
					while {$c != "\]" || $p != $d} {
						set p $c ; set c [next-char]
						if {$c == {}} { error "Unbalanced brackets (\[\])" }
					}
				}
				set c [next-char]
			}
		} elseif {$c == "\\"} {
			# Ignore the backquoted character.
			set c [next-char]
			if {$c == {}} { error "Trailing backslash (\\)" }
		}
		set c [next-char]
	}

	if {$c == $delimiter} {
		set exp [string range $exp 0 [expr [string length $exp]-2]]
	}
	set last_regexp $exp
	return $exp
}
proc next-char {} {
	upvar 1 line line exp exp
	set c [string index $line 0]
	set line [string range $line 1 end]
	set exp $exp$c
	return $c
}

proc parse-subst-template {linevar delimiter} {
	upvar 1 $linevar line
	global input_mode continuation last_subst_template

	if {$line == "%" || [string range $line 0 1] == "%$delimiter"} {
		if {![info exists last_subst_template]} {
			error "No previous substitution template"
		}
		set line [string range $line 2 end]
		return $last_subst_template
	}

	while {[regsub {(^|[^\\])\\$} $line "\\1\n" line]} {
		set input_mode "continuation $input_mode"
		vwait input_mode
		append line $continuation
	}

	set template {}
	while 1 {
		set bsindex [string first "\\" $line]
		set dindex  [string first $delimiter $line]
		if {$bsindex != -1 && $bsindex < $dindex} {
			append template [string range $line 0 [expr $bsindex+1]]
			set line [string range $line [expr $bsindex+2] end]
		} elseif {$dindex != -1} {
			append template [string range $line 0 [expr $dindex-1]]
			set line [string range $line $dindex end]
			break
		} else {
			append template $line
			set line {}
			break
		}
	}
	set last_subst_template $template
	return $template
}


#################################################
# Lower-level interface routines.

# Takes an ed-style regexp and returns a Tcl8.1-style regexp.
proc regexp-compile {re} {
	regsub -all {(.)\\\?} $re {\1\{0,1\}} re
	regsub -all {(.)\\\+} $re {\1\{1,\}} re
	return "(?b)$re"
}
