#include <stdlib.h>
#include <unistd.h>

#include "proc_xlcm_lcp3.h"
#include "lcp3.h"
#include "widgets_xlcm.h"
#include "netio.h"
#include "runtime.h"
#include "interface_xlcm.h"
#include "cfg.h"

gint proc_xlcm_lcp3_callback_id;
gint proc_xlcm_lcp3_refresh_id;
gint proc_xlcm_lcp3_cmdqueue_callback_id;
gboolean proc_xlcm_lcp3_init_ok = FALSE;

void proc_xlcm_lcp3_process_command(gpointer data, gint sock, GdkInputCondition condition)
{
	struct t_lcp3_cmd * cmd;
	int infosize;
	char * info;
	struct t_lcp3_info_auth * info_auth;
	struct t_lcp3_info_enckey * info_enckey;
	struct t_lcp3_info_ack * info_ack;
	struct t_lcp3_info_line * info_line;
	struct t_lcp3_info_tput * info_tput;
	struct t_lcp3_info_status * info_status;
	struct t_lcp3_info_isdn * info_isdn;
	// read from socket:
	if ( !netio_handle_data(sock, &cmd) )
	{
		g_print("netio_handle_data() failed!\n");
		return;
	}
#ifdef DEBUG
	g_print("got cmd %d\n", cmd->cmd);
#endif
	// load pointers
	infosize = get_lcp3_pack_size(cmd->cmd) - sizeof(struct t_lcp3_cmd);
	if ( infosize )
		info = ((char*)cmd) + sizeof(struct t_lcp3_cmd);
	else
		info = NULL;
	info_auth = (void*)info;
	info_enckey = (void*)info;
	info_ack = (void*)info;
	info_line = (void*)info;
	info_tput = (void*)info;
	info_status = (void*)info;
	info_isdn = (void*)info;
	// process status information
	if ( cmd->info_is_valid )
	{
		if ( (cmd->client_stat == CLT_OFFLINE) && (runtime->client_status == CLT_NOAUTH) )
		{
			g_print("authentication successful\n");
			cmd_send(CMD3_LINESREQ, NULL, 0);
		}
		runtime->client_status = cmd->client_stat;
		if ( runtime->selected_line && (runtime->selected_line->num == cmd->line_id)
				&& (runtime->selected_line->status != cmd->line_stat) )
		{
			runtime->selected_line->status = cmd->line_stat;
			switch ( cmd->line_stat )
			{
				case CST_DOWN:
				case CST_DIALING:
				case CST_CLOSING:
				case CST_CLOSE_ERROR:
				case CST_LOCKED_DOWN:
					if ( wdg_activity->realized )
						gtk_widget_destroy(wnd_activity);
					break;
				case CST_UP_SERVER:
				case CST_UP_USER:
				case CST_LOCKED_UP_CLT:
				case CST_LOCKED_UP_TIMED:
					if ( !wdg_activity->realized )
					{
						wnd_activity = create_wnd_activity ();
						gtk_widget_show (wnd_activity);
					}
					break;
			}
		}
	}
	// process command
	switch ( cmd->cmd )
	{
		case CMD3_AUTHBAD:
			proc_xlcm_lcp3_cleanup();
			break;
		case CMD3_CLIENTSTAT:
			break;
		case CMD3_LINE:
			// got info_line --> append the line to our GSList * lines
			{
				struct tline * newline;
				newline = g_malloc(sizeof(struct tline));
				if ( !newline )
				{
					g_print("unable to load line!\n");
					cmd_send(CMD3_UNREG, NULL, 0);
					break;
				}
				memset(newline, 0, sizeof(struct tline));
				newline->ahup = g_malloc(sizeof(struct tautohup));
				if ( !newline->ahup )
				{
					g_print("unable to load autohangup info...\n");
					cmd_send(CMD3_UNREG, NULL, 0);
					break;
				}
				if ( info_line->line_id != config->default_line )
				{
					runtime_read_autohup(newline->ahup);
					g_free(newline->ahup);
					g_free(newline);
					break;
				}
				runtime_read_autohup(newline->ahup);
				newline->num = info_line->line_id;
				newline->name = g_strdup(info_line->linename);
				newline->ahup->cap_up = info_line->capability_baud_up;
				newline->ahup->cap_down = info_line->capability_baud_down;
				newline->ahup->max_up = newline->ahup->cap_up / 8;
				newline->ahup->max_down = newline->ahup->cap_down / 8;
				lines = g_slist_append(lines, newline);
				// selected it if it's the default line
				newline->status = -1;
				runtime->selected_line = newline;
				netio_set_line_id(newline->num);
				cmd_send(CMD3_LINESEL, NULL, 0);
				g_print("selected line #%d '%s'\n", newline->num, newline->name);
			}
			break;
		case CMD3_STATUS:
			// ignore
			break;
		case CMD3_INVALID:
			g_print("command %d seems to be invalid\n", info_ack->ackcmd);
			break;
		case CMD3_ENCINFO:
			// only plaintext supported until now...
			{
				struct t_lcp3_info_auth * iauth = g_malloc(sizeof(struct t_lcp3_info_auth));
				if ( !iauth )
				{
					g_print("unable to g_malloc() authinfo.\n");
					cmd_send(CMD3_UNREG, NULL, 0);
					break;
				}
				*(iauth->name) = 0;
				*(iauth->pass) = 0;
				iauth->enctype = LCP3_ENCTYPE_NONE;
				if ( info_enckey->enctype != LCP3_ENCTYPE_OPEN )
				{
					strncpy(iauth->name,config->user_name,LCP3_MAX_NAME_LEN);
					strncpy(iauth->pass,config->user_passwd,LCP3_MAX_PWD_LEN);
				}
				cmd_send(CMD3_AUTHREQ, iauth, sizeof(struct t_lcp3_info_auth));
			}
			break;
		case CMD3_NOACCESS:
			g_print("we don't have access to command %d\n", info_ack->ackcmd);
			break;
		case CMD3_CHANFAIL:
			break;
		case CMD3_ACK:
			break;
		case CMD3_UNREG:
			proc_xlcm_lcp3_cleanup();
			break;
		case CBR3_TPUT:
			{
				struct tline * line = runtime_get_line(cmd->line_id);
				if ( !line )
				{
					g_print("default line #%d not available!\n", cmd->line_id);
					break;
				}
				line->up_short = info_tput->ups;
				line->up_long = info_tput->upl;
				line->up_total = info_tput->upt / 1024;
				line->down_short = info_tput->dns;
				line->down_long = info_tput->dnl;
				line->down_total = info_tput->dnt / 1024;
				line->client_time = info_tput->client;
				line->server_time = info_tput->line;
				if ( wdg_activity->realized )
				{
					float up = 0, down = 0;
					if ( (float)line->ahup->max_up > 0 )
						up = (float)line->up_short / ((float)line->ahup->max_up);
					if ( (float)line->ahup->max_down > 0 )
						down = (float)line->down_short / ((float)line->ahup->max_down);
					if ( up > 1.0 ) up = 1.0;
					else if ( up < 0.0 ) up = 0.0;
					if ( down > 1.0 ) down = 1.0;
					else if ( down < 0.0 ) down = 0.0;
					gtk_progress_bar_update(wdg_activity->prg_up, up);
					gtk_progress_bar_update(wdg_activity->prg_down, down);
				}
			}
			break;
		case CBR3_LINESTAT:
			// already handled
			break;
		case CBR3_SRVDEATH:
			proc_xlcm_lcp3_cleanup();
			break;
		case CBR3_ISDNACT:
			{
				struct tmsg * msg = g_malloc(sizeof(struct tmsg));
				if ( !msg ) break;
				msg->row[0] = g_strdup(info_isdn->time);
				msg->row[1] = g_strdup(info_isdn->caller);
				wdg_messages->msgs = g_list_append(wdg_messages->msgs, msg);
				if ( wdg_messages->realized )
					gtk_clist_append(wdg_messages->cl_msg_messages, msg->row);
			}
			break;
		default:
			g_print("Received invalid command: %d\n", cmd->cmd);
	}
	if ( cmd ) free(cmd);
}

gboolean proc_xlcm_lcp3_refresh_callback(gpointer data)
{
	switch ( runtime->client_status )
	{
		case CLT_DISCON:
		case CLT_NOAUTH:
			return TRUE;
		default:
				cmd_send(CMD3_REFRESH, NULL, 0);
	}
	return TRUE;
}

gboolean proc_xlcm_lcp3_cmd_queue_callback(gpointer ptr)
{
	time_t now = time(NULL);
	if ( now - runtime->watchdog > LCP3_CLT_TIMEOUT )
	{
		g_print("Timeout detected. Reconnecting...\n");
		runtime->watchdog_barked = TRUE;
		proc_xlcm_lcp3_cleanup();
		proc_xlcm_lcp3_init();
		cmd_send(CMD3_PUBKEYREQ, NULL, 0);
	}
	else
	{
		runtime->watchdog = now;
		netio_proc_queue();
	}
	return TRUE;
}

int proc_xlcm_lcp3_init()
{
	g_print("entering proc_xlcm_lcp3_init()\n");
	if ( proc_xlcm_lcp3_init_ok )
	{
		g_print("proc_lcp3_init(): already initialized!\n");
		return 1;
	}
	runtime->client_status = CLT_DISCON;
	if ( (runtime->sock = netio_initiate_tcp_connection(config->server_hostname, config->server_port)) < 0 )
	{
		g_print("proc_lcp3_init(): didn't get a tcp connection...\n");
		return 0;
	}
	proc_xlcm_lcp3_callback_id = gdk_input_add(runtime->sock, GDK_INPUT_READ, proc_xlcm_lcp3_process_command, NULL);
	proc_xlcm_lcp3_refresh_id = gtk_timeout_add((LCP3_CLT_TIMEOUT-5)*1000, proc_xlcm_lcp3_refresh_callback, NULL);
	proc_xlcm_lcp3_cmdqueue_callback_id = gtk_timeout_add(6000, proc_xlcm_lcp3_cmd_queue_callback, NULL);
	runtime->watchdog_barked = FALSE;
	runtime->watchdog = time(NULL);
	proc_xlcm_lcp3_init_ok = TRUE;
	return 1;
}

int proc_xlcm_lcp3_cleanup()
{
	runtime->client_status = CLT_DISCON;
	proc_xlcm_lcp3_init_ok = FALSE;
	gtk_timeout_remove(proc_xlcm_lcp3_refresh_id);
	gtk_timeout_remove(proc_xlcm_lcp3_cmdqueue_callback_id);
	gdk_input_remove(proc_xlcm_lcp3_callback_id);
	close(runtime->sock);
//	runtime_save();
	runtime_clear_lines();
	if ( !runtime->watchdog_barked )
		gtk_main_quit();
	g_print("proc_xlcm_lcp3_cleanup() done.\n");
	return 1;
}

inline int cmd_send(int cmd, void * info, int infosize)
{
	char res;
#ifdef DEBUG
	g_print("sending %d\n", cmd);
#endif
	res = netio_cmd_queue(cmd, info, infosize);
	if ( info ) g_free(info);
	return(res);
}
