#include <string.h>
#include <gtk/gtk.h>
#include <xmms/configfile.h>

#include "config.h"
#include "infconfig.h"
#include "gettext.h"



typedef struct t_config {
    int x, y;
    int xres;
    int	yres;
    int	sres;
    int teff;
    int	tcol;
} t_config;



static t_config config = {-1, -1, 0, 0, 0, 0, 0};
static GtkWidget *configure_win = NULL;


static void x_value_changed(GtkWidget *widget, gpointer data);
static void y_value_changed(GtkWidget *widget, gpointer data);
static void s_value_changed(GtkWidget *widget, gpointer data);
static void teff_value_changed(GtkWidget *widget, gpointer data);
static void tcol_value_changed(GtkWidget *widget, gpointer data);
static gboolean check_cfg_version (ConfigFile *f);
static void ok_clicked (GtkWidget *w, GtkWidget *window);
static void cancel_clicked (GtkWidget *w, GtkWidget *window);


/*
 *
 * Public functions
 *
 */


void config_plugin_load_prefs (void)
{
    int value;
    ConfigFile *f;

    if ((f = xmms_cfg_open_default_file())) {
        if (check_cfg_version (f)) {
            g_message("version Ok");
            xmms_cfg_read_int (f, "infinity", "xres", &value);
            config_set_xres (value);
            xmms_cfg_read_int (f, "infinity", "yres", &value);
            config_set_yres (value);
            xmms_cfg_read_int (f, "infinity", "sres", &value);
            config_set_sres (value);
            xmms_cfg_read_int (f, "infinity", "teff", &value);
            config_set_teff (value);
            xmms_cfg_read_int (f, "infinity", "tcol", &value);
            config_set_tcol (value);
            xmms_cfg_read_int (f, "infinity", "xorig", &value);
            config_set_x (value);
            xmms_cfg_read_int (f, "infinity", "yorig", &value);
            config_set_y (value);
        } else {
            config_set_default_values();
            /*
             * Update to current version
             */
            config_plugin_save_prefs();
        }
        xmms_cfg_free (f);
    } else {
        config_set_default_values ();
    }
}


void config_plugin_save_prefs (void)
{
    char *name;
    int orig;
    ConfigFile *f;
    
    name = g_strconcat (g_get_home_dir(), "/.xmms/config", 0);
    if	(!(f=xmms_cfg_open_file (name)))
        f = xmms_cfg_new();
    xmms_cfg_write_string (f, "infinity", "version", PACKAGE_VERSION);
    if (!config_is_initialized ())
	config_set_default_values ();
    xmms_cfg_write_int (f, "infinity", "xres", config_get_xres());
    xmms_cfg_write_int (f, "infinity", "yres", config_get_yres());
    xmms_cfg_write_int (f, "infinity", "sres", config_get_sres());
    xmms_cfg_write_int (f, "infinity", "teff", config_get_teff());
    xmms_cfg_write_int (f, "infinity", "tcol", config_get_tcol());
    orig = (config.x < 0 ? -1 : config.x);
    xmms_cfg_write_int (f, "infinity", "xorig", orig);
    orig = (config.y < 0 ? -1 : config.y);
    xmms_cfg_write_int (f, "infinity", "yorig", orig);
    xmms_cfg_write_file (f, name);
    xmms_cfg_free (f);
    g_free (name);
}


void config_set_default_values (void)
{
    config_set_x (20);
    config_set_y (20);
    config_set_xres (380);
    config_set_yres (240);
    config_set_sres (1);
    config_set_teff (100);
    config_set_tcol (100);
}


gboolean config_is_initialized (void)
{
    return ((config.x > -1) && (config.y > -1) &&
        (config.xres != 0) && (config.yres != 0) && 
        (config.sres != 0) && (config.teff != 0) &&
        (config.tcol != 0));
}


void config_plugin_config_window (void)
{
    GtkWidget *buttonbox,*ok,*cancel;
    GtkObject *xadj,*yadj,*sadj,*teadj,*tcadj;
    GtkWidget *label,*xhscale,*yhscale,*shscale;
    GtkTooltips *tooltips;
    static GtkWidget *vbox,*vbox2;
    static GtkWidget *hbox,*frame;	

#ifdef HAVE_SETLOCALE
    setlocale (LC_MESSAGES, "");
    bindtextdomain (PACKAGE, LOCALEDIR);
    textdomain (PACKAGE);
#endif

    if (configure_win)
        return;

    config_plugin_load_prefs ();
    if (!config_is_initialized ())
        config_set_default_values ();

    configure_win = gtk_window_new (GTK_WINDOW_DIALOG);
    gtk_widget_set_usize (configure_win, 558, 280);

    gtk_container_set_border_width (GTK_CONTAINER(configure_win), 10);
    gtk_window_set_title (GTK_WINDOW(configure_win), _("Infinity Configuration"));
    gtk_window_set_policy (GTK_WINDOW(configure_win), FALSE, FALSE, FALSE);
    gtk_window_set_position (GTK_WINDOW(configure_win), GTK_WIN_POS_MOUSE);

    gtk_signal_connect (GTK_OBJECT(configure_win), "destroy", 
                        GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                        &configure_win);

    vbox = gtk_vbox_new (FALSE, 4);
    gtk_container_add (GTK_CONTAINER(configure_win), vbox);

    frame = gtk_frame_new (_("Display:"));
    gtk_box_pack_start (GTK_BOX(vbox), frame, TRUE, TRUE, 4);

    vbox2 = gtk_vbox_new (FALSE, 4);
    gtk_container_add (GTK_CONTAINER(frame), vbox2);
	
    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox2), hbox, TRUE, TRUE, 4);

    label = gtk_label_new (_("X Resolution"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    xadj = gtk_adjustment_new (config_get_xres(), 0, 800, 160, 320, 0);
    xhscale = gtk_hscale_new (GTK_ADJUSTMENT(xadj));
    gtk_scale_set_draw_value (GTK_SCALE(xhscale), TRUE);
    gtk_scale_set_value_pos (GTK_SCALE(xhscale), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(xhscale), 0);
    gtk_widget_show (xhscale);
    gtk_box_pack_start (GTK_BOX(hbox), xhscale, TRUE, TRUE, 4);

    gtk_widget_show (hbox);

    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox2), hbox, TRUE, TRUE, 4);

    label = gtk_label_new (_("Y Resolution"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    yadj = gtk_adjustment_new (config_get_yres(), 0, 600, 120, 240, 0);
    yhscale = gtk_hscale_new (GTK_ADJUSTMENT(yadj));
    gtk_scale_set_draw_value (GTK_SCALE(yhscale), TRUE);
    gtk_scale_set_value_pos (GTK_SCALE(yhscale), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(yhscale), 0);
    gtk_widget_show (yhscale);
    gtk_box_pack_start (GTK_BOX(hbox), yhscale, TRUE, TRUE, 4);
	
    gtk_widget_show (hbox);
	    
    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox2), hbox, TRUE, TRUE, 4);
	
    label = gtk_label_new(_("Scale Factor"));
    gtk_widget_show (label);		
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE,FALSE, 4);

    sadj = gtk_adjustment_new (config_get_sres(), 1, 2, 10, 20, 0);
    shscale = gtk_hscale_new (GTK_ADJUSTMENT(sadj));
    gtk_scale_set_draw_value (GTK_SCALE(shscale), TRUE);
    gtk_scale_set_value_pos (GTK_SCALE(shscale), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(shscale), 0);
    gtk_widget_show (shscale);
    gtk_box_pack_start (GTK_BOX(hbox), shscale, TRUE, TRUE, 4);

    tooltips = gtk_tooltips_new ();
    gtk_tooltips_set_tip (tooltips, shscale,
                          _("Change only if the system doesn't have "
                              "enough power to show the screen at "
                              "a reasonably speed/size relation."), NULL);

    gtk_signal_connect (GTK_OBJECT(xadj), "value-changed", 
                       GTK_SIGNAL_FUNC(x_value_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(yadj), "value-changed", 
                       GTK_SIGNAL_FUNC(y_value_changed), NULL);	
    gtk_signal_connect (GTK_OBJECT(sadj), "value-changed", 
                       GTK_SIGNAL_FUNC(s_value_changed), NULL);	
	
    gtk_widget_show (hbox);
    gtk_widget_show (vbox2);
    gtk_widget_show (frame);
	
    frame = gtk_frame_new (_("General:"));
    gtk_box_pack_start (GTK_BOX(vbox),frame,TRUE,TRUE,4);

    vbox2 = gtk_vbox_new (FALSE, 4);
    gtk_container_add (GTK_CONTAINER(frame),vbox2);
	
    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox2),hbox,TRUE,TRUE,4);

    label = gtk_label_new (_("Effect period:"));
    gtk_box_pack_start (GTK_BOX(hbox),label,FALSE,FALSE,4);
    gtk_widget_show (label);

    teadj = gtk_adjustment_new (config_get_teff(), 50, 400, 100, 100, 0);
    xhscale = gtk_hscale_new (GTK_ADJUSTMENT(teadj));
    gtk_scale_set_draw_value (GTK_SCALE(xhscale), TRUE);
    gtk_scale_set_value_pos (GTK_SCALE(xhscale), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(xhscale), 0);
    gtk_widget_show (xhscale);
    gtk_box_pack_start (GTK_BOX(hbox), xhscale, TRUE, TRUE, 4);

    gtk_tooltips_set_tip (tooltips, xhscale,
                          _("Time to change between an effect and the"
                            "next one, in milliseconds."), NULL);

    gtk_widget_show (hbox);

    hbox = gtk_hbox_new (FALSE, 4);
    gtk_box_pack_start (GTK_BOX(vbox2), hbox, TRUE, TRUE, 4);

    label = gtk_label_new (_("Palette period:"));
    gtk_box_pack_start (GTK_BOX(hbox), label, FALSE, FALSE, 4);
    gtk_widget_show (label);

    tcadj = gtk_adjustment_new (config_get_tcol(), 50, 400, 100, 100, 0);
    yhscale = gtk_hscale_new (GTK_ADJUSTMENT(tcadj));
    gtk_scale_set_draw_value (GTK_SCALE(yhscale), TRUE);
    gtk_scale_set_value_pos (GTK_SCALE(yhscale), GTK_POS_LEFT);
    gtk_scale_set_digits (GTK_SCALE(yhscale), 0);
    gtk_widget_show (yhscale);
    gtk_box_pack_start (GTK_BOX(hbox), yhscale, TRUE, TRUE, 4);

    gtk_tooltips_set_tip (tooltips, yhscale,
                          _("Time to change between a color and the"
                            "next one, in milliseconds."), NULL);

    gtk_signal_connect (GTK_OBJECT(teadj), "value-changed", 
                       GTK_SIGNAL_FUNC(teff_value_changed), NULL);
    gtk_signal_connect (GTK_OBJECT(tcadj), "value-changed", 
                       GTK_SIGNAL_FUNC(tcol_value_changed), NULL);	

    gtk_widget_show (hbox);
    gtk_widget_show(vbox2);
    gtk_widget_show(frame);

    buttonbox = gtk_hbutton_box_new ();
    gtk_box_pack_end (GTK_BOX(vbox), buttonbox, FALSE, FALSE, 8);
    gtk_hbutton_box_set_layout_default (GTK_BUTTONBOX_END);
    gtk_widget_show (buttonbox);
	
    ok = gtk_button_new_with_label (_("Ok"));
    GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
    gtk_box_pack_end (GTK_BOX(buttonbox), ok, FALSE, FALSE, 8);
    gtk_widget_show (ok);

    gtk_tooltips_set_tip (tooltips, ok,
                          _("For to toggle to fullscreen mode, select "
                            "the Infinity plugin window, and press "
                            "the Tab key. Pressing Tab again will return "
                            "to window mode."), NULL);

    cancel = gtk_button_new_with_label (_("Cancel"));
    GTK_WIDGET_SET_FLAGS (cancel, GTK_CAN_DEFAULT);
    gtk_box_pack_end (GTK_BOX(buttonbox), cancel, FALSE, FALSE, 8);
    gtk_widget_show (cancel);
	
    gtk_widget_show (vbox);

    gtk_window_set_default (GTK_WINDOW(configure_win), ok);
    gtk_signal_connect (GTK_OBJECT(cancel), "clicked", GTK_SIGNAL_FUNC(cancel_clicked), configure_win);
    gtk_signal_connect (GTK_OBJECT(ok), "clicked", GTK_SIGNAL_FUNC(ok_clicked), configure_win);

    gtk_tooltips_enable (tooltips);
	
    gtk_widget_show (configure_win);
}


void config_set_x (int value)
{
    config.x = value;
}


void config_set_y (int value)
{
    config.y = value;
}


void config_set_xres (int value)
{
    config.xres = value;
}


void config_set_yres (int value)
{
    config.yres = value;
}


void config_set_sres (int value)
{
    config.sres = value;
}


void config_set_teff (int value)
{
    config.teff = value;
}


void config_set_tcol (int value)
{
    config.tcol = value;
}


int config_get_x (void)
{
    return config.x;
}


int config_get_y (void)
{
    return config.y;
}


int config_get_xres (void)
{
    return config.xres;
}


int config_get_yres (void)
{
    return config.yres;
}


int config_get_sres (void)
{
    return config.sres;
}


int config_get_teff (void)
{
    return config.teff;
}


int config_get_tcol (void)
{
    return config.tcol;
}



/*
 *
 * Private functions
 *
 */



/*
 * Callbacks for config_plugin_config_window
 */
static void x_value_changed(GtkWidget *widget, gpointer data)
{
    config_set_xres ((int)(GTK_ADJUSTMENT(widget)->value));
}

static void y_value_changed(GtkWidget *widget, gpointer data)
{
    config_set_yres ((int)(GTK_ADJUSTMENT(widget)->value));
}

static void s_value_changed(GtkWidget *widget, gpointer data)
{
    config_set_sres ((int)(GTK_ADJUSTMENT(widget)->value));
}

static void teff_value_changed(GtkWidget *widget, gpointer data)
{
    config_set_teff ((int)(GTK_ADJUSTMENT(widget)->value));
}

static void tcol_value_changed(GtkWidget *widget, gpointer data)
{
    config_set_tcol ((int)(GTK_ADJUSTMENT(widget)->value));
}


static gboolean check_cfg_version (ConfigFile *f)
{
    char *vstr;

    if (xmms_cfg_read_string (f, "infinity", "version", &vstr))
	if (!strcmp (vstr, PACKAGE_VERSION))
	    return TRUE;
    return FALSE;
}

static void ok_clicked (GtkWidget *w, GtkWidget *window)
{
    config_plugin_save_prefs();
    gtk_widget_destroy(window);
    configure_win=NULL;
}

static void cancel_clicked (GtkWidget *w, GtkWidget *window)
{
    gtk_widget_destroy (window);
    configure_win = NULL;
}

