/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <config.h>

#include <qfileinfo.h>
#include <qdir.h>

#include <kprocess.h>
#include <klocale.h>
#include <kurl.h>

#ifdef HAVE_AMAROK
#include "amarokplayer.h"

AmarokPlayer::AmarokPlayer() {
  qDebug("xmms-kde: created amarokplayer interface");

  client = new DCOPClient();
  client->attach();
  QCString realAppId = client->registerAs("xmmskde");

  /*
   * test if amarok is up and running
   */
  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);

  amarokId = "amarok";
 
  if (!client->call(amarokId, "player", "isPlaying()",
		    data, replyType, replyData)) {
    running = false;

  } else {
    qDebug("xmms-kde: connected to amarok");
    running = true;
  }

  vol = 100; // not able to get the volume from amarok :-(
}

AmarokPlayer::~AmarokPlayer() {
  qDebug("xmms-kde: deleted amarokplayer interface");
}

void AmarokPlayer::startPlayer() {

  qDebug("xmms-kde: launching amarok");
  KShellProcess proc;
  proc << "amarok";
  proc.start(KShellProcess::DontCare);

}

void AmarokPlayer::exitPlayer() {
  qDebug("xmms-kde: killing amarok is not implemented");
}

void AmarokPlayer::previous() {
  sendNoParam(QString("prev()"));
}
void AmarokPlayer::play() {
  sendNoParam(QString("play()"));
}
void AmarokPlayer::pause() {
  sendNoParam(QString("playPause()"));
}
void AmarokPlayer::stop() {
  sendNoParam(QString("stop()"));
}
void AmarokPlayer::next() {
  sendNoParam(QString("next()"));
}

void AmarokPlayer::toggleRepeat() {
  //sendNoParam(QString("loop()"));
}

void AmarokPlayer::toggleShuffle() {

}

void AmarokPlayer::setVolume(int volume) {
  vol = volume;
  sendIntParam(QString("setVolume(int)"), volume);
}
void AmarokPlayer::setCurrentTime(int time) {

  sendIntParam(QString("seek(int)"), time / 1000);
}

int AmarokPlayer::getVolume() {
  return vol;
}

QString AmarokPlayer::getTitle() {


  bool state = callGetBool(QString("isPlaying()"));

  if (!state) {
    return QString(i18n("Amarok stopped"));
  }

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);

  QString result("");
  
  if (!client->call(amarokId, "player", "nowPlaying()",
 		    data, replyType, replyData)) {
    running = false;
  } else {
    running = true;
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "QString") {
      reply >> result;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply: nowPlaying()");
  }
  return result;
}

int AmarokPlayer::getCurrentTimePercent() {
  return (int) (((float) callGetInt(QString("trackCurrentTime()"))) /
		((float) callGetInt(QString("trackTotalTime()"))) * 100);
		
}

int AmarokPlayer::getCurrentTime() {
  // int length = callGetInt(QString("length()"));

  int time = callGetInt(QString("trackCurrentTime()")) * 1000;

  return time<0?0:time;
}

int AmarokPlayer::getTrackTime() {
  int time =  callGetInt(QString("trackTotalTime()")) * 1000;
  
  return time<0?0:time;
}
bool AmarokPlayer::getRepeat() {
    return false;
}
bool AmarokPlayer::getShuffle() {
  return false;
}
bool AmarokPlayer::isPlaying() {
  return callGetBool(QString("isPlaying()"));
}
bool AmarokPlayer::isPlayerRunning() {
  return running;
}

void AmarokPlayer::openFileSelector() {
  //sendNoParam(QString("toggleListView()"));
}

void AmarokPlayer::playlistClear() {

  //sendNoParam(QString("clear()"));
}

void AmarokPlayer::playlistAdd(QString files) {

  qDebug("playlistAdd(QString)");

  QStringList fileList = QStringList::split("\r\n", files);

  playlistAdd(fileList);
}


void AmarokPlayer::playlistAdd(QStringList files) {

  qDebug("playlistAdd(QStringList)");

  for (QStringList::Iterator it = files.begin();
       it != files.end(); ++it) {
    qDebug("'" + *it + "'");
  }

  // add the files to the amarok playlist
  KURL::List fileList(files);

  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);
  
  arg << fileList;
 
  if (!client->send(amarokId, "player", "addMediaList(KURL::List)",
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP::addMediaList(KURL::List)");
  } else {
    running = true;
  } 
  
}


void AmarokPlayer::sendNoParam(QString command) {

  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);

  if (!client->send(amarokId, "player", command.latin1(),
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP.");
  } else {
    running = true;
  }
}

void AmarokPlayer::sendIntParam(QString command, int param) {

  QByteArray data;
  QDataStream arg(data, IO_WriteOnly);

  arg << param;

  if (!client->send(amarokId, "player", command.latin1(),
		    data)) {
    running = false;
    qDebug("xmms-kde: there was some error using DCOP.");
  } else {
    running = true;
  }
}

int AmarokPlayer::callGetInt(QString command) {

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  
  int result;

  if (!client->call(amarokId, "player", command.latin1(),
 		    data, replyType, replyData))
    running = false;
  else {
    running = true;
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "int") {

      reply >> result;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply");
  }
  return result;
}

int AmarokPlayer::callGetBool(QString command) {

  QByteArray data, replyData;
  QCString replyType;
  QDataStream arg(data, IO_WriteOnly);
  
  bool result;

  if (!client->call(amarokId, "player", command.latin1(),
 		    data, replyType, replyData))
    running = false;
  else {
    running = true;
    QDataStream reply(replyData, IO_ReadOnly);
    if (replyType == "bool") {

      reply >> result;
    } else
      qDebug("xmms-kde: unexpected type of dcop reply");
  }
  return result;
}


#endif
