/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */
#include <config.h>

#include "osdframe.moc"
#include "osdframe.h"

/*
 * TODO: select font for osd
 */

OSDFrame::OSDFrame(KConfig *conf) :
  QFrame(NULL, 0, 
	 WType_TopLevel| WStyle_NoBorder | WX11BypassWM | WStyle_StaysOnTop) {

  config = conf;

  readConfig();

  loop = false;
  frameRoot = new KRootPixmap(this);

  resize(320, 20);

  frameRoot->setFadeEffect(transparency, bgColor);

  frameRoot->start();

  paintTimer = new QTimer(this);
  connect(paintTimer, SIGNAL(timeout()), SLOT(paint()));
  
  effect = FADE_IN;

  title = QString(i18n("this is the song title"));

  QFontMetrics fm(titleFont);

  width = fm.width(title) + 20;
  height = fm.height();

  resize(width, height + 2);

  if (enable && effect == OSD_TEST) {
    paintTimer->start(50);
  }

  move(x, y);
}

OSDFrame::~OSDFrame() {

  paintTimer->stop();
  frameRoot->stop();
}

void OSDFrame::startSequence(QString t) {
  
  if (enable) {
    raise();
    QFontMetrics fm(titleFont);
    
    width = fm.width(t) + 20;
    height = fm.height();
    
    resize(width, height + 2);
    
    effect = FADE_IN;
    title = t;
    
    i = 0;
    wait = 0;
    
    paintTimer->start(30);
  }
}

void OSDFrame::paintEvent(QPaintEvent *e) {


}

void OSDFrame::paint() {

  int xPos = 0;
  int yPos = 0;

  int direction = 0;

  if (effect == FADE_IN) {

    direction = source;
    if (i < M_PI / 2) i += 0.05;
    if (i >= M_PI / 2) {
      i = M_PI / 2;
      effect = FADE_WAIT;
    }
  } else if (effect == FADE_OUT) {

    direction = destination;
    if (i >= 0) i -= 0.05;
    if (i < 0) {
      i = 0;
      if (!loop)
	effect = FADE_STOP;
      else
	effect = FADE_IN;
    }
  }

  if (direction == LEFT) {

    xPos = (int) (sin(i) * width + 10 - width);
    yPos = (int) height - 5;
  
  } else if (direction == RIGHT) {

    xPos = (int) (width + 10 - sin(i) * width);
    yPos = (int) height - 5;

  }  else if (direction == TOP) {

    xPos = (int) 10;
    yPos = (int) (sin(i) * height) - 5;
    
  }  else if (direction == BOTTOM) {

    xPos = (int) 10;
    yPos = (int) 2 * height - 5 - (int) (sin(i) * height);
  }

  if (effect == FADE_WAIT) {

    wait++;
    if (wait > 80) {
      wait = 0;
      effect = FADE_OUT;
    }

    xPos = (int) (10);
    yPos = (int) height - 5;
  }

  if (effect == OSD_TEST) {
    
    xPos = 10;
    yPos = (int) height - 5;
  }

  QPixmap pmt(width, height);
  pmt.fill(this, width, height);

  QPainter pt;
  pt.begin(&pmt);
  pt.setFont(titleFont);

  pt.setPen(fgColor);
  pt.drawText(xPos, yPos, title);

  QPainter hvpainter(this);
  hvpainter.drawPixmap(0, 0, pmt);

  if (!isVisible())
    show();

  if (effect == FADE_STOP) {
    paintTimer->stop();
    hide();
  }
}


void OSDFrame::readConfig() {

  config->setGroup("OSD");
  //size = config->readNumEntry("size", 20);
  x = config->readNumEntry("xpos", 0);
  y = config->readNumEntry("ypos", 0);
  enable = config->readBoolEntry("enable", false);
  bgColor = config->readColorEntry("bgColor");
  fgColor = config->readColorEntry("fgColor");
  transparency = (float) config->readDoubleNumEntry("transparency");
  source = config->readNumEntry("fadeIn", 0);
  destination = config->readNumEntry("fadeOut", 0);

  titleFont = QFont("Helvetica", 20);
  titleFont = config->readFontEntry("font", &titleFont);
}

void OSDFrame::writeConfig() {

  config->setGroup("OSD");
  config->writeEntry("enable", enable);
  config->writeEntry("font", titleFont);
  config->writeEntry("xpos", x);
  config->writeEntry("ypos", y);
  config->writeEntry("fgColor", fgColor);
  config->writeEntry("bgColor", bgColor);
  config->writeEntry("transparency", transparency);
  config->writeEntry("fadeIn", source);
  config->writeEntry("fadeOut", destination);
}


QWidget *OSDFrame::getConfigurationWidget(QWidget *parent) {

  confWidget = new QTabWidget(parent);
  QWidget *osdWidget = new QWidget(confWidget);  
  QVBoxLayout *vbox = new QVBoxLayout(osdWidget);
  
  osdBox = new QCheckBox(i18n("enable On Screen Display"), osdWidget);
  osdBox->setChecked(enable);
  connect(osdBox, SIGNAL(clicked()), this, SLOT(configChanged()));
  vbox->addWidget(osdBox);

  QHBox *box = new QHBox(osdWidget);
  QLabel *osdSize = new QLabel(i18n("Font: "), box);
  QString label;
  label.sprintf("%s, %d pt", titleFont.family().latin1(), 
		titleFont.pointSize());
		
  osdFontButton = new QPushButton(label, box);
  //osdSizeBox = new QSpinBox(10, 100, 1, box);
  //osdSizeBox->setValue(size);
  vbox->addWidget(box);
  
  box = new QHBox(osdWidget);
  QLabel *xpos = new QLabel(i18n("x Position: "), box);
  osdXBox = new QSpinBox(0, 10000, 1, box);
  osdXBox->setValue(x);
  vbox->addWidget(box);
  
  box = new QHBox(osdWidget);
  QLabel *ypos = new QLabel(i18n("y Position: "), box);
  osdYBox = new QSpinBox(0, 10000, 1, box);
  osdYBox->setValue(y);
  vbox->addWidget(box);
  
  box = new QHBox(osdWidget);
  QLabel *fgcolor = new QLabel(i18n("Text Color: "), box);
  osdFGColorButton = new KColorButton(fgColor, box);
  vbox->addWidget(box);
  
  box = new QHBox(osdWidget);
  QLabel *bgcolor = new QLabel(i18n("Background Color: "), box);
  osdBGColorButton = new KColorButton(bgColor, box);
  vbox->addWidget(box);
  
  box = new QHBox(osdWidget);
  QLabel *trans = new QLabel(i18n("Transparency: "), box);
  osdTBox = new QSpinBox(0, 100, 1, box);
  osdTBox->setValue((int) (100 - transparency * 100));
  vbox->addWidget(box);
  
  QWidget *fadeWidget = new QWidget(confWidget);
  vbox = new QVBoxLayout(fadeWidget);
  
  QButtonGroup *fadeInGroup = new QButtonGroup(3, Qt::Horizontal, fadeWidget);
  vbox->addWidget(fadeInGroup);
  QRadioButton *btn1 = new QRadioButton(i18n("Left"), fadeInGroup); // 0
  QRadioButton *btn2 = new QRadioButton(i18n("Right"), fadeInGroup); // 1
  QRadioButton *btn3 = new QRadioButton(i18n("Top"), fadeInGroup); // 2
  QRadioButton *btn4 = new QRadioButton(i18n("Bottom"), fadeInGroup); // 3
  fadeInGroup->setButton(source);
  fadeInGroup->setTitle(i18n("OSD fade in"));

  connect(fadeInGroup, SIGNAL(clicked(int)), this,
	  SLOT(fadeInChanged(int)));

  QButtonGroup *fadeOutGroup = new QButtonGroup(3, Qt::Horizontal, fadeWidget);
  vbox->addWidget(fadeOutGroup);
  btn1 = new QRadioButton(i18n("Left"), fadeOutGroup); // 0
  btn2 = new QRadioButton(i18n("Right"), fadeOutGroup); // 1
  btn3 = new QRadioButton(i18n("Top"), fadeOutGroup); // 2
  btn4 = new QRadioButton(i18n("Bottom"), fadeOutGroup); // 3
  fadeOutGroup->setButton(destination);
  fadeOutGroup->setTitle(i18n("OSD fade out"));

  connect(fadeOutGroup, SIGNAL(clicked(int)), this, 
	  SLOT(fadeOutChanged(int)));
    
  connect(osdXBox, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
  connect(osdYBox, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
  //connect(osdSizeBox, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
  connect(osdFontButton, SIGNAL(clicked()), this, SLOT(fontChanged()));
  connect(osdTBox, SIGNAL(valueChanged(int)), this, SLOT(configChanged()));
    
  connect(osdFGColorButton, SIGNAL(clicked()), this, SLOT(configChanged()));
  connect(osdBGColorButton, SIGNAL(clicked()), this, SLOT(configChanged()));

  confWidget->addTab(osdWidget, i18n("OSD"));
  confWidget->addTab(fadeWidget, i18n("Fading"));
  
  qDebug(".");
  return confWidget;
}


void OSDFrame::configChanged() {

  startDemo();
  
  enable = osdBox->isChecked();

  //size = osdSizeBox->value();
  //titleFont = QFont("Helvetica", size);
  
  fgColor = osdFGColorButton->color();
  bgColor = osdBGColorButton->color();

  transparency = 1 - ((float) osdTBox->value()) / 100;
  frameRoot->setFadeEffect(transparency, bgColor);

  x = osdXBox->value();
  y = osdYBox->value();

  move(x, y);
}

void OSDFrame::fadeInChanged(int index) {

  source = index;
  configChanged();
}

void OSDFrame::fadeOutChanged(int index) {

  destination = index;
  configChanged();
}

void OSDFrame::fontChanged() {

  if (enable) {
    startDemo();
    bool ok;
    
    titleFont = QFontDialog::getFont(&ok, titleFont, NULL);
    
    if (ok) {
      QString label;
      label.sprintf("%s, %d pt", titleFont.family().latin1(), 
		    titleFont.pointSize());
      
      osdFontButton->setText(label);
    }
  }
}


void OSDFrame::startDemo() {

  title = QString(i18n("Test title"));
  QFontMetrics fm(titleFont);
  
  width = fm.width(title) + 20;
  height = fm.height();
  
  resize(width, height + 2);
    
  if (enable) {
   
    if (effect == FADE_STOP || effect == OSD_TEST)
      effect = FADE_IN;
    
    paintTimer->start(30);
    
    loop = true;
  }
}

void OSDFrame::stopDemo() {

  effect = FADE_STOP;
  loop = false;
}
