/*
 * Copyright (C) 2003 the xmms-kde team
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 * TODO: - still (rare) crashes when updating db and dialog is closed (???)
 *       - single inserts fail when querying at the same time (???)
 *         happens also when queried from command line so this seems to be
 *         an sqlite bug
 *       - database locking on nfs doesn't work (sqlite/nfs bug)
 *       - failure to insert non-ascii characters in pathname (locale problem?)
 */

#include <config.h>

#include <qstringlist.h>
#include <qdir.h>
#include <qlayout.h>
#include <qhbox.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qhbuttongroup.h>
#include <qpixmap.h>
#include <qlineedit.h>
#include <qtimer.h>

#include <kfilemetainfo.h>
#include <kwin.h>
#include <kapplication.h>

#ifdef HAVE_ID3LIB
#include <id3/tag.h>
#endif

#ifdef HAVE_VORBIS
#include <ogg/ogg.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>
#endif

#include <configdialog.h>

#include "xmms-kdedb.h"
#include "xmms-kdedb.moc"

struct eqstr {
  bool operator()(const char* s1, const char* s2) const {
    return strcmp(s1, s2) == 0;
 }
};

static int callback(void *NotUsed, int argc, char **argv, char **azColName) {

  int i;
  for(i = 0; i < argc; i++) {
    printf("%s = %s\n", azColName[i], argv[i] ? argv[i] : "NULL");
  }
  printf("\n");
  return 0;
}

/**
 * the table in the database has to be of the following format:
 *
 * create table music
 * ( filename VARCHAR(255),
 *   title VARCHAR(255),
 *   artist VARCHAR(255),
 *   album VARCHAR(255),
 *   track INTEGER,
 *   year INTEGER,
 *   genre VARCHAR(255),
 *   comment VARCHAR(255));
 *
 */
XmmsKdeDB::XmmsKdeDB(KConfig *conf, QPixmap *ico) {

  db = NULL;
  querydb = NULL;
  
  pathListBox = NULL;
  insertThread = NULL;
  statusFrame = NULL;
  connected = false;

  icon = ico;
  config = conf;
  readConfig();

  sync = true;
  connectDB();
}

XmmsKdeDB::~XmmsKdeDB() {

  if (connected)
    disconnectDB();

}

bool XmmsKdeDB::connectDB() {

  char *error = 0;

  if (enable) {

    if (connected) {
      disconnectDB();
    }

    qDebug("xmms-kde: trying to open database");
    db = sqlite_open(name, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }

    sqlite_exec(db, "CREATE TABLE music (filename STRING UNIQUE ON CONFLICT REPLACE, title STRING, artist STRING, album STRING, track STRING, year STRING, genre STRING, comment STRING)", 0, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }

    qDebug("xmms-kde: table created");

    querydb = sqlite_open(name, 0, &error);

    if (error) {
      free(error);
      error = 0;
    }

    if (db && querydb) {
      emit(statusChanged(i18n("connected to database")));
      connected = true;
      return true;
    } else {
      emit(statusChanged(i18n("database connection failed")));
      connected = false;
      return false;
    }
  }
  return false;
}

void XmmsKdeDB::disconnectDB() {

  if (enable) {
    if (connected) {
      connected = false;
      sqlite_close(db);
      sqlite_close(querydb);
      db = NULL;
      querydb = NULL;
      //emit(statusChanged("disconnected"));
    }
  }
}


bool XmmsKdeDB::isConnectedDB() {

  return connected;
}

QWidget* XmmsKdeDB::getConfigurationWidget(QWidget *parent) {

  QWidget *dbWidget = new QWidget(parent);

  QVBoxLayout *vbox = new QVBoxLayout(dbWidget, 5);

  QHBoxLayout *box = new QHBoxLayout(vbox, 10);
  enableBox = new QCheckBox(i18n("enable Database support"), dbWidget);
  enableBox->setChecked(enable);

  box->addWidget(enableBox);
  box->addStretch(10);

  connect(enableBox, SIGNAL(clicked()), this, SLOT(configurationChanged()));

  pathListBox = new QListBox(dbWidget);
  pathListBox->insertStringList(pathList);
  vbox->addWidget(pathListBox);

  box = new QHBoxLayout(vbox, 10);

  QPushButton *addButton = new QPushButton(i18n("Add Path"), dbWidget);
  connect(addButton, SIGNAL(clicked()), this, SLOT(addPathToList()));

  QPushButton *removeButton = new QPushButton(i18n("Remove Path"), dbWidget);
  connect(removeButton, SIGNAL(clicked()), this, SLOT(removePathFromList()));

  box->addWidget(addButton);
  box->addStretch(10);
  box->addWidget(removeButton);

  return dbWidget;
}

void XmmsKdeDB::configurationChanged() {

  enable = enableBox->isChecked();
}

void XmmsKdeDB::updateDatabase() {

  if (insertThread != NULL) {

    if (insertThread->running()) {
      if (statusFrame)
	statusFrame->show();
      return;
    } else {
      delete insertThread;
      insertThread = NULL;
    }
  }

  if (!enable)
    return;

  if (!connected)
    connectDB();

  if (statusFrame) {
    delete statusFrame;
    statusFrame = NULL;
  }

  if (pathList.empty())
    return;

  statusFrame = new QFrame(NULL, "Database status", WStyle_Title);
  statusFrame->setCaption(i18n("Database status"));
  QVBoxLayout *box = new QVBoxLayout(statusFrame, 10);
  QHBoxLayout *hbox = new QHBoxLayout(box, 20);

  QLabel *pixLabel = new QLabel(statusFrame);
  pixLabel->setPixmap(*icon);

  QVBoxLayout *vbox2 = new QVBoxLayout(hbox, 0);
  QVBoxLayout *vbox3 = new QVBoxLayout(hbox, 0);

  QLabel *label = new QLabel(i18n("Updating database (this will take a while)."), statusFrame);
  StatusLabel *statusLabel = new StatusLabel("", statusFrame, 45);

  StatusLabel *fileLabel = new StatusLabel("", statusFrame, 45);

  vbox2->addWidget(pixLabel);
  vbox3->addWidget(label);
  vbox3->addWidget(statusLabel);
  vbox3->addWidget(fileLabel);

  ProgressLabel *progressLabel = new ProgressLabel(statusFrame);
  box->addWidget(progressLabel);

  QHBoxLayout *hbox2 = new QHBoxLayout(box, 10);

  QPushButton *okButton = new QPushButton(i18n(i18n("OK")), statusFrame);
  okButton->setDefault(true);
  //QPushButton *cancelButton = new QPushButton(i18n("Cancel"), statusFrame);

  hbox2->addStretch(10);
  hbox2->addWidget(okButton, 1);
  //hbox2->addWidget(cancelButton, 1);

  connect(okButton, SIGNAL(clicked()), statusFrame, SLOT(hide()));
  //connect(cancelButton, SIGNAL(clicked()), this, SLOT(stopInsertThread()));
  //connect(cancelButton, SIGNAL(clicked()), statusFrame, SLOT(hide()));

  statusFrame->show();

  fileLabel->setMaximumSize(label->size());
  statusLabel->setMaximumSize(label->size());
  fileLabel->setMinimumSize(label->size());
  statusLabel->setMinimumSize(label->size());

  insertThread = new InsertThread(db, pathList, statusLabel, fileLabel,
				  progressLabel, updated);
  insertThread->start();

  updated = QDateTime::currentDateTime();

  writeConfig();
  sync = true;
}

void XmmsKdeDB::readConfig() {

  config->setGroup("DB");
  enable = config->readBoolEntry("enable", false);
  QString dbfile(locateLocal("data", "xmms-kde/music.db"));
  //QString dbfile("/tmp/music.db");
  name = config->readEntry("name", dbfile);
  qDebug("xmms-kde: using database '" + dbfile + "'");
  pathList = config->readListEntry("paths");
  updated = config->readDateTimeEntry("updated");
}


void XmmsKdeDB::writeConfig() {

  config->setGroup("DB");
  config->writeEntry("enable", enable);
  config->writeEntry("paths", pathList);
  config->writeEntry("updated", updated);
}


void XmmsKdeDB::addPathToList() {

  QString dir = QFileDialog::getExistingDirectory();
  if (dir != NULL)
    pathListBox->insertItem(dir);

  pathList.clear();
  for (unsigned int index = 0; index < pathListBox->count(); index ++)
    pathList << pathListBox->text(index);


  sync = false;
}

void XmmsKdeDB::removePathFromList() {

  pathListBox->removeItem(pathListBox->currentItem());

  pathList.clear();
  for (unsigned int index = 0; index < pathListBox->count(); index ++)
    pathList << pathListBox->text(index);

  sync = false;
}

void XmmsKdeDB::stopInsertThread() {

 // FIXME
  /*
  if (insertThread) {

    if (insertThread->running())
      insertThread->terminate();
  }
  */
}


XmmsKdeDBQuery::XmmsKdeDBQuery(XmmsKdeDB *datab, PlayerInterface *p,
			       QPixmap *icon, KConfig *conf)
  : QSplitter(0, "Database Query") {

  config = conf;
  readConfig();

  db = datab;
  player = p;

  QVBox *vbox1 = new QVBox(this);
  vbox1->setSpacing(2);

  firstCombo = new QComboBox(false, vbox1);
  firstBox = new QListBox(vbox1);
  firstBox->setSelectionMode(QListBox::Extended);
  firstFilterEdit = new QLineEdit(vbox1);


  QSplitter *vsplitter = new QSplitter(this);
  vsplitter->setOrientation(QSplitter::Vertical);

  QVBox *vbox2 = new QVBox(vsplitter);
  vbox2->setSpacing(2);

  secondCombo = new QComboBox(false, vbox2);
  secondBox = new QListBox(vbox2);
  secondBox->setSelectionMode(QListBox::Extended);
  secondFilterEdit = new QLineEdit(vbox2);

  QVBox *vbox3 = new QVBox(vsplitter);
  resultBox = new QListBox(vbox3);
  resultBox->setSelectionMode(QListBox::Extended);
  resultFilterEdit = new QLineEdit(vbox3);

  QHBox *hbox = new QHBox(vbox3);

  QPushButton *set = new QPushButton(i18n("set"), hbox);
  QPushButton *add = new QPushButton(i18n("add"), hbox);

  connect((QObject *) set, SIGNAL(clicked()),
	  this, SLOT(setPlayList()));

  connect((QObject *) add, SIGNAL(clicked()),
	  this, SLOT(addPlayList()));

  connect((QObject *) resultBox, SIGNAL(selected(int)),
	  (QObject *) this, SLOT(play(int)));
  connect((QObject *) firstBox, SIGNAL(selectionChanged()),
	  (QObject *) this, SLOT(firstBoxChanged()));
  connect((QObject *) secondBox, SIGNAL(selectionChanged()),
	  (QObject *) this, SLOT(secondBoxChanged()));
  connect((QObject *) secondBox, SIGNAL(doubleClicked(QListBoxItem *)),
	  (QObject *) this, SLOT(secondDClicked(QListBoxItem *)));
  connect((QObject *) firstCombo, SIGNAL(activated(int)),
	  (QObject *) this, SLOT(firstComboChanged(int)));
  connect((QObject *) secondCombo, SIGNAL(activated(int)),
	  (QObject *) this, SLOT(secondComboChanged(int)));
  connect((QObject *) firstFilterEdit, SIGNAL(textChanged(const QString&)),
	  (QObject *) this, SLOT(firstFilterChanged(const QString&)));
  connect((QObject *) firstFilterEdit, SIGNAL(returnPressed()),
	  (QObject *) this, SLOT(firstBoxChanged()));
  connect((QObject *) secondFilterEdit, SIGNAL(textChanged(const QString&)),
	  (QObject *) this, SLOT(secondFilterChanged(const QString&)));
  connect((QObject *) secondFilterEdit, SIGNAL(returnPressed()),
	  (QObject *) this, SLOT(secondBoxChanged()));
  connect((QObject *) resultFilterEdit, SIGNAL(textChanged(const QString&)),
	  (QObject *) this, SLOT(resultFilterChanged(const QString&)));
  connect((QObject *) resultFilterEdit, SIGNAL(returnPressed()),
	  (QObject *) this, SLOT(setPlayList()));


  allText=i18n("--All--");
  setCaption(i18n("Database Query"));
  modeTexts << i18n("Artist") <<i18n("Album") <<i18n("Year") <<i18n("Genre");
  firstCombo->insertStringList(modeTexts);
  secondCombo->insertStringList(modeTexts);
  searchTexts <<"artist" <<"album" <<"year" <<"genre";

  numActiveSearch = 0;
     
  KWin::setIcons(winId(), *icon, *icon);

  searchThread = new SearchThread(this, db);
  searchThread->start();

  firstCombo->setCurrentItem((int)ARTIST);
  secondCombo->setCurrentItem((int)ALBUM);
  firstComboChanged((int)ARTIST);
  secondComboChanged((int)ALBUM);
}

XmmsKdeDBQuery::~XmmsKdeDBQuery() {

  searchThread->kill();
}

void XmmsKdeDBQuery::setPlayer(PlayerInterface *p) {

  player = p;
}

void XmmsKdeDBQuery::customEvent(QCustomEvent *e) {

  SearchEvent *search = (SearchEvent *)e;
  QListBoxItem *item;
  switch (search->type()){
  case TO_RESULTBOX:
    resultBox->clear();
    while (item = search->items.dequeue())
      resultBox->insertItem(item, 0);
    break;

  case TO_FIRSTBOX:
    firstBox->clear();
    firstBox->insertItem(allText);
    while (item = search->items.dequeue())
      firstBox->insertItem(item);
    break;

  case TO_SECONDBOX:
    secondBox->clear();
    secondBox->insertItem(allText);
    while (item = search->items.dequeue())
      secondBox->insertItem(item);
    break;

  case TO_FIRSTBOX_FOCUS:
    firstBox->clear();
    firstBox->insertItem(allText);
    while (item = search->items.dequeue())
      firstBox->insertItem(item);
    
    QListBoxItem *current = firstBox->findItem(search->focusText);
    firstBox->setCurrentItem(current);
    firstBox->centerCurrentItem();
    firstBox->setSelected(current, true);
    break;
  }

  //kapp->restoreOverrideCursor();
}

void XmmsKdeDBQuery::activateSearch() {
   
  numActiveSearch--;

   if (numActiveSearch == 0) {
     searchThread->waitCondition.wakeAll();
     //     kapp->setOverrideCursor(waitCursor);
   } else
     searchThread->getSearch();
}

void XmmsKdeDBQuery::firstFilterChanged(const QString &text) {

   searchThread->addSearch(new SearchEvent(TO_FIRSTBOX, getFirstQuery()), false);
   numActiveSearch++;
   if(text.length() > 2)
     activateSearch();
   else
     QTimer::singleShot(400, this, SLOT(activateSearch()));
}

void XmmsKdeDBQuery::secondFilterChanged(const QString &text) {

   searchThread->addSearch(new SearchEvent(TO_SECONDBOX, getSecondQuery()), false);
   numActiveSearch++;
   if(text.length() > 2)
     activateSearch();
   else
     QTimer::singleShot(400, this, SLOT(activateSearch()));
}

void XmmsKdeDBQuery::resultFilterChanged(const QString &text) {

   searchThread->addSearch(new SearchEvent(TO_RESULTBOX, getResultQuery()), false);
   numActiveSearch++;
   if(text.length() > 2)
     activateSearch();
   else
     QTimer::singleShot(400, this, SLOT(activateSearch()));
}

QString XmmsKdeDBQuery::getFirstQuery() {

  QString lQuery = QString("SELECT DISTINCT %1 FROM music %3 ORDER BY %2 ASC").arg(searchTexts[firstMode]).arg(searchTexts[firstMode]);

if (!firstFilterEdit->text().isEmpty())
    lQuery = lQuery.arg("WHERE %1 LIKE '%%2%'").arg(searchTexts[firstMode]).arg(firstFilterEdit->text());
  else
    lQuery = lQuery.arg("");

  return lQuery;
}

QString XmmsKdeDBQuery::getSecondQuery() {

  QString lQuery = QString("SELECT DISTINCT %1 FROM music %3 %4 %5 %6 ORDER BY %2 ASC").arg(searchTexts[secondMode]).arg(searchTexts[secondMode]);
  bool useFirst = true, useFilter = true;

  if (firstList.isEmpty() || firstList.contains(allText))
    useFirst = false;
  if (secondFilterEdit->text().isEmpty())
		useFilter = false;

  if (useFirst || useFilter)
    lQuery = lQuery.arg("WHERE");
  else
    lQuery = lQuery.arg("");

  if (useFirst)
    lQuery = lQuery.arg("%1 IN %2").arg(searchTexts[firstMode]).arg(prepareList(firstList));
  else
    lQuery = lQuery.arg("");

  if (useFirst && useFilter)
    lQuery = lQuery.arg("AND");
  else
    lQuery = lQuery.arg("");

  if (useFilter)
    lQuery = lQuery.arg("%1 LIKE '%%2%'").arg(searchTexts[secondMode]).arg(secondFilterEdit->text());
  else
    lQuery = lQuery.arg("");

  return lQuery;
}

QString XmmsKdeDBQuery::getResultQuery() {

  QString lQuery = QString("SELECT artist, title, filename FROM music %3 %4 %5 %6 %7 %8 ORDER BY album ASC");

  bool useFirst = true, useSecond = true, useFilter = true;

  if (firstList.isEmpty() || firstList.contains(allText))
    useFirst = false;
  if (secondList.isEmpty() || secondList.contains(allText))
    useSecond = false;
  if (resultFilterEdit->text().isEmpty())
    useFilter = false;

  if (useFirst || useSecond || useFilter)
    lQuery = lQuery.arg("WHERE");
  else
    lQuery = lQuery.arg("");

  if (useFirst)
    lQuery = lQuery.arg("%1 IN %2").arg(searchTexts[firstMode]).arg(prepareList(firstList));
  else
    lQuery = lQuery.arg("");

  if (useFirst && useSecond)
    lQuery = lQuery.arg("AND");
  else
    lQuery = lQuery.arg("");

  if (useSecond)
    lQuery = lQuery.arg("%1 IN %2").arg(searchTexts[secondMode]).arg(prepareList(secondList));
  else
    lQuery = lQuery.arg("");

  if ((useFirst || useSecond) && useFilter)
    lQuery = lQuery.arg("AND");
  else
    lQuery = lQuery.arg("");

  if (useFilter)
    lQuery = lQuery.arg("(artist LIKE '%%1%' OR title LIKE '%%2%')").arg(resultFilterEdit->text()).arg(resultFilterEdit->text());
  else
    lQuery = lQuery.arg("");

  return lQuery;
}



void XmmsKdeDBQuery::firstBoxChanged() {

  firstList.clear();
  secondList.clear();	//so getResultQuery wont pull conditions out of this one.

  for (QListBoxItem *lItem = firstBox->firstItem(); lItem; lItem = lItem->next())
    if (lItem->isSelected())
      firstList << lItem->text();

  if (!firstFilterEdit->text().isEmpty() && (firstList.empty() || firstList.contains(allText))) {

    firstList.clear(); //rebuild list
    QListBoxItem *lItem = firstBox->firstItem();
    while (lItem = lItem->next()) //dont use first item since it's the text ---All---
      firstList << lItem->text();
  }

  searchThread->addSearch(new SearchEvent(TO_SECONDBOX, getSecondQuery()));
  searchThread->addSearch(new SearchEvent(TO_RESULTBOX, getResultQuery()));

  //  kapp->setOverrideCursor(waitCursor);
}

void XmmsKdeDBQuery::secondBoxChanged() {

  secondList.clear();
  for (QListBoxItem *lItem = secondBox->firstItem(); lItem; lItem = lItem->next())
    if (lItem->isSelected())
      secondList << lItem->text();

  if(!secondFilterEdit->text().isEmpty() && (secondList.empty() || secondList.contains(allText))) {

    secondList.clear(); //rebuild list
    QListBoxItem *lItem = secondBox->firstItem();
    while (lItem = lItem->next()) //dont use first item since it's the text ---All---
      secondList << lItem->text();
  }

  searchThread->addSearch(new SearchEvent(TO_RESULTBOX, getResultQuery()));

  //  kapp->setOverrideCursor(waitCursor);
}

void XmmsKdeDBQuery::secondDClicked(QListBoxItem *item) {

  QString text;
  if (item)
    text = item->text();
  else
    text = allText;

  firstList.clear();
  secondList.clear();

  firstList << text;

  //swap modes
  EFilterMode temp = firstMode;
  firstMode = secondMode;
  secondMode = temp;

  firstCombo->setCurrentItem((int)firstMode);
  secondCombo->setCurrentItem((int)secondMode);

  searchThread->addSearch(new SearchEvent(TO_FIRSTBOX_FOCUS, getFirstQuery(), text));
  searchThread->addSearch(new SearchEvent(TO_SECONDBOX, getSecondQuery()));

  //  kapp->setOverrideCursor(waitCursor);

  firstBox->setFocus();
}

void XmmsKdeDBQuery::firstComboChanged(int index) {

  firstMode = (EFilterMode) index;
  searchThread->addSearch(new SearchEvent(TO_FIRSTBOX, getFirstQuery()));
  //kapp->setOverrideCursor(waitCursor);
}

void XmmsKdeDBQuery::secondComboChanged(int index) {

  secondMode = (EFilterMode) index;
  searchThread->addSearch(new SearchEvent(TO_SECONDBOX, getSecondQuery()));
  //kapp->setOverrideCursor(waitCursor);
}

void XmmsKdeDBQuery::play(int index) {

  if (player) {

    if (!click) // set
      player->playlistClear();

    player->playlistAdd(((QueryItem *) resultBox->item(index))->getFile());
    
    if (!click)
      player->play();
  }
}

void XmmsKdeDBQuery::setPlayList() {

  if (player) {

    player->playlistClear();
    addPlayList();

    if (resultBox->count() > 0)
      player->play();
  }
}


void XmmsKdeDBQuery::addPlayList() {

  if (player) {
    QStringList files;
    QStringList allfiles;

    for (unsigned int index = 0; index < resultBox->count(); index ++) {
      allfiles += ((QueryItem *) resultBox->item(index))->getFile();
      //printf("   [%s]\n", ((QueryItem *) resultBox->item(index))->getFile().latin1());
      if (resultBox->isSelected(index))
	files += ((QueryItem *) resultBox->item(index))->getFile();
    }

    if (!files.empty())
      player->playlistAdd(files);//.join(QString("\r\n")));
    else if (!allfiles.empty())
      player->playlistAdd(allfiles);//.join(QString("\r\n")));
  }

  resultBox->clearSelection();
}

QString XmmsKdeDBQuery::prepareList(QStringList list) {

  for (unsigned int i = 0; i < list.count(); i++) {
    QString &string = list[i];
    for (unsigned int i = 0; i < string.length(); i++) {
      if (string[i] == '\'') {
	string.insert(i, '\'');
	i++;
      }
    }
  }
  return "(\'" + list.join("\', \'") + "\')";
}

void XmmsKdeDBQuery::popup() {

  if (db->isEnabled()) {

#if defined(KDE_IS_VERSION)
#if KDE_IS_VERSION(3, 1, 90)
    if (!KWin::windowInfo(winId()).isOnCurrentDesktop())
      KWin::setOnDesktop(winId(), KWin::currentDesktop());
#endif
#else
    if (KWin::info(winId()).desktop != KWin::currentDesktop())
    KWin::setOnDesktop(winId(), KWin::currentDesktop());
#endif
    if (!isVisible())
      show();
    if (isMinimized())
      showNormal();
    if (!isActiveWindow()) {
      setActiveWindow();
      raise();
    }

    firstBox->setFocus();
  }
}

void XmmsKdeDBQuery::readConfig() {

  config->setGroup("DB");
  framePos = QPoint(0, 0);
  framePos = config->readPointEntry("queryframeposition", &framePos);
  frameSize = QSize(200, 320);
  frameSize = config->readSizeEntry("queryframesize", &frameSize);
  pop = config->readNumEntry("popup", 2);
  click = config->readNumEntry("click", 0);

  resize(frameSize);
  move(framePos);
}


void XmmsKdeDBQuery::writeConfig() {

  config->setGroup("DB");
  config->writeEntry("queryframeposition", pos());
  config->writeEntry("queryframesize", size());
  config->writeEntry("popup", pop);
  config->writeEntry("click", click);
}

QWidget* XmmsKdeDBQuery::getConfigurationWidget(QWidget *parent) {

  QWidget *dbWidget = new QWidget(parent);

  QVBoxLayout *vbox = new QVBoxLayout(dbWidget, 10);

  queryGroup = new QVButtonGroup(i18n("Show query window"), dbWidget);
  connect(queryGroup, SIGNAL(clicked(int)), this, SLOT(popupChanged(int)));

  QRadioButton *kdeStartup = new QRadioButton(i18n("at KDE startup"), queryGroup);
  QRadioButton *playerStartup = new QRadioButton(i18n("at player startup"), queryGroup);
  QRadioButton *noStartup = new QRadioButton(i18n("when selected from menu"), queryGroup);
  queryGroup->setButton(pop);

  vbox->addWidget(queryGroup);

  clickGroup = new QVButtonGroup(i18n("Doubleclick action"), dbWidget);
  connect(clickGroup, SIGNAL(clicked(int)), this, SLOT(clickChanged(int)));

  QRadioButton *setB = new QRadioButton(i18n("set as playlist"), clickGroup);
  QRadioButton *addB = new QRadioButton(i18n("add to playlist"), clickGroup);

  clickGroup->setButton(click);

  vbox->addWidget(clickGroup);

  return dbWidget;
}

void XmmsKdeDBQuery::popupChanged(int index) {

  pop = index;
}

void XmmsKdeDBQuery::clickChanged(int index) {

  click = index;
}


QueryItem::QueryItem(QString text, QString file):
  QListBoxText(text) {

  filename = file;
}

QueryItem::~QueryItem() {

}

QString QueryItem::getFile() {

  return filename;
}


void SearchThread::run() {

  SearchEvent *search;
  while (true) {

    //perform all searches that are queued
    while (search = getSearch())
      doSearch(search);

    //wait for something to happen
    waitCondition.wait();
    if (shouldExit)
      exit();
  }
}

void SearchThread::doSearch(SearchEvent *search) {

  QStringList returnList;
  char *error;
  char **result;
  int nrow;
  int ncolumn;
  int rc;

  if (!db)
    return;
  if (!db->isConnectedDB())
    if (!db->connectDB())
      return;

  rc = sqlite_get_table(db->querydb, search->query.latin1(), &result, &nrow, &ncolumn, &error);
  qDebug("xmms-kde: querying: %s", search->query.latin1());

  if (error) {
    qDebug("xmms-kde: sqlite error: %s", error);
    free(error);
    error = 0;
  }

  if (rc != SQLITE_OK) {
    qDebug("xmms-kde: database query failed");
  }
  else if (nrow >= 1) {
    qDebug("xmms-kde: num rows in result: %i", nrow);


    if ((int)search->type() == (int)TO_RESULTBOX) {
      for (int i = 1; i <= nrow; i++) {
	// only add it when the file exists
	QString fileName(result[i * 3 + 2]);
	if (QFile::exists(fileName)) {
	  QString name = QString(result[i*3]) + " - " + QString(result[i*3 + 1]);
	  search->items.enqueue(new QueryItem(name, QString(result[i*3 +2])));
	}
      }
    } else {
      for (int i = 1; i <= nrow; i++)
	search->items.enqueue(new QListBoxText(QString(result[i])));
    }
  }
 
  sqlite_free_table(result);

  postEvent(receiver, search);
}


InsertThread::InsertThread(sqlite *datab, QStringList dirs,
			   StatusLabel *status, StatusLabel *file,
			   ProgressLabel *progess, QDateTime up) {
  db = datab;
  dir = dirs;

  statusLabel = status;
  fileLabel = file;
  progressLabel = progess;
  updated = up;
}

void InsertThread::run() {

  updateDatabase(dir);
}

void InsertThread::updateDatabase(QStringList dir) {

  StatusEvent *event = new StatusEvent(i18n("reading files from database..."));
  postEvent(statusLabel, event);


  ProgressTotalEvent *ptevent = new ProgressTotalEvent(1000);
  postEvent(progressLabel, ptevent);

  ProgressStepEvent *psevent = new ProgressStepEvent(1);
  postEvent(progressLabel, psevent);

  QTime t, r;
  t.start();
  r.start();

  // the contents of the database in a hash
  hash_set<const char *, hash<const char *>, eqstr> database;

  // the contents of the filesystem in a hash
  hash_set<const char *, hash<const char *>, eqstr> files;

  hash_set<const char*, hash<const char*>, eqstr>::const_iterator it, it2;

  QString q("SELECT filename from music");

  int nrow, ncolumn;
  char *error;
  char **result;

  int rc = sqlite_get_table(db, q.latin1(), &result, &nrow, &ncolumn, &error);

  if (error) {
    free(error);
    error = 0;
  }

  printf("xmms-kde: read files from database: %d seconds\n", t.elapsed() /1000);
  t.start();

  database.resize(nrow * 2);

  for (int index = 0; index < nrow; index ++) {
    database.insert(result[index]);
  }

  printf("xmms-kde: inserted database into hash: %d seconds\n", t.elapsed() /1000);
  t.start();

  event = new StatusEvent(i18n("reading files from filesystem..."));
  postEvent(statusLabel, event);

  vector<QString *> v;

  for (QStringList::Iterator it = dir.begin(); it != dir.end(); ++it )
    addPathToVector(*it, &v);

  printf("xmms-kde: read files from filesystem: %d seconds\n", t.elapsed() /1000);
  t.start();

  files.resize(v.size() * 2);

  for (unsigned int index = 0; index < v.size(); index ++)
    files.insert(v[index]->latin1());

  printf("xmms-kde: inserted files into hash: %d seconds\n", t.elapsed() /1000);
  t.start();

  ptevent = new ProgressTotalEvent(v.size() + nrow + 1);
  postEvent(progressLabel, ptevent);

  event = new StatusEvent(i18n("removing old entries..."));
  postEvent(statusLabel, event);

  float thresh = 0.0;
  float size = v.size() + nrow; // number of files in filesystem + in database
  if (size == 0) size = 1;

  QString query("BEGIN TRANSACTION remove");
  rc = sqlite_exec(db, query.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  int progressIndex = 0;

  it = database.begin();

  while (it != database.end()) {

    if (((float) progressIndex) / size - thresh >= 0.01) {
      thresh = ((float) progressIndex) / size;
      ProgressStepEvent *psevent = new ProgressStepEvent(progressIndex);
      postEvent(progressLabel, psevent);
    }

    progressIndex ++;

    it2 = files.find(*it);
    if (it2 == files.end()) {
      deleteFromDatabase(QString(*it));
    }
    it++;
  }

  QString query2("COMMIT TRANSACTION remove");
  rc = sqlite_exec(db, query2.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  printf("xmms-kde: removed files from database: %d seconds\n", t.elapsed() / 1000);
  t.start();

  event = new StatusEvent(i18n("adding new files..."));
  postEvent(statusLabel, event);

  QString query3("BEGIN TRANSACTION insert ON CONFLICT REPLACE");
  rc = sqlite_exec(db, query3.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  for (unsigned int index = 0; index < v.size(); index ++) {

    // only generate an event when progressbar should move (>=1% difference)
    if (((float) progressIndex) / size - thresh >= 0.01) {
      thresh = ((float) progressIndex) / size;

      ProgressStepEvent *psevent = new ProgressStepEvent(progressIndex);
      postEvent(progressLabel, psevent);
    }

    progressIndex++;

    it = database.find(v[index]->latin1());
    if (it == database.end()) {
      insertIntoDatabase(*v[index]);
    } else {

      QFileInfo inf(*v[index]);
      QDateTime d = inf.lastModified();
      if (d > updated)
	insertIntoDatabase(*v[index]);
    }
  }

  psevent = new ProgressStepEvent(v.size() + nrow + 1);
  postEvent(progressLabel, psevent);

  QString query4("COMMIT TRANSACTION insert");
  rc = sqlite_exec(db, query4.latin1(), NULL, 0, &error);

  if (error) {
    free(error);
    error = 0;
  }

  event = new StatusEvent(i18n("updating database: done."));
  postEvent(statusLabel, event);
  event = new StatusEvent("");
  postEvent(fileLabel, event);

  sqlite_free_table(result);

  for (unsigned int index = 0; index < v.size(); index ++)
    delete v[index];

  printf("xmms-kde: updated database: %d seconds\n", r.elapsed() / 1000);
}

void InsertThread::insertIntoDatabase(QString file) {

  QFileInfo inf(file);

  if (inf.isFile()) {

    KFileMetaInfo m(file);

    if (m.isValid())
      qDebug("mime: %s\n", m.mimeType().latin1());

    if (m.isValid() && (!m.mimeType().endsWith("x-mp3") ||
			!m.mimeType().endsWith("x-ogg"))) {
      
      QFileInfo f(file);
      StatusEvent *event = new StatusEvent("[" + m.item("Artist").string() +
					   "] " + m.item("Title").string());

      postEvent(fileLabel, event);

      QString title = m.item("Title").string().stripWhiteSpace();
      QString artist = m.item("Artist").string().stripWhiteSpace();
      QString album = m.item("Album").string().stripWhiteSpace();
      QString track = m.item("Tracknumber").string().stripWhiteSpace();
      QString date = m.item("Date").string().stripWhiteSpace();
      QString genre = m.item("Genre").string().stripWhiteSpace();
      QString comment = m.item("Comment").string().stripWhiteSpace();

      char *error;

      int rc = sqlite_exec_printf(db,
				  "INSERT INTO music VALUES(%Q, %Q, %Q, %Q, %Q, %Q, %Q, %Q)", 0, 0, &error,
				  file.latin1(),
				  title.latin1(), artist.latin1(),
				  album.latin1(), track.latin1(),
				  date.latin1(), genre.latin1(),
				  comment.latin1());

      if (error) {
	free(error);
	error = 0;
      }

      if (rc != SQLITE_OK)
	qDebug("xmms-kde: database insert failed on [%s]\n", file.latin1());
      /*
      else
	printf("xmms-kde: inserted [%s]\n", file.latin1());
      */
    }
#ifdef HAVE_ID3LIB
    else if (file.lower().endsWith("mp3")) {
      printf("xmms-kde id3lib: %s\n", file.latin1());
      // use id3lib if available
      const char *artist = NULL;
      const char *title = NULL;
      const char *album = NULL;
      const char *track = NULL;
      const char *date = NULL;

      ID3_Tag tag(file.latin1());
      ID3_Frame *frame = NULL;
      ID3_Field *field = NULL;

      frame = tag.Find(ID3FID_LEADARTIST);
      if (frame) {
	field = frame->GetField(ID3FN_TEXT);
	if (field) {
	  artist = field->GetRawText();
	}
      }

      frame = tag.Find(ID3FID_TITLE);
      if (frame) {
	field = frame->GetField(ID3FN_TEXT);
	if (field) {
	  title = field->GetRawText();
	}
      }

      frame = tag.Find(ID3FID_ALBUM);
      if (frame) {
	field = frame->GetField(ID3FN_TEXT);
	if (field) {
	  album = field->GetRawText();
	}
      }

      frame = tag.Find(ID3FID_TRACKNUM);
      if (frame) {
	field = frame->GetField(ID3FN_TEXT);
	if (field) {
	  track = field->GetRawText();
	}
      }

      frame = tag.Find(ID3FID_DATE);
      if (frame) {
	field = frame->GetField(ID3FN_TEXT);
	if (field) {
	  date = field->GetRawText();
	}
      }

      char *error;

      int rc = sqlite_exec_printf(db,
				  "INSERT INTO music VALUES(%Q, %Q, %Q, %Q, %Q, %Q, %Q, %Q)", 0, 0, &error,
				  file.latin1(),
				  title, artist, album, track, date, "", "");

      if (error) {
	free(error);
	error = 0;
      }

      StatusEvent *event = new StatusEvent("[" + QString(artist) +
					   "] " + QString(title));
      postEvent(fileLabel, event);

      /*
      delete title;
      delete artist;
      delete album;
      delete track;
      delete date;
      */

      if (rc != SQLITE_OK)
 	qDebug("xmms-kde: database insert failed on [%s]\n", file.latin1());
    }
#endif
#ifdef HAVE_VORBIS
    else if (file.lower().endsWith("ogg")) {
      printf("xmms-kde vorbis: %s\n", file.latin1());
      FILE *fh;

      if ((fh = fopen(file.latin1(), "r")) != NULL) {
	OggVorbis_File *vf = (OggVorbis_File *) malloc(sizeof(OggVorbis_File));
	if (ov_open(fh, vf, NULL, 0) == 0) {

	  vorbis_comment *comment = 0;
	  comment = ov_comment(vf, -1);

	  char **ptr = comment->user_comments;

	  char *title_utf8 = 0;
	  char *artist_utf8 = 0;
	  char *album_utf8 = 0;
	  char *track_utf8 = 0;
	  char *date_utf8 = 0;
	  char *genre_utf8 = 0;
	
	  for (int i = 0; i < comment->comments; i++, ptr++) {

	    if (!*ptr)
	      continue;
	    
	    if (!strncasecmp(*ptr, "ARTIST=", 7))
	      artist_utf8 = (*ptr) + 7;

	    else if (!strncasecmp(*ptr, "TITLE=", 6))
	      title_utf8 = (*ptr) + 6;

	    else if (!strncasecmp(*ptr, "ALBUM=", 6))
	      album_utf8 = (*ptr) + 6;

	    else if (!strncasecmp(*ptr, "DATE=", 5))
	      date_utf8 = (*ptr) + 5;

	    else if (!strncasecmp(*ptr, "GENRE=", 6))
	      genre_utf8 = (*ptr) + 6;

	    else if (!strncasecmp(*ptr, "TRACKNUMBER=", 12))
	      track_utf8 = (*ptr) + 12;
	  }

	  QString title = QString::fromUtf8(title_utf8);
	  QString artist = QString::fromUtf8(artist_utf8);
	  QString album = QString::fromUtf8(album_utf8);
	  QString track = QString::fromUtf8(track_utf8);
	  QString date = QString::fromUtf8(date_utf8);
	  QString genre = QString::fromUtf8(genre_utf8);
	
	  char *error;
	  
	  int rc = sqlite_exec_printf(db,
				      "INSERT INTO music VALUES(%Q, %Q, %Q, %Q, %Q, %Q, %Q, %Q)", 0, 0, &error,
				      file.latin1(),
				      title.latin1(), artist.latin1(), album.latin1(), track.latin1(), date.latin1(), genre.latin1(), "");
	  
	  if (error) {
	    free(error);
	    error = 0;
	  }

	  StatusEvent *event = new StatusEvent("[" + QString(artist) +
					       "] " + QString(title));
	  postEvent(fileLabel, event);

	  if (rc != SQLITE_OK)
	    qDebug("xmms-kde: database insert failed on [%s]\n", file.latin1());
	} else {
	  fclose(fh);
	}
	
	if (vf) {
	  ov_clear(vf);
	}
      }
    }
#endif
  }
}


void InsertThread::deleteFromDatabase(QString file) {

  char *error;

  int rc = sqlite_exec_printf(db, "DELETE FROM music WHERE filename = %Q", NULL, 0, &error, file.latin1());

  if (error) {
    free(error);
    error = 0;
  }

  if (rc != SQLITE_OK)
    qDebug("xmms-kde: database delete failed on [%s]\n", file.latin1());
  /*
  else
    printf("xmms-kde: deleted [%s]\n", file.latin1());
  */
}

void InsertThread::addPathToVector(QString path, vector<QString *> *v) {

  QDir dir(path);
  if (!dir.exists())
    return;

  QStringList fileList(dir.entryList());

  for (QStringList::Iterator it = fileList.begin(); it != fileList.end(); ++it) {
    QString absfile(dir.absPath() + "/" + (*it));
    QString file(*it);

    QFileInfo inf(absfile);

    if (inf.isDir()) {
      if (!file.startsWith(".")) {
	addPathToVector(absfile, v);
      }
    } else if (inf.isFile()
	       && ((file.endsWith(".mp3") || file.endsWith(".ogg") ||
		    file.endsWith(".MP3") || file.endsWith(".OGG")))) {

      v->push_back(new QString(absfile));
    }
  }
}

