/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "ctexture.h"
#include "cpicture.h"

#include <GL/glu.h>
#include <math.h>

#define HAS_ALPHA ((tex_format = GL_RGBA) || (tex_format == GL_ALPHA))

CTexture::CTexture()
{
	textureID = 0;
	reset();
}

CTexture::CTexture(gchar i_bpp, guint32 i_width, guint32 i_height, const gchar *i_data)
	:CPicture(i_bpp, i_width, i_height, i_data)
{
	textureID = 0;
	reset();

	switch (i_bpp) {
	case 8: tex_format = pic_format = GL_ALPHA; break;
	case 24: tex_format = pic_format = GL_RGB; break;
	case 32: tex_format = pic_format = GL_RGBA; break;
	}
}

CTexture::~CTexture()
{
//	g_print("~CTexture\n");
	if (textureID != 0) {
		glDeleteTextures(1, &textureID);
		textureID = 0;
	}
}

void CTexture::reset()
{
	if (textureID != 0) {
		glDeleteTextures(1, &textureID);
		textureID = 0;
	}

	multiTexturingAvailable = false;
	pic_format = tex_format = GL_NONE;
	wrapS = wrapT = GL_REPEAT;
	minFilter = GL_NEAREST;
	magFilter = GL_NEAREST;
	build_mipmaps = false;
	texEnv = GL_NONE;
	lastError = 0;
}

/*      glTexImage2D(GL_TEXTURE_2D, 0, internal, width, height, 0, format,
       GL_UNSIGNED_BYTE, data);*/

bool CTexture::loadFile(const gchar *szFileName, PicType type, GLenum nf)
{
	bool result = CPicture::loadFile(szFileName, type);

	// Does file exist ?
	if (result) {
		switch (bpp) {
		case  8: tex_format = pic_format = GL_ALPHA; break;
		case 24: tex_format = pic_format = GL_RGB;   break;
		case 32: tex_format = pic_format = GL_RGBA;  break;
		}
	}
	else {
		tex_format = pic_format = GL_NONE;
		return result;
	}

	// Success
	return result;
}

bool CTexture::loadData(const gchar *data_p, const guint size, PicType type, GLenum nf)
{
	bool result = CPicture::loadData(data_p, size, type);

	// Does data exist ?
	if (result) {
		switch (bpp) {
		case 8: tex_format = pic_format = GL_ALPHA; break;
		case 24: tex_format = pic_format = GL_RGB; break;
		case 32: tex_format = pic_format = GL_RGBA; break;
		}
	}
	else {
		tex_format = pic_format = GL_NONE;
	}

	// Success
	return result;
}

bool CTexture::setRAW(gchar i_bpp, guint32 i_width, guint32 i_height)
{
	bool result = CPicture::setRAW(i_bpp, i_width, i_height);

	// Does data exist ?
	if (result) {
		switch (bpp) {
		case 8: tex_format = pic_format = GL_ALPHA; break;
		case 24: tex_format = pic_format = GL_RGB; break;
		case 32: tex_format = pic_format = GL_RGBA; break;
		}
	}
	else {
		tex_format = pic_format = GL_NONE;
	}

	// Success
	return result;

}

bool CTexture::realize (bool freePic)
{
	if (!(data && isTexSizeOk())) {
//		g_print("Error\n");
		return false;
	}

	if (textureID != 0) {
		glDeleteTextures(1, &textureID);
		textureID = 0;
	}

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
	glGenTextures(1, &textureID);
	if (glGetError() != GL_NO_ERROR) {
		g_print("ctexture.cpp [CTexture::realize] : Error - no texture generated\n");
	}

	glBindTexture(GL_TEXTURE_2D, textureID);

	// Texture parameters
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrapS);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrapT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, minFilter);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, magFilter);

	// Set texture environment
	if (texEnv != GL_NONE)
		{ glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, texEnv); }

	// Build mip-maps
	if (build_mipmaps)
		gluBuild2DMipmaps(GL_TEXTURE_2D, pic_format, width, height,
			tex_format, GL_UNSIGNED_BYTE, (const void*) data);
	else
		glTexImage2D(GL_TEXTURE_2D, 0, pic_format, width, height,
			0, tex_format, GL_UNSIGNED_BYTE, (const void*) data);

	if (freePic) {
//		g_print("freePic\n");
		this->CPicture::free();
	}

	return true;
}

// Make the texture the current texture
bool CTexture::bind (bool freePic)
{
	if (!textureID)
		if (!realize(freePic)) { return false; }
	glBindTexture(GL_TEXTURE_2D, textureID);
	return true;
}

// Set texture state to environment mapping
void CTexture::setEnvironmentMapping()
{
	// Set up texture coordinate generation
	glTexGenf(GL_S, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_NV);
	glTexGenf(GL_T, GL_TEXTURE_GEN_MODE, GL_REFLECTION_MAP_NV);

	// Turn on the coordinate generation
	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);
}

// Set texture state to sphere mapping
void CTexture::setSphereMapping()
{
	// Set up texture coordinate generation
	glTexGenf(GL_S, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);
	glTexGenf(GL_T, GL_TEXTURE_GEN_MODE, GL_SPHERE_MAP);

	// Turn on the coordinate generation
	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);
}

// Enable transparency
void CTexture::setTransparency()
{
//	if (HAS_ALPHA) { return; }

	// Enable blending
	glEnable(GL_BLEND);

	// Set correct blending mode
	glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_COLOR);
}

// Enable lightmapping
void CTexture::setLightmap()
{
	// Enable blending
	glEnable(GL_BLEND);

	// Set correct blending mode
	glBlendFunc(GL_DST_COLOR, GL_SRC_COLOR);
}

// Set up the automatic texture coord generation
void CTexture::genTexCoords()
{
	float fS[4] = { -0.1f, 0.0f, 0.0f, 0.0f };
	float fT[4] = { 0.0f, 0.0f, -0.1f, 0.0f };

	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);

	glTexGenfv(GL_S, GL_OBJECT_PLANE, fS);
	glTexGenfv(GL_T, GL_OBJECT_PLANE, fT);
	glTexGeni(GL_S, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
	glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);

	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);
}

// Init multitexturing
void CTexture::initMultitexturing(glActiveTextureARBFunc f1,
	glMultiTexCoord2fARBFunc f2,
	glMultiTexCoord4fARBFunc f3,
	glClientActiveTextureARBFunc f4)
{
	// Get function pointers
	m_glActiveTextureARB = f1;
	m_glMultiTexCoord2fARB = f2;
	m_glMultiTexCoord4fARB = f3;
	m_glClientActiveTextureARB = f4;

	// Verify pointers
	if (!m_glActiveTextureARB || !m_glMultiTexCoord2fARB ||
		!m_glMultiTexCoord4fARB || !m_glClientActiveTextureARB)
	{
		// Can't initialize multitexturing
		multiTexturingAvailable = false;
		return;
	}

	// Activate texturing, leave first TMU activated
	m_glActiveTextureARB(GL_TEXTURE1_ARB);
	glEnable(GL_TEXTURE_2D);
	m_glActiveTextureARB(GL_TEXTURE0_ARB);
	glEnable(GL_TEXTURE_2D);

	// Success
	multiTexturingAvailable = true;
}

// Chose the active texturing unit
bool CTexture::setActiveTMU(GLenum target)
{
	if (!multiTexturingAvailable) { return false; }

	m_glActiveTextureARB(target);

	return true;
}

// Specify texture coordinate for the given texturing unit
bool CTexture::multiTexCoord2f(GLenum target, GLfloat s, GLfloat t)
{
	if (!multiTexturingAvailable) { return false; }

	m_glMultiTexCoord2fARB(target, s, t);

	return true;
}

// Specify texture coordinate for the given texturing unit
bool CTexture::multiTexCoord4f(GLenum target, GLfloat s, GLfloat t, GLfloat r, GLfloat q)
{
	if (!multiTexturingAvailable) { return false; }

	m_glMultiTexCoord4fARB(target, s, t, r, q);

	return true;
}

// Chose texture environment combine operation
void CTexture::setCombineOp(float fOperation)
{
	// Combine mode
	glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_COMBINE_EXT);

	// Operation
	glTexEnvf (GL_TEXTURE_ENV, GL_COMBINE_RGB_EXT, fOperation);
}

GLenum CTexture::getFormat(bool texture) { return (texture ? tex_format : pic_format); }

bool CTexture::isTexSizeOk()
{
	guint32 power = 0;
	bool result[2] = { false };

	for (guint32 i = 0; (power = (guint32)pow(2, i)) <= maxTexSize; i++) {
		if (!result[0])
			if (width == power) { result[0] = true; }
		if (!result[1])
			if (height == power) { result[1] = true; }
		if (result[0] && result[1]) { return true; }
	}
	return false;
}

bool CTexture::setTexFormat (GLenum f)
{
	if (!textureID) {
		tex_format = f;
		return true;
	}
	return false;
}
