/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "cscene.h"

#include "maths.h"
#include "data.h"

extern gint16 static_pcm_data[2][512];
extern gint16 static_freq_data[2][256];

CScene::CScene()
{
	if (logo_texture.loadData(data_xmms_rgb, -1, sgi))
		{ logo_texture.realize(TRUE); }
	else { g_print("Error\n"); }

	if (growbar_texture.loadData(data_growbar_rgb, -1, sgi))
		{ growbar_texture.realize(TRUE); }
	else { g_print("Error\n"); }

	if (mesh_texture.loadData(data_blur_mesh_rgb, -1, sgi))
		{ mesh_texture.realize(TRUE); }
	else { g_print("Error\n"); }

	generate_pcm_texture();

/*	g_print("%i / %i\n", logo_texture.getWidth(), logo_texture.getHeight());
	g_print("%i / %i\n", growbar_texture.getWidth(), growbar_texture.getHeight());
	g_print("%i / %i\n", mesh_texture.getWidth(), mesh_texture.getHeight());
	g_print("%i / %i\n", pcm_texture.getWidth(), pcm_texture.getHeight());*/

	build_world_rings();
}

CScene::~CScene()
{
	glDeleteLists(world_gl_list, 1);
}

void CScene::draw_pcm(void)
{
	glShadeModel(GL_SMOOTH);

	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);

	pcm_texture.bind();

	glPushMatrix();

	glBegin(GL_TRIANGLES);
	for (int j = 0;j != 2;j++) {
		const int skip = 4;

		float y;
		float dy;
		y = j;
		y -= 0.5f;
		dy = y;
		y *= 1.0f;
		dy /= 20.0f;

		for (int i = 0;i != 512;i += skip) {
			float u;

			y += skip * dy;
			u = static_pcm_data[j][i];
			u /= 16384 / 8;
			u = (u < 0.0f) ? -u : u;
			u += 0.8f;

			glColor4f(1.0f, 0.0f, 0.0f, 1.0f);
			glTexCoord2f(0.5f,0.0f);
			glVertex3f(0.0f, y, -u);
			glColor4f(0.0f, 1.0f, 0.0f, 1.0f);
			glTexCoord2f(0.0f,1.0f);
			glVertex3f(-u, y, u);
			glColor4f(0.0f, 0.0f, 1.0f, 1.0f);
			glTexCoord2f(1.0f,1.0f);
			glVertex3f(u, y, u);
		}
	}
	glEnd();

	glPopMatrix();

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
}

void CScene::generate_pcm_texture()
{
	gchar *texture_image_p = new gchar[4 * 32 * 32];
	const int wh = 32;

	for (int v = 0;v != wh;v++) {
		for (int u = 0;u != wh;u++) {
			float x,y;
			x = u - wh/2;
			y = v - wh/2;
			float a,c;

			a = 255.0f - 32.0f * hypot(x,y);
			if (a < 0.0f) {
			    a = 0.0f;
			}
			c = a;

			*(texture_image_p + 0 + u + v*wh) = (gchar)c;
			*(texture_image_p + 1 + u + v*wh) = (gchar)c;
			*(texture_image_p + 2 + u + v*wh) = (gchar)c;
			*(texture_image_p + 3 + u + v*wh) = (gchar)a;
		}
	}
	pcm_texture = CTexture(32, wh, wh, (const gchar*) texture_image_p);
	delete [] texture_image_p;
}

void CScene::construct_ring(int elements, float ur, float dr, float h2, float *rgba_p)
{
	glShadeModel(GL_FLAT);

	glBegin(GL_QUADS);

	glColor4fv(rgba_p);

	fix16_16 fix_rad = 0;

	float bx = Sin(fix_rad);
	float bz = Cos(fix_rad);

	for (int i = 0;i != elements;i++)
	{
		fix_rad += ANGLE_2PI / elements;
		float nx = Sin(fix_rad);
		float nz = Cos(fix_rad);

		glTexCoord2f(0.0f, 0.4f);
		glVertex3f(ur * bx, -h2, ur * bz);

		glTexCoord2f(1.0f, 0.4f);
		glVertex3f(dr * bx,  h2, dr * bz);

		glTexCoord2f(1.0f, 0.6f);
		glVertex3f(dr * nx,  h2, dr * nz);

		glTexCoord2f(0.0f, 0.6f);
		glVertex3f(ur * nx, -h2, ur * nz);

		bx = nx;
		bz = nz;
	}

	glEnd();
}

void CScene::build_world_rings(void)
{
	enum {
		ELEMENTS = 6
	};

	float rgba[4] = { 1.0f, 1.0f, 1.0f, 0.5f };

	float angle[ELEMENTS][4] =
	{ {  60, 0, 0, 1 },
	  {  60, 0, 1, 0 },
	  {  60, 1, 0, 0 },
	  { 120, 0, 0, 1 },
	  { 120, 0, 1, 0 },
	  { 120, 1, 0, 0 } };

	world_gl_list = glGenLists(1);

	glPushMatrix();

	glNewList(world_gl_list, GL_COMPILE);
	glEnable(GL_BLEND);
	for (int i = 0; i != ELEMENTS; i++)
	{
		glPushMatrix();

		glRotatef(angle[i][0],
			  angle[i][1],
			  angle[i][2],
			  angle[i][3]);
		construct_ring(32,
				27.0f,
				27.0f,
				0.1f,
				rgba);
		glPopMatrix();
	}
	glDisable(GL_BLEND);
	glEndList();

	glPopMatrix();
}

void CScene::draw_world_rings(bool animate)
{
	static float full_angle = 30.0f;

	glEnable(GL_BLEND);
//	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	glPushMatrix();

	glRotatef(full_angle, Sin(full_angle), Cos(full_angle), full_angle);

	if (animate)
		{ full_angle += 0.05f; }

	glCallList(world_gl_list);

	glPopMatrix();

	glDisable(GL_BLEND);
}

void CScene::draw_bg_mesh(void)
{
	enum {
		 VX,  VY,  VZ,  VW, // Punkt
		 VR,  VG,  VB,  VA, // Farbe
		VNX, VNY, VNZ, VNW, // Normale

		VTX_ELEMENTS
	};
	enum {
		ELEMENTS = 128
	};
	static int mesh_init_flag;
	static float vtx[4 * VTX_ELEMENTS * ELEMENTS];
	static fix16_16 alpha_radian[4 * ELEMENTS];

	const float u = 16.0f;
	const float r = 28.0f;

	// Nur einmal initialisieren
	if (mesh_init_flag == 0) {

		// Gesamtgroesse des Raumes
		float src[] = {
			-u, u, r, 1.0f,
			 u, u, r, 1.0f,
			 u,-u, r, 1.0f,
			-u,-u, r, 1.0f
		};
		CMatrix m;

		for (int i = 0; i < ELEMENTS; i++) {
			// Beliebige Rotations-Matrix
			m.Unit();
			m.MulRotX(RandFloat() * PI2);
			m.MulRotY(RandFloat() * PI2);
			m.MulRotZ(RandFloat() * PI2);

			// Ein Quadrat positionieren
			for (int h = 0; h < 4; h++) {
				// Punkt "setzen"
				m.MulVector(&src[h*4], &vtx[VX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				// Normale bestimmen
				GetOuterProduct(&vtx[VX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS],
					&vtx[VNX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				UnitVector(&vtx[VNX + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
				// Farbe bestimmen
				vtx[VR + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VG + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VB + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = RandFloat();
				vtx[VA + h*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] = 0.0f;
				alpha_radian[h + i*4] = Rand(ANGLE_2PI);
			}
		}
		mesh_init_flag = 1;
	}

	glPushMatrix();

	glShadeModel(GL_SMOOTH);

	glEnable(GL_BLEND);
	glEnable(GL_TEXTURE_2D);

	glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
	mesh_texture.bind();

	glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
	glTexGeni(GL_T,GL_TEXTURE_GEN_MODE,GL_SPHERE_MAP);
	glEnable(GL_TEXTURE_GEN_S);
	glEnable(GL_TEXTURE_GEN_T);

	glBegin(GL_QUADS);
	for (int i = 0;i != ELEMENTS;i++)
	{
		vtx[VA + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[0 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[0 + i*4] += ANGLE_2PI/512;
		alpha_radian[0 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(0.0f,1.0f);
		glNormal3fv(&vtx[VNX + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 0*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[1 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[1 + i*4] += ANGLE_2PI/256;
		alpha_radian[1 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(1.0f,1.0f);
		glNormal3fv(&vtx[VNX + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 1*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[2 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[2 + i*4] += ANGLE_2PI/512;
		alpha_radian[2 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(1.0f,0.0f);
		glNormal3fv(&vtx[VNX + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 2*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);

		vtx[VA + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS] =
			(Sin(alpha_radian[3 + i*4]) + 1.0f) / 2.0f;
		alpha_radian[3 + i*4] += ANGLE_2PI/128;
		alpha_radian[3 + i*4] &= ANGLE_MASK;

		glColor4fv(&vtx[VR + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glTexCoord2f(0.0f,0.0f);
		glNormal3fv(&vtx[VNX + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
		glVertex3fv(&vtx[VX + 3*VTX_ELEMENTS + i * 4*VTX_ELEMENTS]);
	}
	glEnd();

	glDisable(GL_TEXTURE_GEN_S);
	glDisable(GL_TEXTURE_GEN_T);

	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);

	glPopMatrix();
}


void CScene::draw_xmms_logo(bool animate)
{
	/* Funtions On */
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);

	/* Options */
//	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);
	glTexEnvf(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);

	glShadeModel(GL_FLAT);

	logo_texture.bind();

	glPushMatrix();

	glTranslatef(0.0f, -8.0f, 0.0f);

	const float w2 = 16.0f;
	const float h2 = 16.0f;

	static fix16_16 radian;

	int times = (int)(Sin(radian) * 16.0f);
	if (times < 1) { times = 1; }

	if (animate) {
		radian += ANGLE_2PI / 512;
		radian &= ANGLE_MASK;
	}
	float y = 0;
	float a = 1.0f;

	glBegin(GL_QUADS);
	for (int i = 0;i != times;i++)
	{
		glColor4f(1.0f, 1.0f, 1.0f, a);

		glTexCoord2f(0.0f,0.0f);
		glVertex3f(w2, y, -h2);

		glTexCoord2f(1.0f,0.0f);
		glVertex3f(-w2, y, -h2);

		glTexCoord2f(1.0f,1.0f);
		glVertex3f(-w2, y, h2);

		glTexCoord2f(0.0f,1.0f);
		glVertex3f(w2, y, h2);

		a *= 0.54f;
		y += 0.07f;
	}

	glEnd();

	glPopMatrix();

	/* Funtions Off */
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_BLEND);
}

void CScene::draw(guint time, bool animate)
{
	bool run_animation = animate && !lyric_text_engine.have_song();

	glDepthMask(GL_TRUE);

//	glClearColor(0.1, 0.1, 0.45, 0.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Save before rotate state for the song text position
	glPushMatrix();

	if (run_animation) { camera.rotate(); }
	camera.lookAt();


/*
	opaque
*/
	draw_world_rings(run_animation);

/*
	transparent
*/
	glDepthMask(GL_FALSE);

	draw_bg_mesh();

	draw_xmms_logo(run_animation);

	if (!lyric_text_engine.have_song())
		draw_pcm();

	// Restore state for the song text position
	glPopMatrix();


/*
	opaque
*/
	glEnable(GL_DEPTH_TEST);
	glDepthMask(GL_TRUE);

	lyric_text_engine.draw(time);
}
