
#include "PixPort.h"

#include <math.h>
#include "RectUtils.h"

#include "WindowDevice.h"
#include "FrameBuffer.h"
#include "CEgIOFile.h"

#ifdef EG_X
#include "libmfl.h"
#endif

#ifdef EG_SDL

#include <SDL/SDL_error.h>

#define __setupPort \
	if (mBM && SDL_MUSTLOCK(((SDL_Surface*) mBM))) { \
		if ( SDL_LockSurface((SDL_Surface*) mBM) < 0 ) { \
			fprintf(stderr, "Can't lock video surface: %s\n", SDL_GetError()); \
		} \
	}

#define __restorePort \
	if (mBM && SDL_MUSTLOCK(((SDL_Surface*) mBM)) ) { \
		SDL_UnlockSurface(((SDL_Surface*) mBM)); \
	}
#endif

#if EG_MAC
#include <QuickDraw.h>
#include <QDOffscreen.h>
#include <Displays.h>
#include <string.h>
#include <Palettes.h>

#define __setupPort \
	PP_GDHandle		saveDev;		\
 	PP_GWorldPtr		savePort;		\
	::GetGWorld( &savePort, &saveDev );		\
	if ( mWorld )					\
		::SetGWorld( mWorld, nil );

#define __restorePort \
	::SetGWorld( savePort, saveDev );

#endif

#if EG_WIN
#define __setupPort
#define __restorePort
#endif

long		PixPort::sTempSize			= 0;
char*		PixPort::sTemp				= 0;

PixPort::PixPort() {

	mBM = nil;
	mWorld = nil;
	mBounds.left = mBounds.top = 0;
	mBounds.bottom = mBounds.right = 0;
	mLineWidth = 1;
	mBackColor	= 0;
	mDepth = 0;
	mCurFontID = 0;

#if EG_WIN
	mWorld		= ::CreateCompatibleDC( nil );
	mBM		= nil;
#endif

	// We do this to make calls to SetFont() and TextRect() possible before the user calls Init()
#if EG_MAC
	Init( 1, 1, 8 );
#endif

#if EG_X
	// We need a font context to assign the new font,
	// even if we don't have a initilaized pixport
	mWorld = (void*) mfl_CreateContext(NULL, 0, 0, 0, 0);
#endif

#if WIN_QUICKTIME_PRESENT
	mQTML_World = nil;
#endif
}



PixPort::~PixPort() {

	Init();

#if EG_WIN
	::SelectObject( mWorld, ::GetStockObject( SYSTEM_FONT ) );

	if ( mWorld )
		::DeleteDC( mWorld );

	// Dealloc the offscreen bitmap we made
	if ( mBM )
		::DeleteObject( mBM );
#endif

#ifdef EG_X
	if (mWorld) {
		mfl_DestroyContext( (mfl_context) mWorld);
		mWorld = 0;
	}

	if ( mBM )
		SDL_FreeSurface(mBM);
#endif


	if ( sTemp ) {
		delete []sTemp;
		sTemp = nil;
		sTempSize = 0;
	}
}

void PixPort::Init() {

#if EG_MAC
	if ( mWorld ) {
		::UnlockPixels( mBM );
		::DisposeGWorld( mWorld );
		mWorld = nil;
		mBM = nil;
	}
#endif

#if WIN_QUICKTIME_PRESENT
	if ( mQTML_World ) {
		::DisposeGWorld( mQTML_World );
		mQTML_World = nil;
	}
#endif

#ifdef EG_X
	if ( mBM ) {
		SDL_FreeSurface( mBM );
		mBM = NULL;
	}
#endif

	mCurFontID = 0;
	mCurPaletteSeed = -1;
}

void PixPort::SetClipRect( const Rect* inRect ) {

	mClipRect = mBounds;

	if ( inRect )
		IntersectRect( inRect, &mClipRect, &mClipRect );

	if ( mWorld ) {
		__setupPort

		#if EG_MAC
		::ClipRect( &mClipRect );
		#endif

		#if EG_WIN
		HRGN rgn = ::CreateRectRgn( mClipRect.left, mClipRect.top, mClipRect.right, mClipRect.bottom );
		::SelectObject( mWorld, rgn );
		::DeleteObject( rgn );
		#endif

		__restorePort
	}
}

void PixPort::SetClipRect( long inSX, long inSY, long inEX, long inEY ) {

	Rect r;

	SetRect( &r, inSX, inSY, inEX, inEY );
	SetClipRect( &r );
}

void PixPort::Init( int inWidth, int inHeight, int inDepth ) {

	long oldFontID = mCurFontID;

	if ( inWidth < 1 )		inWidth = 1;
	if ( inHeight < 1 )		inHeight = 1;

	// Catch any invalid depth levels.
	if ( inDepth != 32 && inDepth != 16 && inDepth != 8 )
		inDepth = 32;

	// If we don't need to do anything, then don't do anything!
	if ( mWorld && mDepth == inDepth && inWidth == mBounds.right && inHeight == mBounds.bottom )
		return;

	mBounds.right	= inWidth;
	mBounds.bottom	= inHeight;
	inWidth = 4 * (( inWidth + 3 ) / 4 );

	Init();


#if EG_MAC
	// Save current draw envir
	__setupPort

	Rect r = mBounds;
	r.bottom += 2;
	::NewGWorld( &mWorld, inDepth, &r, nil, nil, useTempMem );
	mBM = ::GetGWorldPixMap( mWorld );
	mBytesPerRow	= (**mBM).rowBytes & 0x1FFF;
	mDepth			= (**mBM).pixelSize;
	::LockPixels( mBM );
	mBits = ::GetPixBaseAddr( mBM );

	__restorePort

#elif EG_WIN
	// Initialize a bmap info struct.  Negative height means origin at top-left corner
	mInfo.bmiHeader.biSize		= sizeof( BITMAPINFOHEADER );
	mInfo.bmiHeader.biWidth		= inWidth;
	mInfo.bmiHeader.biHeight	= - ( mBounds.bottom + 2 );
	mInfo.bmiHeader.biPlanes	= 1;
	mInfo.bmiHeader.biBitCount	= inDepth;
	mInfo.bmiHeader.biCompression	= BI_RGB;
	mInfo.bmiHeader.biSizeImage	= 0;
	mInfo.bmiHeader.biXPelsPerMeter	= 0;
	mInfo.bmiHeader.biYPelsPerMeter = 0;
	mInfo.bmiHeader.biClrUsed	= 0;
	mInfo.bmiHeader.biClrImportant	= 0;

	// Tell windows to make a bitmap and give us acess to its pixel data
	mBM = ::CreateDIBSection( mWorld, &mInfo, DIB_RGB_COLORS, (void**) &mBits, nil, 0 );
	HGDIOBJ oldBM = ::SelectObject( mWorld, mBM );
	if ( oldBM )
		::DeleteObject( oldBM );

	BITMAP b;
	::GetObject( mBM, sizeof( BITMAP ), &b );
	mBytesPerRow	= b.bmWidthBytes;
	mDepth		= b.bmBitsPixel;

	::SetTextAlign( mWorld, TA_BASELINE | TA_LEFT );
	::SetBkMode( mWorld, TRANSPARENT );
	if ( mDepth == 8 ) {
		RGBQUAD* c = new RGBQUAD[ 256 ];
		for ( int i = 0; i < 256; i++ )
			((long*) c )[ i ] = RGB( i, i, i );
		::SetDIBColorTable( mWorld, 0, 256, c );
		delete []c;
	}
	::SetTextColor( mWorld, RGB( 255, 255, 255 ) );

#elif EG_X && EG_SDL
	if (mBM != NULL)
		SDL_FreeSurface((SDL_Surface*) mBM);
	mBM = SDL_CreateRGBSurface(SDL_SWSURFACE, inWidth,
		mBounds.bottom + 2, inDepth, 0, 0, 0, 0);
	mBytesPerRow = mBM->pitch;
	mDepth = mBM->format->BitsPerPixel;
	mBits = (char*) mBM->pixels;

	// We want to keep the font so just set a new "target"
	mfl_UpdateContext((mfl_context) mWorld, mBM->pixels,
		mBM->format->BitsPerPixel, mBM->pitch, mBM->w, inHeight);
#endif
	SetClipRect();
	EraseRect();

	SetFont( oldFontID );
}

#define __clipPt( x, y )	\
	if ( x < mClipRect.left )			\
		x = mClipRect.left;				\
	else if ( x > mClipRect.right )		\
		x = mClipRect.right;			\
	if ( y < mClipRect.top )			\
		y = mClipRect.top;				\
	else if ( y > mClipRect.bottom )	\
		y = mClipRect.bottom;



#define __clipRect( inRect )			\
	Rect r = inRect;					\
	__clipPt( r.left, r.top )			\
	__clipPt( r.right, r.bottom )		\
	long width 	= r.right - r.left;		\
	long height = r.bottom - r.top;


long PixPort::GetPortColor( long inR, long inG, long inB ) {

	long c;

	if ( inR > 0xFFFF )	inR = 0xFFFF;
	if ( inG > 0xFFFF )	inG = 0xFFFF;
	if ( inB > 0xFFFF )	inB = 0xFFFF;
	if ( inR < 0 )		inR = 0;
	if ( inG < 0 )		inG = 0;
	if ( inB < 0 )		inB = 0;

	if ( mDepth == 32 )
		c = __RGBTo32( inR, inG, inB );
	else if ( mDepth == 16 )
		c = __RGBTo16( inR, inG, inB );
	else
		c = __RGBTo8( inR, inG, inB );

	return c;
}

long PixPort::SetBackColor( const RGBColor& RGB ) {

	mBackColor = GetPortColor( RGB );

	return mBackColor;
}

long PixPort::SetBackColor( long inR, long inG, long inB ) {

	mBackColor = GetPortColor( inR, inG, inB );

	return mBackColor;
}

void PixPort::SetPalette( PixPalEntry inPal[ 256 ], long inSeed ) {

	if ( mDepth == 8 && mCurPaletteSeed != inSeed ) {
		// Tell the offscreen world a new color table exists

		mCurPaletteSeed = inSeed;

#if EG_WIN
		::SetDIBColorTable( mWorld, 0, 256, inPal );
#elif EG_MAC
		CTabHandle offscreenTable = (**mBM).pmTable;
		::BlockMove( inPal, (**offscreenTable).ctTable, 256 * sizeof( ColorSpec ) );
		::CTabChanged( offscreenTable );
		if ( inSeed != PP_DONT_SET_OS_SEED )
			(**offscreenTable).ctSeed = inSeed;
#elif EG_SDL
		SDL_SetPalette(mBM, (SDL_LOGPAL | SDL_PHYSPAL), (SDL_Color*) inPal, 0, 256);
#endif
	}
}

void PixPort::GaussBlur( int inBoxWidth, const Rect& inRect, void* inDestBits ) {

	// Don't let us draw in random parts of memory -- clip inRect
	__clipRect( inRect )

	if ( inBoxWidth <= 1 )
		return;

	// 3 box convolutions, 3 colors per pixel, 4 bytes per color
	long 	boxTempSize	= 36 * inBoxWidth;
	char*	tempBits	= nil;
	unsigned long*	boxTemp;
	long	imgOffset	= mDepth * r.left / 8 + r.top * mBytesPerRow;
	long	bytesNeeded	= mBytesPerRow * (mBounds.bottom + 2) + boxTempSize;

	// Resort to app's heap for temp mem if failed temp mem attempt or in win32
	tempBits = mBlurTemp.Dim( bytesNeeded );

	// Have the box temp and the pixel temp rgns use the same handle
	boxTemp = (unsigned long*) tempBits;
	tempBits += boxTempSize;

	if ( ! inDestBits )
		inDestBits = mBits;

	// Do a box blur on the x axis, transposing the source rgn to the dest rgn
	// Then o a box blur on the transposed image, effectively blurring the y cords, transposing it to the dest
	if ( mDepth == 16 )  {
		BoxBlur16( ( mBits + imgOffset), tempBits, inBoxWidth, width, height, mBytesPerRow, mDepth * height / 8, boxTemp, mBackColor );
		BoxBlur16( tempBits, ((char*) inDestBits + imgOffset), inBoxWidth, height, width, mDepth * height / 8, mBytesPerRow, boxTemp, mBackColor );  }
	else if ( mDepth == 32 ) {
		BoxBlur32( ( mBits + imgOffset), tempBits, inBoxWidth, width, height, mBytesPerRow, mDepth * height / 8, boxTemp, mBackColor );
		BoxBlur32( tempBits, ((char*) inDestBits + imgOffset), inBoxWidth, height, width, mDepth * height / 8, mBytesPerRow, boxTemp, mBackColor );
	}
}

void PixPort::CrossBlur( const Rect& inRect ) {

	// Don't let us draw in random parts of memory -- clip inRect
	__clipRect( inRect )

	// 3 box convolutions, 3 colors per pixel, 4 bytes per color
	long	imgOffset	= mDepth * r.left / 8 + r.top * mBytesPerRow;

	unsigned char* tempBits = (unsigned char*) mBlurTemp.Dim( mBounds.right * 3 );


	if ( mDepth == 16 )
		CrossBlur16( ( mBits + imgOffset), width, height, mBytesPerRow, tempBits );
	else if ( mDepth == 32 )
		CrossBlur32( ( mBits + imgOffset), width, height, mBytesPerRow, tempBits );
}

void PixPort::CopyBits( PP_GrafPtr inPort, const Rect* inSrce, const Rect* inDest ) {

	if (	inSrce -> left <= inSrce -> right && inSrce -> top <= inSrce -> bottom &&
			inDest -> left <= inDest -> right && inDest -> top <= inDest -> bottom )
	{
#if EG_MAC
#if TARGET_API_MAC_CARBON
		::CopyBits( (BitMap*) *mBM, GetPortBitMapForCopyBits(  inPort), inSrce, inDest, srcCopy, nil );
#else
		::CopyBits( (BitMap*) *mBM, &inPort->portBits, inSrce, inDest, srcCopy, nil );
#endif
#endif

#if EG_WIN
		::BitBlt( inPort, inDest -> left, inDest -> top, inDest -> right - inDest -> left, inDest -> bottom - inDest -> top, mWorld, inSrce -> left, inSrce -> top, SRCCOPY );
#endif
	}
}

void PixPort::CopyBits( PixPort& inDestPort, const Rect* inSrce, const Rect* inDest ) {

	if (	inSrce -> left <= inSrce -> right && inSrce -> top <= inSrce -> bottom &&
			inDest -> left <= inDest -> right && inDest -> top <= inDest -> bottom ) {

#if EG_MAC
		if ( mDepth == 8 ) {
			// Below prevents MacOS from reindexing the colors in our GWorld during CopyBits, thinking the colors in our GWorld
			// index to the standard system colors.  It's a hack, but then again, so is MacOS 8.x
			CTabHandle destTable = (**inDestPort.mBM).pmTable;
			(**destTable).ctSeed = (**(*mBM) -> pmTable).ctSeed;
		}

		::CopyBits( (BitMap*) *mBM, (BitMap*) *inDestPort.mBM, inSrce, inDest, srcCopy, nil );
#elif EG_WIN
		::BitBlt( inDestPort.mWorld, inDest -> left, inDest -> top, inDest -> right - inDest -> left, inDest -> bottom - inDest -> top, mWorld, inSrce -> left, inSrce -> top, SRCCOPY );
#endif
	}
}

#if EG_MAC || EG_X
#define BI_RGB        0L
#define BI_RLE8       1L
#define BI_RLE4       2L
#define BI_BITFIELDS  3L
#define BI_JPEG       4L
#define BI_PNG        5L
#endif

void PixPort::WriteImage( const CEgFileSpec& inSpec ) {

	long mod_Depth = ( mDepth == 32 ) ? 24 : mDepth;
	long size, start, safeX, logicalRowSize;
	CEgIOFile oFile;
	long x, y;
	unsigned long* row;

	// We must choose a width such that the byte length of each row is dbl-word aligned
	// We do this b/c many image viewers behave differently if the row len isn't a multiple of 4.
	safeX = GetX();
	logicalRowSize = safeX * ( mod_Depth / 8 );
	while ( logicalRowSize % 4 != 0 ) {
		safeX--;
		logicalRowSize = safeX * ( mod_Depth / 8 );
	}

	oFile.open( &inSpec );

	// BITMAPFILEHEADER
	oFile.CEgOStream::skip( 14 );

	// BITMAPINFOHEADER
	oFile.PutLong( 40 );										// biSize
	oFile.PutLong( safeX );										// biWidth
	oFile.PutLong( GetY() );									// biHeight
	oFile.PutShort( 1 );										// biPlanes
	oFile.PutShort( mod_Depth );								// biBitCount
	oFile.PutLong( BI_RGB );									// biCompression
	oFile.PutLong( safeX * logicalRowSize );	// biSizeImage
	oFile.PutLong( 0x0EC4 );									// biXPelsPerMeter
	oFile.PutLong( 0x0EC4 );									// biYPelsPerMeter
	oFile.PutLong( 0 );											// biClrUsed
	oFile.PutLong( 0 );											// biClrImportant

	// Write the palette if in 8 bit mode (256 RGBQUADs)...
	if ( mDepth == 8 ) {

#if EG_MAC
		CTabHandle offscreenTable = (**GetPixMap()).pmTable;
		for ( int i = 0; i < 256; i++ )
			oFile.PutLong( __REFTo32( (**offscreenTable).ctTable[ i ].rgb ) );
#endif

#if EG_WIN
		long pal[ 256 ];
		::GetDIBColorTable( GetDC(), 0, 256, (RGBQUAD*) pal );
		oFile.PutBlock( pal, 256 * 4 );
#endif
	}

	// The hdr needs to know the start of the bits...
	start = oFile.tell();

	if ( mDepth == 8 || mDepth == 16 ) {

		// Write the bits...
		for ( y = GetY(); y > 0; y-- ) {
			oFile.PutBlock( GetRow( y - 1 ), logicalRowSize );
		} }

	else if ( mDepth == 32 ) {

		for ( y = GetY(); y > 0; y-- ) {
			for ( x = 0, row = (unsigned long*) GetRow( y - 1 ); x < safeX; x++ ) {
				oFile.PutByte( ( row[ x ] >>  0 ) & 0xFF );
				oFile.PutByte( ( row[ x ] >>  8 ) & 0xFF );
				oFile.PutByte( ( row[ x ] >> 16 ) & 0xFF );
			}
		}
	}

	// Go back and fill bmp header info
	size = oFile.tell();

	// BITMAPFILEHEADER
	oFile.seek( 0 );
	oFile.PutShort( MCC2_TO_INT("MB") );	// bfType
	oFile.PutLong( size ); 		// bfSize
	oFile.PutLong( 0 );		// reserved
	oFile.PutLong( start );		// bfOffBits

	oFile.close();
}



/*
void PixPort::CopyBits( FrameBuffer& inBuffer, Rect& inSrce, int inDepth ) {
	Rect r;

	SectRect( &mBounds, &inSrce, &r );
	crap
	if ( IsValidRect( r ) ) {

		// *** Note this calc may be need to be fixed for windoze b/c y's are flipped
		char* bits = mBits + mBytesPerRow * r.top + r.left;
		int x = r.right - r.left;
		int y = r.bottom - r.top;

		if ( inDepth == 1 )
			inBuffer.Assign1Bit( bits, x, y, mBytesPerRow, mDepth );
		else if ( inDepth == 8 )
			inBuffer.Assign8Bit( bits, x, y, mBytesPerRow, mDepth );
		else
			inBuffer.Assign( bits, x, y, mBytesPerRow, mDepth );
	}
}
*/

void PixPort::Line( int sx, int sy, int ex, int ey, long inColor ) {

	if ( mDepth == 16 )
		Line16( sx, sy, ex, ey, inColor );
	else if ( mDepth == 8 )
		Line8 ( sx, sy, ex, ey, inColor );
	else if ( mDepth == 32 )
		Line32( sx, sy, ex, ey, inColor );
}


#define __ABS( n )  ( ( n > 0 ) ? (n) : (-n) )
#define CLR_LINE_THR	520

void PixPort::Line( int sx, int sy, int ex, int ey, const RGBColor& inS, const RGBColor& inE ) {

	long R, G, B, dR, dG, dB;

	R = inS.red;
	G = inS.green;
	B = inS.blue;
	dR = inE.red - R;
	dG = inE.green - G;
	dB = inE.blue - B;

	// If the endpoints have the same color, run the faster line procs (that just use one color)
	if (	dR > - CLR_LINE_THR && dR < CLR_LINE_THR &&
			dG > - CLR_LINE_THR && dG < CLR_LINE_THR &&
			dB > - CLR_LINE_THR && dB < CLR_LINE_THR ) {
		long color;

		if ( mDepth == 16 ) {
			color = __RGBTo16( R, G, B );
			Line16( sx, sy, ex, ey, color ); }
		else if ( mDepth == 32 ) {
			color = __RGBTo32( R, G, B );
			Line32( sx, sy, ex, ey, color ); }
		else if ( mDepth == 8 ) {
			color = __RGBTo8( R, G, B );
			Line8 ( sx, sy, ex, ey, color );
		} }
	else {
		if ( mDepth == 16 )
			Line16( sx, sy, ex, ey, inS, dR, dG, dB );
		else if ( mDepth == 32 )
			Line32( sx, sy, ex, ey, inS, dR, dG, dB );
		else if ( mDepth == 8 )
			Line8 ( sx, sy, ex, ey, inS.red, dR );
	}
}

long PixPort::SetFont( const char* inFontName, long inSize, long inStyleFlags ) {

	PixTextStyle* font = new PixTextStyle;
	PixTextStyle* storedFont;

	font -> mFontName.Assign( inFontName );
	font -> mPointSize			= inSize;
	font -> mStyle				= inStyleFlags;
	font -> mOSStyle			= 0;

	if ( mFontTable.Get( font, (void**) &storedFont ) ) {

		// Cleanup the key we made to look in the font table
		delete font;

		SetFont( (long) storedFont );   }
	else {

#if EG_MAC
		short fontNum;
#if TARGET_API_MAC_CARBON
		fontNum = ::FMGetFontFamilyFromName( font -> mFontName.getPasStr() );
#else
		::GetFNum( font -> mFontName.getPasStr(), &fontNum );
#endif
		font -> mOSFontID = fontNum;
		if ( font -> mStyle & PP_BOLD )
			font -> mOSStyle |= bold;
		if ( font -> mStyle & PP_ITALIC )
			font -> mOSStyle |= italic;
		if ( font -> mStyle & PP_UNDERLINE )
			font -> mOSStyle |= underline;
#endif // EG_MAC

#if EG_WIN
		long height = - MulDiv( inSize, ::GetDeviceCaps( mWorld, LOGPIXELSY ), 72 );
		font -> mFontName.Keep( 31 );
		font -> mOSFontID = (long) ::CreateFont( height, 0, 0, 0,
			( inStyleFlags & PP_BOLD ) ? FW_BOLD : FW_NORMAL,
			( inStyleFlags & PP_ITALIC ) ? true : false,
			( inStyleFlags & PP_UNDERLINE ) ? true : false,
			 0, DEFAULT_CHARSET,
			OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
			DEFAULT_PITCH | FF_DONTCARE, font -> mFontName.getCStr());
#endif // EG_WIN

#if EG_X
		font -> mOSFontID = (long) mfl_LoadRawFont(inFontName);
		font -> mAscent = 0;
		font -> mDescent = 0;
		font -> mLineHeight = 8;
#endif

		mFontTable.PutOwned( font, font, HT_NO_FLAGS );
		storedFont = font;

		// SetFont( long ) does all the work of setting the port text style, etc
		SetFont( (long) storedFont );


#if EG_MAC
		__setupPort

		FontInfo fontInfo;
		::GetFontInfo( &fontInfo );
		font -> mAscent = fontInfo.ascent;
		font -> mDescent = fontInfo.descent;
		font -> mLineHeight = font -> mAscent + font -> mDescent + fontInfo.leading;

		__restorePort
#endif

#if EG_WIN
		TEXTMETRIC fontInfo;
		::GetTextMetrics( mWorld, &fontInfo );
		font -> mAscent = fontInfo.tmAscent;
		font -> mDescent = fontInfo.tmDescent;
		font -> mLineHeight = fontInfo.tmHeight; // !!
#endif
	}

	return (long) storedFont;
}

void PixPort::SetFont( long inPixFontID ) {

	// Exit if we're already in in this text face
	if ( inPixFontID == mCurFontID || inPixFontID == 0 )
		return;

	PixTextStyle* font = (PixTextStyle*) inPixFontID;

	// Exit if the incoming ID isn't something this port created
	//if ( mFonts.FindIndexOf( font ) == 0 )
	//	return;

	mCurFontID = inPixFontID;

	if ( mWorld ) {
		__setupPort

#if EG_MAC
		::TextFont( font -> mOSFontID );
		::TextSize( font -> mPointSize );
		::TextFace( font -> mOSStyle );
#elif EG_WIN
		::SelectObject( mWorld, (HFONT) font -> mOSFontID );
#elif EG_X
		mfl_SetFont( (mfl_context) mWorld, (mfl_font) font->mOSFontID );
#endif
		__restorePort
	}
}

void PixPort::SetTextMode( PixDrawMode inMode ) {

	if ( ! mWorld )
		return;

	__setupPort

#if EG_MAC
	long mode = srcCopy;
	if ( inMode == SRC_OR )
		mode = srcOr;
	else if ( inMode == SRC_BIC )
		mode = srcBic;
	else if ( inMode == SRC_XOR )
		mode = srcXor;
	::TextMode( mode );
#endif

#if EG_WIN
	long mode = R2_COPYPEN;
	if ( inMode == SRC_BIC )
		mode = R2_WHITE;
	else if ( inMode == SRC_XOR )
		mode = R2_NOT;
	::SetROP2( mWorld, mode );
#endif

#ifdef EG_X
	int mfl_mode;

	if (inMode == SRC_OR) { mfl_mode = MFL_OR; }
	else if (inMode == SRC_XOR) { mfl_mode = MFL_XOR; }
	else { mfl_mode = MFL_NORMAL; }

	mfl_SetDrawMode((mfl_context) mWorld, mfl_mode);
#endif

	__restorePort
}

void PixPort::SetPaletteColor( long inEntryNum ) {

	if ( ! mWorld || mDepth != 8 )
		return;

	__setupPort

	if ( inEntryNum < 0 )
		inEntryNum = 0;
	if ( inEntryNum > 0xFF )
		inEntryNum = 0xFF;

#if EG_MAC
#if TARGET_API_MAC_CARBON
	CTabHandle myTable = (**mBM).pmTable;
	::RGBForeColor( &(**myTable).ctTable[ inEntryNum ].rgb );
#else
	// The above 2 lines *used* to work, but in MacOS 9.1, MacOS intercepts palette entries 0 and 255--blarg!
	mWorld -> fgColor = inEntryNum;
#endif
#endif // EG_MAC

#if EG_WIN
	RGBQUAD color;
	::GetDIBColorTable( mWorld, inEntryNum, 1, &color );
	::SetTextColor( mWorld, RGB( color.rgbRed, color.rgbGreen, color.rgbBlue ) );
#endif
	__restorePort
}

void PixPort::SetTextColor( RGBColor& inColor ) {

	if ( ! mWorld )
		return;

	__setupPort

#if EG_MAC
	::RGBForeColor( &inColor );
#endif

#if EG_WIN
	::SetTextColor( mWorld, RGB( inColor.red >> 8, inColor.green >> 8, inColor.blue >> 8 ) );
#endif

#if EG_X
	mfl_SetTextColor((mfl_context) mWorld, 255);
#endif

	__restorePort
}

void PixPort::SetTextColor( PixPalEntry& inColor ) {

	if ( ! mWorld )
		return;

#if EG_MAC
	SetTextColor( inColor.rgb );
#endif

#if EG_WIN
	::SetTextColor( mWorld, RGB( inColor.rgbRed, inColor.rgbGreen, inColor.rgbBlue ) );
#endif
}

void PixPort::TextRect( const char* inStr, long& outWidth, long& outHeight ) {

	long width, pos, lineHeight = 0;
	char c;

	outWidth  = 0;
	outHeight = 0;

	__setupPort

	if ( mCurFontID )
		lineHeight = ( (PixTextStyle*) mCurFontID ) -> mLineHeight;

	while ( *inStr ) {
		c = inStr[ 0 ];
		pos = 0;

		while ( c != '\r' && c != 0 ) {
			pos++;
			c = inStr[ pos ];
		}

#if EG_MAC
		width = ::TextWidth( inStr, 0, pos );
#endif

#if EG_WIN
		SIZE dim;
		::GetTextExtentPoint( mWorld, inStr, pos, &dim );
		width  = dim.cx;
#endif

#if EG_X
		width = mfl_GetTextWidthL((mfl_context) mWorld, inStr, pos);
#endif

		if ( width > outWidth )
			outWidth = width;

		outHeight += lineHeight;

		if ( c == 0 )
			break;

		inStr += pos + 1;

		if ( c == '\r' && inStr[ 0 ] == '\n' )
			inStr++;
		else if ( c == '\n' && inStr[ 0 ] == '\r' )
			inStr++;
	}

	if ( outHeight > 0 && mCurFontID )
		outHeight += ( (PixTextStyle*) mCurFontID ) -> mDescent;

	__restorePort

}

void PixPort::DrawText( long inX, long inY, const char* inStr ) {

	long pos, lineHeight = 0;
	char c;

	if ( ! mWorld )
		return;

	if ( mCurFontID ) {
		inY += ( (PixTextStyle*) mCurFontID ) -> mAscent;
		lineHeight = ( (PixTextStyle*) mCurFontID ) -> mLineHeight;
		//inY += lineHeight;
	}

	__setupPort

	while ( *inStr ) {
		c = inStr[ 0 ];
		pos = 0;

		while ( c != '\r' && c != '\n' && c ) {
			pos++;
			c = inStr[ pos ];
		}

#if EG_MAC
		::MoveTo( inX, inY );
		::DrawText( inStr, 0, pos );
#elif EG_WIN
		::TextOut( mWorld, inX, inY, inStr, pos );
#elif EG_X
		mfl_OutTextL((mfl_context) mWorld, inX, inY, inStr, pos);
#endif
		inY += lineHeight;

		if ( c == 0 )
			break;

		inStr += pos + 1;

		if ( c == '\r' && inStr[ 0 ] == '\n' )
			inStr++;
		else if ( c == '\n' && inStr[ 0 ] == '\r' )
			inStr++;
	}

	__restorePort
}

void PixPort::SetLineWidth( long inLineWidth ) {

	if ( inLineWidth <= 0 )
		mLineWidth = 1;
	else if ( inLineWidth > MAX_LINE_WIDTH )
		mLineWidth = MAX_LINE_WIDTH;
	else
		mLineWidth = inLineWidth;
}

void PixPort::EraseRect( const Rect* inRect ) {

	if ( mDepth == 16 )
		EraseRect16( inRect );
	else if ( mDepth == 8 )
		EraseRect8 ( inRect );
	else if ( mDepth == 32 )
		EraseRect32( inRect );
}


#define P_SZ	1
#include "DrawXX.cpp"


#undef P_SZ
#define P_SZ	2
#include "DrawXX.cpp"


#undef P_SZ
#define P_SZ	4
#include "DrawXX.cpp"


#if WIN_QUICKTIME_PRESENT || EG_MAC

GWorldPtr PixPort::GetGWorld() {

	#if EG_MAC
	return mWorld;
	#endif

	#if WIN_QUICKTIME_PRESENT
	if ( ! mQTML_World ) {
		long err = ::NewGWorldFromHBITMAP( &mQTML_World, nil, nil, 0, mBM, mWorld );
		if ( err != noErr )
			::NewGWorldFromHBITMAP( &mQTML_World, nil, nil, 0, nil, nil );
	}

	return mQTML_World;
	#endif

}

#endif
