
#include "VideoExporter.h"

#if EG_MAC
#include <FixMath.h>
#endif

#include "PixPort.h"
#include "EgOSUtils.h"

VideoExporter::VideoExporter() :
	VideoDevice( true ) {

	mExporting = false;

}

VideoExporter::~VideoExporter() {

	EndExport();
}

#if EG_MAC

// DEFINES
#define	BailNil(n)		if (!n) goto bail;
#define	BailError(n)	if (n) goto bail;


// FUNCTION PROTOTYPES
pascal short
DefaultsHookProc(DialogPtr theDialog, short itemHit, void *params, long refcon);

void VideoExporter::StartExport( PixPort& inSample ) {

	OSErr		result;
	Point		where;

	//SCDataRateSettings ds;

	EndExport();

	mMovieRefNum = 0;
	mMovie = 0;

	//---------------------------------------------------------------------------------------------
	//	Open the Standard Compression component and adjust it to meet our needs.
	//---------------------------------------------------------------------------------------------

	mComponent = OpenDefaultComponent( StandardCompressionType, StandardCompressionSubType );
	BailNil( mComponent );

	//	Turn off best depth option in the compression dialog.
	//	Because all of our buffering is done at 32-bits, regardless
	//	of the depth of the source data, best depth isn't very useful
	//	as it will always choose 32-bit.


	//	Because we are recompressing a movie that may have a variable frame rate,
	//	we want to allow the user to leave the frame rate text field blank.  We
	//	can then preserve the frame durations of the source movie.  If they enter
	//	a number we will resample the movie at a new frame rate.  If we don't clear
	//	this flag, the compression dialog will not allow zero in the frame rate field.

	{
		long flags;

		SCGetInfo( mComponent,scPreferenceFlagsType,&flags);
		flags &= ~scShowBestDepth;

		//flags |= scAllowZeroFrameRate;

		SCSetInfo(mComponent,scPreferenceFlagsType,&flags);
	}



	// Make a preview image for the SC setting dialog...
	{



		//	Use the gworld image of the movie poster frame as a compression dialog test image.
		//	Pass nil for srcRect to use the entire image, and pass zero to use the default display method.

		result = SCSetTestImagePixMap( mComponent, inSample.GetPixMap(), nil , 0 );
		BailError(result);

	}


	//---------------------------------------------------------------------------------------------
	//	Set compression dialog extend procs for the "Defaults" button.
	//	We wait until now so we can pass the srcGWorld pixmap in as a refcon.
	//---------------------------------------------------------------------------------------------

	{
		SCExtendedProcs xprocs;

		//	Because the compression dialog is movable modal, a filter proc
		//	to handle update events for application windows is required.
		//	Since our example application is completely modal, and we have
		//	no other windows to update, we don't have a filter proc.
		xprocs.filterProc = nil;

		//	Proc to handle custom button click.
		xprocs.hookProc = NewSCModalHookUPP(DefaultsHookProc);

		//	Any information useful to the extended procs can be put
		//	in the refcon.  For an application that has globals, a refcon
		//	of the current A5 is handy for getting at those globals in
		//	the extended procs.  In our case, we put the srcGWorld pixmap
		//	in the refcon so we can set defaults on it in our hook proc.
		xprocs.refcon = (long) ( inSample.GetPixMap() );

		//	Set the custom button name.
		BlockMove("\pDefaults",xprocs.customName,9);

		//	Tell the compression dialog about the extended procs.
		SCSetInfo( mComponent,scExtendedProcsType,&xprocs);
	}


	//---------------------------------------------------------------------------------------------
	//	Set up some default settings for the compression dialog if needed, and ask
	//	for compression settings from the user.
	//---------------------------------------------------------------------------------------------

	result = SCDefaultPixMapSettings( mComponent, inSample.GetPixMap(), true );
	BailError(result);


	//	Get compression settings from the user.  The first time through this loop,
	//	we choose default compression settings for the test image we set above.
	//	On subsequent passes, the settings previously chosen by the user will be the defaults.

	result = SCRequestSequenceSettings( mComponent );
	if (result == scUserCancelled) {
		// deal with user cancelling.
	}
	BailError(result);

	//	Get a copy of the temporal settings the user entered.  We'll need them for
	//	some of our calculations.  In a simpler application we'd never have to look at them.

	result = SCGetInfo( mComponent,scTemporalSettingsType,&mTimeSettings);
	BailError(result);




	//---------------------------------------------------------------------------------------------
	//	Ask the user for the name of the new movie file.
	//	We'll be lazy here and just use "Untitled".  A real app would
	//	base it on the name of the source movie and check if the user
	//	tried to enter a name the same as the source movie.
	//
	//	Note we use the SCPositionDialog call to get a good position
	//	point for the SFPutFile.  -3999 is the resource I.D. for
	//	SFPutFile dialog.
	//---------------------------------------------------------------------------------------------






	//---------------------------------------------------------------------------------------------
	//	Create the new movie file and prepare it for edits.
	//---------------------------------------------------------------------------------------------

	{
		MatrixRecord	matrix;

		where.h = where.v = -2;
		SCPositionDialog( mComponent,-3999,&where);
		CEgFileSpec	fileSpec;

		if ( ! EgOSUtils::AskSaveAs( "Save movie as:", "Untitled", fileSpec, 'MooV' ) )
			goto bail;


		//	Using the FSSpec create a movie file for the destination movie.
		result = CreateMovieFile( (FSSpec*) fileSpec.OSSpec(),'TVOD',0,createMovieFileDeleteCurFile /*| createMovieFileDontCreateResFile*/, &mMovieRefNum, &mMovie );
		BailError(result);

		//	Create a new video movie track with the same dimensions as the entire source movie.
		mTrack = NewMovieTrack( mMovie,
				(long)( inSample.GetX() ) << 16,
				(long)( inSample.GetY() ) << 16, 0 );

		//	Create a media for the new track with the same time scale
		mMedia = NewTrackMedia( mTrack,VIDEO_TYPE,1000,0,0);
		BailError(GetMoviesError());

		//	Set movie matrix to identity and clear the movie clip region
		//	because conversion process transforms and composites all video
		//	tracks into one untransformed video track.
		SetIdentityMatrix(&matrix);
		SetMovieMatrix( mMovie,&matrix);
		SetMovieClipRgn( mMovie,nil);

		//	Prepare for adding frames to the movie.
		//	Make sure you do this.  It's very easy to forget about.
		result = BeginMediaEdits( mMedia );
		BailError(result);

		mNeedToStartSeq = true;
		mExporting = true;

	}

bail:;

}

void VideoExporter::TransferBits( PixPort& inFrame, Rect& , Rect& ) {
	OSErr		result;
//	Point		where;
//	SFReply		dstFile;

	if ( ! Exporting() )
		return;

	if ( mNeedToStartSeq ) {

		//---------------------------------------------------------------------------------------------
		//	Start a compression sequence using the parameters chosen by the user.
		//	Pass nil for the source rect to use the entire image.  An image description
		//	handle will be returned in idh.  Nil could be passed for that if we
		//	didn't need it for our progress window display.  We do not have dispose
		//	the image description handle.  It is disposed for use by SCCompressSequenceEnd.
		//---------------------------------------------------------------------------------------------


		result = SCCompressSequenceBegin( mComponent, inFrame.GetPixMap(),nil,&mImageDesc );
		BailError(result);

		mNeedToStartSeq = false;

	}
	// Greater than frame 1
	else {

		short			syncFlag;
		TimeValue		duration;
		long			dataSize;
		Handle			compressedData;

		duration = mT - mPrevFrameT;

		//	If data rate constraining is being done, tell Standard Compression the
		//	duration of the current frame in milliseconds.  We only need to do this
		//	if the frames have variable durations.
		// 	Without this, the compressor can't limit the data rate.
		{
			SCDataRateSettings ds;
			if (!SCGetInfo( mComponent,scDataRateSettingsType,&ds)) {
				ds.frameDuration = duration;
				SCSetInfo( mComponent,scDataRateSettingsType,&ds);
			}
		}


		// Tell user we're not frozen and update last recorded time index
		EgOSUtils::SpinCursor();


		//	Compress the frame.  compressedData will hold a handle to the newly compressed
		//	image data.  dataSize is the size of the compressed data, which will usually be
		//	different than the size of the compressedData handle.  syncFlag is a value that
		//	can be passed directly to AddMediaSample which indicates whether or not the frame
		//	is a key frame.  Note that we do not have to dispose of the compressedData handle.
		//	It will be dispose for us when we call SCCompressSequenceEnd.
		result = SCCompressSequenceFrame( mComponent, inFrame.GetPixMap(),nil,&compressedData,&dataSize,&syncFlag);
		BailError(result);

		//	Append the compressed image data to the media.

		result = AddMediaSample( mMedia,compressedData,0,dataSize,duration,
				(SampleDescriptionHandle) mImageDesc,1,syncFlag,nil);
		BailError(result);
	}

bail:
	EgOSUtils::ResetCursor();
}

void VideoExporter::EndExport() {

	OSErr result;

	if ( ! Exporting() )
		return;

	mExporting = false;

	//	Close the compression sequence.  This will dispose of the image description
	//	and compressed data handles allocated by SCCompressSequenceBegin.

	SCCompressSequenceEnd( mComponent );

	//---------------------------------------------------------------------------------------------
	//	Now that we're finished compressing video data, make that data part of our movie.
	//---------------------------------------------------------------------------------------------

	if ( mTrack ) {
		short resID = 128;

		//	End changes to the media.

		result = EndMediaEdits( mMedia);
		BailError(result);

		//	Insert the newly created media into the newly created track at
		//	the begining of the track and lasting for the entire duration of
		//	the media.  The media rate is 1.0 for normal playback rate.

		InsertMediaIntoTrack( mTrack,0,0,GetMediaDuration(mMedia),fixed1);

		//	Add the movie resource to the dst movie file.

		result = AddMovieResource( mMovie,mMovieRefNum,&resID,"\pExported Movie");
		//result = AddMovieResource(dstMovie,movieRefNum, (short*) movieInDataForkResID,"\pWhiteCap Movie");
		BailError(result);


		//---------------------------------------------------------------------------------------------
		//	Next we flatten the movie.  The movie file we just created has all of
		//	the video data at the beginning and all of the sound data at the end.
		//	This unnatural situation may cause the movie to play poorly.  By flattening
		//	the movie, we are reinterleaving all of the tracks in a way optimal for playback.
		//	Because we cannot flatten a movie in place, we create a temp file, flatten to
		//	that file and if successful, delete the first file and rename the temp file to
		//	be the real file.  For large files this takes some time.
		//
		//	Once again, a real application would choose a temp file name that doesn't
		//	possibly conflict with other file names, as well as delete the temp file
		//	if the flatten fails.
		//---------------------------------------------------------------------------------------------
		{
			FSSpec	theFSSpec;
			short	resID = 128;
			mDestFile.fName[++ mDestFile.fName[0]] = '@';
			result = FSMakeFSSpec( mDestFile.vRefNum,0, mDestFile.fName,&theFSSpec);
			if (result != fnfErr)
				BailError(result);
			FlattenMovie( mMovie,0,&theFSSpec,'TVOD',-1,createMovieFileDeleteCurFile,&resID,"\p");
			result = GetMoviesError();
			CloseMovieFile( mMovieRefNum );
			DisposeMovie( mMovie);
			mTrack = nil;
			if (!result) {
				FSSpec spec;
				Str255 tempStr;


				::BlockMove( mDestFile.fName, tempStr, 255 );
				tempStr[0]--;
				FSMakeFSSpec( mDestFile.vRefNum, theFSSpec.parID, tempStr, &spec );

				result = ::FSpDelete( &spec );
				result = ::FSpRename( &theFSSpec, tempStr );
			}
		}
	}

	//	Just to be overly safe, we clear our the test image
	//	because we just disposed the pixmap it depends on.
	SCSetTestImagePixMap( mComponent,nil,nil,0);


	//	Close the Standard Compression component.
	CloseComponent( mComponent );
bail:
	EgOSUtils::ShowCursor();
}


//	Handle clicks in the "Defaults" custom button.

pascal short
DefaultsHookProc(DialogPtr theDialog, short itemHit, void *params, long refcon)
{
	#pragma unused(theDialog)
	if (itemHit == scCustomItem) {
		SCTemporalSettings ts;

		//	Set defaults for our test image (passed in refcon).

		SCDefaultPixMapSettings( (ComponentInstanceRecord*) params,(PixMapHandle)refcon, true );

		//	Once again, don't use the default frame rate chosen
		//	by Standard Compression.  Clear it out to zero.

		SCGetInfo( (ComponentInstanceRecord*) params,scTemporalSettingsType,&ts);
		ts.frameRate = 0;
		SCSetInfo( (ComponentInstanceRecord*) params,scTemporalSettingsType,&ts);
	}
	return (itemHit);
}

#else

void VideoExporter::StartExport( PixPort& ) {}
void VideoExporter::EndExport() {}
void VideoExporter::TransferBits( PixPort&, Rect&, Rect& ) {}

#endif
