
#ifndef __VIDEO_DEVICE_H__
#define __VIDEO_DEVICE_H__

#if EG_MAC
#include <Types.h>
#endif

#include "Eg Common.h"
#include "PixPort.h"

/* This class is a virtual video device that accepts frame buffers and delivers
them to the virtual device.  VideoDevice also maintains an averaged frame rate.  Examples of
sublcases of this class VideoExporter, VideoWindow, and VideoMirror.  */

class VideoDevice {

public:
	VideoDevice( bool inForceFullRedraws );

	void			Init();

	// Exports the given frame
	// <inDirtyRect> describes the region in <inFrame> that's different compared to the frame passed to the
	// previous call to ExportFrame().  If it's NULL, the entire frame is assumed to be different.
	// <inT_MS> is an absolute time index in millisections that the given frame lasts to.
	void			AcceptFrame( PixPort& inFrame, const Rect* inDirtyRect, long inT_MS );

	// Returns the minimum time index when a frame will be accepted.  This fcn is only useful if
	// a mimium frame druation is set (ie, a max framerate is set).
	long			NextFrameTime()								{ return mT + mMinDuration;				}

	// Sets the minibum duration of all frames (ie, sets a max framerate).
	void			SetMinFrameDur( long inDurMS )				{ mMinDuration = inDurMS;				}
	long			MinFrameDur()								{ return mMinDuration;					}

	// Returns the current frame rate (rounded)
	float			FPS()										{ return ( (float) mAvgFrameRate ) / 10.0 + 0.5;	}
	bool 			FPS_Steady()								{ return mFPS_Steady;								}

	// Returns a number from 0 to 100, describing what priority level the app should be in the OS (0 means highest priority, 100 means lowest)
	// The recommended priority is a function of the desired FPS (set via SetDesiredFPS()) and the current FPS.
	// It's up to the caller to figure out how to make more calls to AcceptFrame()
	long			GetPriority()								{ return mCurPriority;								}

	// Affects the value returned in GetPriority() based on current frame rate.
	void			SetDesiredFPS( long inFPS )					{ mDesiredFPS = inFPS;								}

	void			InitFPS()									{ mFrameCountStart = 0xDEADBEEF;					}

	// Attempts to return the depth of this device.
	// 0 is returned if unsuccessful
	virtual int		GetDepth()									{ return 0;								}

	// Resize the frame
	virtual void		Resize( Rect& inRect );

	void			GetOutputRect( Rect& outRect )				{ outRect = mOutputRect;				}

	virtual void		Erase( bool inUpdateNow );

	void			SetOrigin( int inX, int inY )				{ mOrigin.h = inX; mOrigin.v = inY;		}

	// Callbacks the device may need to know.  They should surround the graphics routines, etc.
	void			BeginFrame();
	void			EndFrame();

protected:
	// The pane/local rect that video is allowed to appear in.
	// The top left of given frames will be matched to the top left of the output rect + mOrigin
	Rect			mOutputRect;
	Point			mOrigin;		// Rel offset from the topLeft of mOutput where the top left of the video appears

	// Callbacks the device may need to know
	void			PreFrame( PixPort* inSrceFrame );

	// Specialized fcn to copy bits to the physical destination
	virtual void		TransferBits( PixPort& inFrame, Rect& inSrceRect, Rect& inDestRect ) = 0;

	long			mPrevFrameT;
	long			mT;

	bool			mForceFullRedraws;

	virtual void		BeginFrame_Self()										{ }
	virtual void		PreFrame_Self( PixPort* )								{ }
	virtual void		EndFrame_Self()											{ }

	bool			mBeginFrameCalled;
	bool			mPreFrameCalled;

private:
	PP_GDHandle		mPrevDev;
	PP_GrafPtr		mPrevPort;

	long			mMinDuration;
	bool			mNeedsRefresh;

	// Frame count related
	long			mAvgFrameRate;
	long			mFrameCountStart;
	long			mFrameCount;
	bool			mFPS_Steady;

	// Priority related...
	long			mCurPriority;
	long			mDesiredFPS;

	void			FPS_Notify();
};

#endif // __VIDEO_DEVICE_H__
