/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit/macros.h"
#include "singit/displayer_plugin.h"
#include "singit/control.h"

#include "displayer_sdl_whitecap.h"
#include "sdl_whitecap_config.h"
#include "sdl_whitecap_status.h"
#include "sdl_whitecap_config_dlg.h"
#include "sdl_whitecap_handlers.h"

#include "sdl_lyrics.h"
#include "sdl_helpers.h"

#include "WhiteCap_Proj.h"
#include "WhiteCap.h"
#include "EgOSUtils.h"
#include "RectUtils.h"
#include "CEgFileSpec.h"
#include "SDL_prim.h"
#include "SDL_VideoDevice.h"

#define SND_BUF_SIZE 550
#define NUMSAMPLES 200
#define SAMPSKIP 1

extern DisplayerPlugin sdl_whitecap_dp;
extern SingitConfigGen *displayer_sdl_config;

static SPluginData		gSound;
static SDL_VideoDevice*		gVideoOutput;
static PluginGlue*		gPlugin = nil;
// static long			gLastRecTime = -1;
static gint16                   plugin_freq_data[2][256];
static gint16                   plugin_pcm_data[2][512];

static gboolean CreateScreen(gint nWidth, gint nHeight, gboolean bFullscreen)
{
	static gint save_full[2] = { -1 };
	gint nFlags, w, h;

	if (bFullscreen == TRUE) {
		nFlags = SDL_HWSURFACE | SDL_FULLSCREEN;
		save_full[0] = STATUS->screen->w;
		save_full[1] = STATUS->screen->h;
		w = 640;
		h = 480;
	}
	else {
		nFlags = SDL_SWSURFACE | SDL_RESIZABLE;
		if ((save_full[0] != -1) && (save_full[1] != -1)) {
			w = save_full[0];
			h = save_full[1];
			save_full[0] = -1;
		}
		else {
			w = nWidth;
			h = nHeight;
		}
	}

	STATUS->screen = SDL_SetVideoMode(w, h, bFullscreen ? 16 : 0, nFlags);

	if (STATUS->screen == NULL)
		{ return FALSE; }

	STATUS->is_fullscreen = bFullscreen;

	if (bFullscreen == TRUE) {
		sdl_helper_blanking(BLANK_DISABLE);
		SDL_WM_GrabInput(SDL_GRAB_ON);
		SDL_ShowCursor(0);
	}
	else {
		sdl_helper_blanking(BLANK_ENABLE);
		SDL_WM_GrabInput(SDL_GRAB_OFF);
		SDL_ShowCursor(1);
	}

	return (STATUS->screen != NULL);
}

extern "C" gboolean sdl_handler_init(void)
{
	gchar *title;

#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_init]\n"));
#endif

	// Initialise the SDL library
	if (SDL_WasInit(SDL_INIT_VIDEO) > 0) {
		g_print("SDL Video was initialized before : no two SDL windows allowed : exit\n");
		STATUS->sdl_inuse = TRUE;
		return FALSE;
	}
	else {
		if (SDL_Init(SDL_INIT_VIDEO) != 0) {
			g_print("Unable to init SDL Video: %s\n", SDL_GetError());
			return FALSE;
		}
	}
	STATUS->song = NULL;

	if (CreateScreen(480, 300, FALSE) == FALSE) {
		g_print("Graphic mode is not available: %s\n", SDL_GetError());
		gtk_idle_add (displayer_sdl_whitecap_disable_func, NULL);
		return FALSE;
	}

	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("SDL displayer"), NULL);
	SDL_WM_SetCaption(title, PACKAGE_NAME);
	g_free(title);
//	SDL_WM_SetCaption(SINGIT_VERSION_STRING "-" "SDL displayer");

	TTF_Init();
	STATUS->font = TTF_OpenFont(SINGIT_DATA_DIR "/vixar.ttf", 32);
//	TTF_SetFontStyle(STATUS->font, TTF_STYLE_BOLD);

	SDL_Color colors[256];

	for (int pp = 0; pp < 256 ; pp++) {
		colors[pp].r = pp / 4;
		colors[pp].g = pp / 4;
		colors[pp].b = pp / 4;
	}

	EgOSUtils::Initialize( nil );

	gPlugin = _NEW_PLUGIN( 0, nil );
	gVideoOutput = new SDL_VideoDevice( true );
	gVideoOutput -> SetOutput( STATUS->screen );
	gPlugin -> SetVideoOutput( gVideoOutput, false );

	gSound.mFFT_NumBins = 0;
	gSound.mSpectrum_L = &plugin_freq_data[0][0];
	gSound.mSpectrum_R = &plugin_freq_data[1][0];
	gSound.mSample_NumBins = 0;
	gSound.mWaveform_L = &plugin_pcm_data[0][0];
	gSound.mWaveform_R = &plugin_pcm_data[1][0];

	return TRUE;
}

extern "C" void sdl_handler_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_finish]\n"));
#endif

	if ( gPlugin ) {
		gPlugin -> PrepareToExit();
		delete gPlugin;
		gPlugin = NULL;
	}

	if (gVideoOutput) {
		delete gVideoOutput;
		gVideoOutput = NULL;
	}

      	EgOSUtils::Shutdown();

	if (STATUS->font) {
		TTF_CloseFont(STATUS->font);
		STATUS->font = NULL;
	}

	// Close SDL (prevents the ticks to continue running)
	TTF_Quit();
	SDL_Quit();
}

void draw_border_rect(SDL_Surface *surface, SDL_Rect *rect, guint border, Uint32 color)
{
	guint x2 = rect->x + rect->w;
	guint y2 = rect->y + rect->h;

	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, x2 + border, rect->y - border - 1, color);
	SDL_drawLine(surface, rect->x - border - 1, rect->y - border - 1, rect->x - border - 1, y2 + border, color);
	SDL_drawLine(surface, rect->x - border - 1, y2 + border, x2 + border, y2 + border, color);
	SDL_drawLine(surface, x2 + border, rect->y - border, x2 + border, y2 + border, color);
}

extern "C" void displayer_sdl_whitecap_draw_text()
{
	sdl_lyrics_draw_solid(STATUS->screen);
}

extern "C" void sdl_handler_key_down(void *pEvent)
{
	SDL_KeyboardEvent* pKeyEvent = (SDL_KeyboardEvent*)pEvent;
	gint key = pKeyEvent->keysym.sym;
	gint mod = pKeyEvent->keysym.mod;
	gint volume, value;
	gboolean draw;

#ifdef CODEDEBUG
	DEBUG(("sdl_handlers.cpp [sdl_handler_key_down]\n"));
#endif

	switch (key) {
	case SDLK_ESCAPE:
	case SDLK_q:
		if (STATUS->is_fullscreen) {
			draw = CreateScreen(640, 480, !STATUS->is_fullscreen);
		}
		gtk_idle_add (displayer_sdl_whitecap_disable_func, NULL);
		return;
	case SDLK_a:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_about_dlg
				(sdl_whitecap_dp.singit_session);
			GDK_THREADS_LEAVE();
			return;
		}
		break;
	case SDLK_b:
		xmms_remote_playlist_next(sdl_whitecap_dp.xmms_session);
		return;
	case SDLK_c:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			displayer_sdl_whitecap_config_show();
			GDK_THREADS_LEAVE();
			return;
		}
		else { xmms_remote_pause(sdl_whitecap_dp.xmms_session); }
		break;
	case SDLK_e:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_editor_dlg
				(sdl_whitecap_dp.singit_session);
			GDK_THREADS_LEAVE();
			return;
		}
		break;
	case SDLK_i:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_tag_manager_dlg
				(sdl_whitecap_dp.singit_session);
			GDK_THREADS_LEAVE();
			return;
		}
		break;
	case SDLK_p:
		if ((mod & KMOD_CTRL) && !STATUS->is_fullscreen) {
			GDK_THREADS_ENTER();
			singit_remote_show_prefs_dlg
				(sdl_whitecap_dp.singit_session);
			GDK_THREADS_LEAVE();
			return;
		}
		break;
	case SDLK_v:
		xmms_remote_stop(sdl_whitecap_dp.xmms_session);
		return;
	case SDLK_x:
		xmms_remote_play(sdl_whitecap_dp.xmms_session);
		return;
	case SDLK_y:
		if (STATUS->toggle_yz) {
			xmms_remote_playlist_prev(sdl_whitecap_dp.xmms_session);
			return;
		}
		break;
	case SDLK_z:
		if (!STATUS->toggle_yz) {
			xmms_remote_playlist_prev(sdl_whitecap_dp.xmms_session);
			return;
		}
		break;
	case SDLK_KP_PLUS:
		volume = xmms_remote_get_main_volume(sdl_whitecap_dp.xmms_session);
		if (volume < 100) { volume++; }
		xmms_remote_set_main_volume(sdl_whitecap_dp.xmms_session, volume);
		return;
	case SDLK_KP_MINUS:
		volume = xmms_remote_get_main_volume(sdl_whitecap_dp.xmms_session);
		if (volume > 0) { volume--; }
		xmms_remote_set_main_volume(sdl_whitecap_dp.xmms_session, volume);
		return;

	case SDLK_RETURN:
	case SDLK_KP_ENTER:
		if (mod & KMOD_ALT) {
			draw = CreateScreen(640, 480, !STATUS->is_fullscreen);
			if ((draw == TRUE) && ( gPlugin )) {
				gVideoOutput -> SetOutput( STATUS->screen );
				gPlugin -> SetVideoOutput( gVideoOutput, false );
				sdl_lyrics_resize();
			}
			return;
		}
		if ((STATUS->jump_to_song == 0) ||
			(key == SDLK_RETURN)) { return; }
		if (xmms_remote_get_playlist_pos(sdl_whitecap_dp.xmms_session) !=
			(STATUS->jump_to_song - 1))
		{
			xmms_remote_set_playlist_pos
				(sdl_whitecap_dp.xmms_session,
				(STATUS->jump_to_song - 1));
		}
	case SDLK_KP_PERIOD:
		STATUS->jump_to_song = 0;
		return;
	case SDLK_KP0:
	case SDLK_KP1:
	case SDLK_KP2:
	case SDLK_KP3:
	case SDLK_KP4:
	case SDLK_KP5:
	case SDLK_KP6:
	case SDLK_KP7:
	case SDLK_KP8:
	case SDLK_KP9:
	case SDLK_0:
	case SDLK_1:
	case SDLK_2:
	case SDLK_3:
	case SDLK_4:
	case SDLK_5:
	case SDLK_6:
	case SDLK_7:
	case SDLK_8:
	case SDLK_9:
		value = key - SDLK_0;
		if (value < 0 || value > 9)
			value = key - SDLK_KP0;

		if (mod & KMOD_CTRL) {
			GDK_THREADS_ENTER();
			enable_plugin(((value + 9) % 10));
			GDK_THREADS_LEAVE();
		}
		else {
			STATUS->jump_to_song = (STATUS->jump_to_song * 10) + value;
			if (STATUS->jump_to_song > xmms_remote_get_playlist_length(sdl_whitecap_dp.xmms_session))
				{ STATUS->jump_to_song = value; }
		}
		return;
	default:
		break;
        }

	if (gPlugin != NULL) {
		gPlugin->ExecuteKey(key, mod & KMOD_SHIFT, mod & KMOD_CTRL);
	}
}

extern "C" void sdl_handler_resize(void *pEvent)
{
	gboolean draw = TRUE;
	SDL_ResizeEvent* pResizeEvent = (SDL_ResizeEvent*) pEvent;

	if (!STATUS->is_fullscreen)
		{ draw = CreateScreen(pResizeEvent->w, pResizeEvent->h, false); }

	if (draw == TRUE) {
		if ( gPlugin ) {
			gVideoOutput -> SetOutput( STATUS->screen );
			gPlugin -> SetVideoOutput( gVideoOutput, false );
		}
		sdl_lyrics_resize();
	}
}

extern "C" void displayer_sdl_whitecap_render()
{
	if (gPlugin)
		gPlugin -> Draw( &gSound );
}

// Waveform
extern "C" void displayer_sdl_whitecap_render_pcm(gint16 pcm_data[2][512], SingitSoundPrecalcs *ssp)
{
	memcpy(&plugin_pcm_data, pcm_data, sizeof(gint16) * 2 * 512);
	gSound.mSample_NumBins = 512;
}

// Spectrum
extern "C" void displayer_sdl_whitecap_render_freq(gint16 freq_data[2][256], SingitSoundPrecalcs *ssp)
{
	memcpy(&plugin_freq_data, freq_data, sizeof(gint16) * 2 * 256);
	gSound.mFFT_NumBins = 256;
}

extern "C" void displayer_sdl_whitecap_new_song_signal(gchar *title, gchar* artist)
{
/*
	if ((title != NULL) || (artist != NULL)) {
		if (title != NULL) {
			gGF -> mSongTitle.Wipe();
			gGF -> mSongTitle.Assign( title );
		}
		if (artist != NULL) {
			gGF -> mArtist.Wipe();
			gGF -> mArtist.Assign( artist );
		}

		// Send the signal
//		gGF -> NewSong();
	}
*/
}

extern "C" void sdl_handler_active(void *pEvent)
{
	if (STATUS->is_fullscreen == TRUE)
		{ sdl_helper_gain_focus(); }
}

extern "C" SDL_Color getPaletteColor(const gint index)
{
	SDL_Color result = { 0, 0, 0, 0 };

	if (index > 200)
		result.r = 255;

//	if (gGF) { result = gGF->GetPaletteColor(index); }
	return result;
}

extern "C" void displayer_sdl_whitecap_set_active_song_line(gchar *line)
{
//	if (gGF != NULL)
//		gGF -> SetActiveSongLine( line );
}
